(function () {

'use strict';

angular.module('OpenSlidesApp.agenda', ['OpenSlidesApp.users'])

.factory('Speaker', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'agenda/speaker',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

.factory('AgendaUpdate',[
    'Agenda',
    'operator',
    function(Agenda, operator) {
        return {
            saveChanges: function (item_id, changes) {
                // change agenda item only if user has the permission to do that
                if (operator.hasPerms('agenda.can_manage agenda.can_see_hidden_items')) {
                    Agenda.find(item_id).then(function (item) {
                        var something = false;
                        _.each(changes, function(change) {
                            if (change.value !== item[change.key]) {
                                item[change.key] = change.value;
                                something = true;
                            }
                        });
                        if (something === true) {
                            Agenda.save(item);
                        }
                    });
                }
            }
        };
    }
])

.factory('Agenda', [
    '$http',
    'DS',
    'Speaker',
    'jsDataModel',
    'Projector',
    'gettextCatalog',
    'gettext',
    'CamelCase',
    'EditForm',
    function($http, DS, Speaker, jsDataModel, Projector, gettextCatalog, gettext, CamelCase, EditForm) {
        var name = 'agenda/item';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Agenda'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getContentObject: function () {
                    return DS.get(this.content_object.collection, this.content_object.id);
                },
                getContentObjectDetailState: function () {
                    return CamelCase(this.content_object.collection).replace('/', '.') +
                        '.detail({id: ' + this.content_object.id + '})';
                },
                getContentObjectForm: function () {
                    return EditForm.fromCollectionString(this.content_object.collection);
                },
                getContentResource: function () {
                    return DS.definitions[this.content_object.collection];
                },
                getTitle: function () {
                    var title;
                    try {
                        title =  this.getContentObject().getAgendaTitle();
                    } catch (e) {
                        // when the content object is not in the DS store.
                        title = this.title;
                    }
                    if (this.item_number) {
                        title = this.item_number + ' · ' + title;
                    }
                    return title;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
                getCSVExportText: function () {
                    var text;
                    try {
                        text =  this.getContentObject().getCSVExportText();
                    } catch (e) {
                        // when the content object is not in the DS store
                        // or 'getCSVExportText' is not defined return nothing.
                    }
                    return text;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getAgendaTitle();
                },
                // return true if a specific relation matches for given searchquery
                // (here: speakers)
                hasSearchResult: function (results) {
                    var item = this;
                    // search for speakers (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(item.speakers, {'user_id': result.id})) {
                                return true;
                            }
                        }
                    });
                },
                getListViewTitle: function () {
                    var title;
                    try {
                        title =  this.getContentObject().getAgendaListViewTitle();
                    } catch (e) {
                        // when the content object is not in the DS store
                        title = this.list_view_title;
                    }
                    if (this.item_number) {
                        title = this.item_number + ' · ' + title;
                    }
                    return title;
                },
                getItemNumberWithAncestors: function (agendaId) {
                    if (!agendaId) {
                        agendaId = this.id;
                    }
                    var agendaItem = DS.get(name, agendaId);
                    if (!agendaItem) {
                        return '';
                    } else if (agendaItem.item_number) {
                        return agendaItem.item_number;
                    } else if (agendaItem.parent_id) {
                        return this.getItemNumberWithAncestors(agendaItem.parent_id);
                    } else {
                        return '';
                    }
                },
                // override project function of jsDataModel factory
                project: function (projectorId, tree) {
                    var isProjectedIds = this.isProjected(tree);
                    _.forEach(isProjectedIds, function (id) {
                        $http.post('/rest/core/projector/' + id + '/clear_elements/');
                    });
                    // Activate, if the projector_id is a new projector.
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        var name = tree ? 'agenda/item-list' : this.content_object.collection;
                        var id = tree ? this.id : this.content_object.id;
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/prune_elements/',
                            [{name: name, tree: tree, id: id}]
                        );
                    }
                },
                // override isProjected function of jsDataModel factory
                isProjected: function (tree) {
                    // Returns all ids of all projectors with an agenda-item element. Otherwise an empty list.
                    if (typeof tree === 'undefined') {
                        tree = false;
                    }
                    var self = this;
                    var predicate = function (element) {
                        var value;
                        if (tree) {
                            // Item tree slide for sub tree
                            value = element.name == 'agenda/item-list' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id;
                        } else {
                            // Releated item detail slide
                            value = element.name == self.content_object.collection &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.content_object.id;
                        }
                        return value;
                    };
                    var isProjectedIds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjectedIds.push(projector.id);
                        }
                    });
                    return isProjectedIds;
                },
                // project list of speakers
                projectListOfSpeakers: function(projectorId) {
                    var isProjectedIds = this.isListOfSpeakersProjected();
                    _.forEach(isProjectedIds, function (id) {
                        $http.post('/rest/core/projector/' + id + '/clear_elements/');
                    });
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/prune_elements/',
                            [{name: 'agenda/list-of-speakers', id: this.id}]
                        );
                    }
                },
                // check if list of speakers is projected
                isListOfSpeakersProjected: function () {
                    // Returns all ids of all projectors with an element with the
                    // name 'agenda/list-of-speakers' and the same id. Else returns an empty list.
                    var self = this;
                    var predicate = function (element) {
                        return element.name == 'agenda/list-of-speakers' &&
                               typeof element.id !== 'undefined' &&
                               element.id == self.id;
                    };
                    var isProjecteds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjecteds.push(projector.id);
                        }
                    });
                    return isProjecteds;
                },
                hasSubitems: function(items) {
                    var self = this;
                    var hasChild = false;
                    // Returns true if the item has at least one child item.
                    _.each(items, function (item) {
                        if (item.parent_id == self.id) {
                            hasChild = true;
                        }
                    });
                    return hasChild;
                }
            },
            relations: {
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'agenda/speaker': {
                        localField: 'speakers',
                        foreignKey: 'item_id',
                    }
                }
            },
            beforeInject: function (resource, instance) {
                Speaker.ejectAll({where: {item_id: {'==': instance.id}}});
            }
        });
    }
])

.factory('AgendaTree', [
    function () {
        return {
            getTree: function (items) {
                // Sort items after there weight
                items.sort(function(itemA, itemB) {
                    return itemA.weight - itemB.weight;
                });

                // Build a dict with all children (dict-value) to a specific
                // item id (dict-key).
                var itemChildren = {};

                _.each(items, function (item) {
                    if (item.parent_id) {
                        // Add item to his parent. If it is the first child, then
                        // create a new list.
                        try {
                            itemChildren[item.parent_id].push(item);
                        } catch (error) {
                            itemChildren[item.parent_id] = [item];
                        }
                    }

                });

                // Recursive function that generates a nested list with all
                // items with there children
                function getChildren(items) {
                    var returnItems = [];
                    _.each(items, function (item) {
                        returnItems.push({
                            item: item,
                            children: getChildren(itemChildren[item.id]),
                            id: item.id,
                        });
                    });
                    return returnItems;
                }

                // Generates the list of root items (with no parents)
                var parentItems = items.filter(function (item) {
                    return !item.parent_id;
                });
                return getChildren(parentItems);
            },

            // Returns a list of all items as a flat tree
            getFlatTree: function(items) {
                var tree = this.getTree(items);
                var flatItems = [];

                function generateFlatTree(tree, parentCount) {
                    _.each(tree, function (item) {
                        item.item.parentCount = parentCount;
                        flatItems.push(item.item);
                        generateFlatTree(item.children, parentCount + 1);
                    });
                }
                generateFlatTree(tree, 0);
                return flatItems;
            }
        };
    }
])

.factory('CurrentListOfSpeakersItem', [
    'Projector',
    'Agenda',
    function (Projector, Agenda) {
        return {
            getItem: function (projectorId) {
                var projector = Projector.get(projectorId), item;
                if (projector) {
                    _.forEach(projector.elements, function(element) {
                        if (element.agenda_item_id) {
                            item = Agenda.get(element.agenda_item_id);
                        }
                    });
                }
                return item;
            }
        };
    }
])

.factory('CurrentListOfSpeakersSlide', [
    '$http',
    'Projector',
    function($http, Projector) {
        var name = 'agenda/current-list-of-speakers';
        return {
            project: function (projectorId, overlay) {
                var isProjected = this.isProjectedWithOverlayStatus();
                _.forEach(isProjected, function (mapping) {
                    $http.post('/rest/core/projector/' + mapping.projectorId + '/deactivate_elements/',
                        [mapping.uuid]
                    );
                });

                // The slide was projected, if the id matches. If the overlay is given, also
                // the overlay is checked
                var wasProjectedBefore = _.some(isProjected, function (mapping) {
                    var value = (mapping.projectorId === projectorId);
                    if (overlay !== undefined) {
                        value = value && (mapping.overlay === overlay);
                    }
                    return value;
                });
                overlay = overlay || false; // set overlay if it wasn't defined

                if (!wasProjectedBefore) {
                    var activate = function () {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                             [{name: name,
                               stable: overlay, // if this is an overlay, it should not be
                                                // removed by changing the main content
                               overlay: overlay}]
                        );
                    };
                    if (!overlay) {
                        // clear all elements on this projector, because we are _not_ using the overlay.
                        $http.post('/rest/core/projector/' + projectorId + '/clear_elements/').then(activate);
                    } else {
                        activate();
                    }
                }
            },
            isProjected: function () {
                // Returns the ids of all projectors with an agenda-item element. Else return an empty list.
                var predicate = function (element) {
                    return element.name === name;
                };
                var isProjectedIds = [];
                Projector.getAll().forEach(function (projector) {
                    if (typeof _.findKey(projector.elements, predicate) === 'string') {
                        isProjectedIds.push(projector.id);
                    }
                });
                return isProjectedIds;
            },
            // Returns a list of mappings between pojector id, overlay and uuid.
            isProjectedWithOverlayStatus: function () {
                var mapping = [];
                _.forEach(Projector.getAll(), function (projector) {
                    _.forEach(projector.elements, function (element, uuid) {
                        if (element.name === name) {
                            mapping.push({
                                projectorId: projector.id,
                                uuid: uuid,
                                overlay: element.overlay || false,
                            });
                        }
                    });
                });
                return mapping;
            },
        };
    }
])



// Make sure that the Agenda resource is loaded.
.run(['Agenda', function(Agenda) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.csv', [])

.factory('AgendaCsvExport', [
    'HumanTimeConverter',
    'gettextCatalog',
    'CsvDownload',
    function (HumanTimeConverter, gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Text', 'Duration', 'Comment', 'Internal item'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (agenda) {
                var csvRows = [
                    makeHeaderline()
                ];
                _.forEach(agenda, function (item) {
                    var row = [];
                    var duration = item.duration ? HumanTimeConverter.secondsToHumanTime(item.duration*60,
                            { seconds: 'disabled',
                                hours: 'enabled' }) : '';
                    row.push('"' + (item.title || '') + '"');
                    row.push('"' + (item.getCSVExportText() || '') + '"');
                    row.push('"' + duration + '"');
                    row.push('"' + (item.comment || '') + '"');
                    row.push('"' + (item.is_hidden ? '1' : '')  + '"');
                    csvRows.push(row);
                });
                CsvDownload(csvRows, 'agenda-export.csv');
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.pdf', ['OpenSlidesApp.core.pdf'])

.factory('AgendaContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    function(gettextCatalog, PDFLayout) {

    var createInstance = function(items) {

        // page title
        var title = PDFLayout.createTitle(gettextCatalog.getString("Agenda"));

        // generate the item list with all subitems
        var createItemList = function() {
            var agenda_items = [];
            angular.forEach(items, function (item) {
                if (item.is_hidden === false) {

                    var itemIndent = item.parentCount * 20;

                    var itemStyle;
                    if (item.parentCount === 0) {
                        itemStyle = 'listParent';
                    } else {
                        itemStyle = 'listChild';
                    }

                    var itemNumberWidth;
                    if (item.item_number === "") {
                        itemNumberWidth = 0;
                    } else {
                        itemNumberWidth = 60;
                    }

                    var agendaJsonString = {
                        style: itemStyle,
                        columns: [
                            {
                                width: itemIndent,
                                text: ''
                            },
                            {
                                width: itemNumberWidth,
                                text: item.item_number
                            },
                            {
                                text: item.title
                            }
                        ]
                    };

                    agenda_items.push(agendaJsonString);
                }
            });
            return agenda_items;
        };

        var getContent = function() {
            return [
                title,
                createItemList()
            ];
        };

        return {
            getContent: getContent
        };
    };

    return {
        createInstance: createInstance
    };

}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.projector', ['OpenSlidesApp.agenda'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('agenda/list-of-speakers', {
            template: 'static/templates/agenda/slide-list-of-speakers.html',
        });
        slidesProvider.registerSlide('agenda/item-list', {
            template: 'static/templates/agenda/slide-item-list.html',
        });
        slidesProvider.registerSlide('agenda/current-list-of-speakers', {
            template: 'static/templates/agenda/slide-current-list-of-speakers.html',
        });
    }
])

.controller('SlideCurrentListOfSpeakersCtrl', [
    '$scope',
    'Agenda',
    'CurrentListOfSpeakersItem',
    'Config',
    'Projector',
    function ($scope, Agenda, CurrentListOfSpeakersItem, Config, Projector) {
        $scope.overlay = $scope.element.overlay;
        // Watch for changes in the current list of speakers reference
        $scope.$watch(function () {
            return Config.lastModified('projector_currentListOfSpeakers_reference');
        }, function () {
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
            $scope.updateCurrentListOfSpeakers();
        });
        // Watch for changes in the referenced projector
        $scope.$watch(function () {
            return Projector.lastModified($scope.currentListOfSpeakersReference);
        }, function () {
            $scope.updateCurrentListOfSpeakers();
        });
        // Watch for changes in the current item.
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            $scope.updateCurrentListOfSpeakers();
        });
        $scope.updateCurrentListOfSpeakers = function () {
            $scope.agendaItem = CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
        };
    }
])

.controller('SlideListOfSpeakersCtrl', [
    '$scope',
    'Agenda',
    'User',
    function ($scope, Agenda, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        Agenda.bindOne(id, $scope, 'item');
    }
])

.controller('SlideItemListCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Agenda',
    'AgendaTree',
    function ($scope, $http, $filter, Agenda, AgendaTree) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.

        // Bind agenda tree to the scope
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            if ($scope.element.id) {
                $scope.rootItem = Agenda.get($scope.element.id);
                var tree = AgendaTree.getFlatTree(Agenda.getAll());
                var startIndex = tree.indexOf($scope.rootItem);
                tree = tree.slice(startIndex);
                // define delta to move the whole subtree to level 0
                var parentCountDelta = 0;
                if (tree[0]) {
                    parentCountDelta = tree[0].parentCount;
                }
                $scope.items = [];
                for (var i = 1; i < tree.length; i++) {
                    if (tree[i].parentCount - parentCountDelta <= 0) {
                        break;
                    }
                    var item = tree[i];
                    // move rootItem (and all childs) to level 0
                    item.parentCount = item.parentCount - parentCountDelta;
                    $scope.items.push(item);
                }
            } else if ($scope.element.tree) {
                $scope.items = AgendaTree.getFlatTree(Agenda.getAll());
            } else {
                $scope.items = Agenda.filter({
                    where: { parent_id: null },
                    orderBy: 'weight'
                });
            }
        });
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.site', [
    'OpenSlidesApp.agenda',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.agenda.pdf',
    'OpenSlidesApp.agenda.csv',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'agenda.item.list',
            'img_class': 'calendar-o',
            'title': gettext('Agenda'),
            'weight': 200,
            'perm': 'agenda.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Agenda'),
            'collectionName': 'agenda/item',
            'urlDetailState': 'agenda.item.detail',
            'weight': 200,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('agenda', {
                url: '/agenda',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Agenda'),
                    basePerm: 'agenda.can_see',
                },
            })
            .state('agenda.item', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('agenda.item.list', {})
            .state('agenda.item.detail', {
                resolve: {
                    itemId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            .state('agenda.item.sort', {
                url: '/sort',
                controller: 'AgendaSortCtrl',
            })
            .state('agenda.current-list-of-speakers', {
                url: '/speakers',
                controller: 'CurrentListOfSpeakersViewCtrl',
                data: {
                    title: gettext('Current list of speakers'),
                },
            });
    }
])

.controller('ItemListCtrl', [
    '$scope',
    '$filter',
    '$http',
    '$state',
    'DS',
    'operator',
    'ngDialog',
    'Agenda',
    'TopicForm', // TODO: Remove this dependency. Use template hook for "New" and "Import" buttons.
    'AgendaTree',
    'Projector',
    'ProjectionDefault',
    'AgendaContentProvider',
    'PdfMakeDocumentProvider',
    'gettextCatalog',
    'gettext',
    'osTableFilter',
    'AgendaCsvExport',
    'PdfCreate',
    'ErrorMessage',
    function($scope, $filter, $http, $state, DS, operator, ngDialog, Agenda, TopicForm,
        AgendaTree, Projector, ProjectionDefault, AgendaContentProvider, PdfMakeDocumentProvider,
        gettextCatalog, gettext, osTableFilter, AgendaCsvExport, PdfCreate, ErrorMessage) {
        // Bind agenda tree to the scope
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            // Filter out items that doesn't have the list_item_title. This happens, if the
            // item is a hidden item but provides the list of speakers, but should not be
            // visible in the list view.
            var allowedItems = _.filter(Agenda.getAll(), function (item) {
                return item.list_view_title;
            });
            $scope.items = AgendaTree.getFlatTree(allowedItems);
            var subitems = $filter('filter')($scope.items, {'parent_id': ''});
            if (subitems.length) {
                $scope.agendaHasSubitems = true;
            }
        });
        Projector.bindAll({}, $scope, 'projectors');
        $scope.mainListTree = true;
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault =  ProjectionDefault.filter({name: 'agenda_all_items'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId_all_items = projectiondefault.projector_id;
            }
            $scope.projectionDefaults = ProjectionDefault.getAll();
        });
        $scope.alert = {};


        // Filtering
        $scope.filter = osTableFilter.createInstance('AgendaTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.booleanFilters = {
                closed: {
                    value: undefined,
                    displayName: gettext('Closed items'),
                    choiceYes: gettext('Closed items'),
                    choiceNo: gettext('Open items'),
                },
                is_hidden: {
                    value: undefined,
                    displayName: gettext('Internal items'),
                    choiceYes: gettext('Internal items'),
                    choiceNo: gettext('No internal items'),
                    permission: 'agenda.can_see_hidden_items',
                },
            };
        }
        $scope.filter.propertyList = ['item_number', 'title', 'title_list_view', 'comment', 'duration'];
        $scope.filter.propertyFunctionList = [
            function (item) {return item.getListViewTitle();},
        ];
        $scope.filter.propertyDict = {
            'speakers' : function (speaker) {
                return '';
            },
        };

        // pagination
        $scope.currentPage = 1;
        $scope.itemsPerPage = 25;
        $scope.limitBegin = 0;
        $scope.pageChanged = function() {
            $scope.limitBegin = ($scope.currentPage - 1) * $scope.itemsPerPage;
        };

        // parse duration for inline editing
        $scope.generateDurationText = function (item) {
            //convert data from model format (m) to view format (hh:mm)
            if (item.duration) {
                var time = "",
                    totalminutes = item.duration;
                if (totalminutes < 0) {
                    time = "-";
                    totalminutes = -totalminutes;
                }
                var hh = Math.floor(totalminutes / 60);
                var mm = Math.floor(totalminutes % 60);
                // Add leading "0" for double digit values
                mm = ("0"+mm).slice(-2);
                time += hh + ":" + mm;
                item.durationText = time;
            } else {
                item.durationText = "";
            }
        };
        $scope.setDurationText = function (item) {
            //convert data from view format (hh:mm) to model format (m)
            var time = item.durationText.replace('h', '').split(':');
            var data;
            if (time.length > 1 && !isNaN(time[0]) && !isNaN(time[1])) {
                data = (+time[0]) * 60 + (+time[1]);
                if (data < 0) {
                    data = "-"+data;
                }
                item.duration = parseInt(data);
            } else if (time.length == 1 && !isNaN(time[0])) {
                data = (+time[0]);
                item.duration = parseInt(data);
            } else {
                item.duration = 0;
            }
            $scope.save(item);
        };

        /** Duration calculations **/
        $scope.sumDurations = function () {
            var totalDuration = 0;
            $scope.items.forEach(function (item) {
                if (item.duration) {
                    totalDuration += item.duration;
                }
            });
            return totalDuration;
        };
        $scope.calculateEndTime = function () {
            var totalDuration = $scope.sumDurations();
            var startTimestamp = $scope.config('agenda_start_event_date_time');
            if (startTimestamp) {
                var endTimestamp = startTimestamp + totalDuration * 60 * 1000;
                var endDate = new Date(endTimestamp);
                var mm = ("0" + endDate.getMinutes()).slice(-2);
                var dateStr = endDate.getHours() + ':' + mm;
                return dateStr;
            } else {
                return '';
            }
        };

        /** Agenda item functions **/
        // open dialog for new topics // TODO Remove this. Don't forget import button in template.
        $scope.newDialog = function () {
            ngDialog.open(TopicForm.getDialog());
        };
        // save changed item
        $scope.save = function (item) {
            Agenda.save(item).then(
                function (success) {
                    $scope.alert.show = false;
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                });
        };
        // delete related item
        $scope.deleteRelatedItem = function (item) {
            DS.destroy(item.content_object.collection, item.content_object.id);
        };
        // auto numbering of agenda items
        $scope.autoNumbering = function() {
            $http.post('/rest/agenda/item/numbering/', {});
        };
        // check open permission
        // TODO: Use generic solution here.
        $scope.isAllowedToSeeOpenLink = function (item) {
            var collection = item.content_object.collection;
            switch (collection) {
                case 'topics/topic':
                    return operator.hasPerms('agenda.can_see');
                case 'motions/motion':
                    return operator.hasPerms('motions.can_see');
                case 'motions/motion-block':
                    return operator.hasPerms('motions.can_see');
                case 'assignments/assignment':
                    return operator.hasPerms('assignments.can_see');
                default:
                    return false;
            }
        };
        $scope.edit = function (item) {
            ngDialog.open(item.getContentObjectForm().getDialog({id: item.content_object.id}));
        };
        // export
        $scope.pdfExport = function () {
            var filename = gettextCatalog.getString('Agenda') + '.pdf';
            var agendaContentProvider = AgendaContentProvider.createInstance($scope.itemsFiltered);
            var documentProvider = PdfMakeDocumentProvider.createInstance(agendaContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };
        $scope.csvExport = function () {
            AgendaCsvExport.export($scope.itemsFiltered);
        };

        /** select mode functions **/
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            angular.forEach($scope.items, function (item) {
                item.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isDeleteMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.items, function (item) {
                    item.selected = false;
                });
            }
        };
        // delete selected items
        $scope.deleteMultiple = function () {
            angular.forEach($scope.items, function (item) {
                if (item.selected) {
                    DS.destroy(item.content_object.collection, item.content_object.id);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };

        /** Project functions **/
        // get ProjectionDefault for item
        $scope.getProjectionDefault = function (item) {
            if (item.tree) {
                return $scope.defaultProjectorId_all_items;
            } else {
                var app_name = item.content_object.collection.split('/')[0];
                var id = 1;
                $scope.projectionDefaults.forEach(function (projectionDefault) {
                    if (projectionDefault.name == app_name) {
                        id = projectionDefault.projector_id;
                    }
                });
                return id;
            }
        };
        // project agenda
        $scope.projectAgenda = function (projectorId, tree, id) {
            var isAgendaProjectedIds = $scope.isAgendaProjected($scope.mainListTree);
            _.forEach(isAgendaProjectedIds, function (id) {
                $http.post('/rest/core/projector/' + id + '/clear_elements/');
            });
            if (_.indexOf(isAgendaProjectedIds, projectorId) == -1) {
                $http.post('/rest/core/projector/' + projectorId + '/prune_elements/',
                    [{name: 'agenda/item-list', tree: tree, id: id}]);
            }
        };
        // change whether all items or only main items should be projected
        $scope.changeMainListTree = function () {
            var isAgendaProjectedId = $scope.isAgendaProjected($scope.mainListTree);
            $scope.mainListTree = !$scope.mainListTree;
            if (isAgendaProjectedId > 0) {
                $scope.projectAgenda(isAgendaProjectedId, $scope.mainListTree);
            }
        };
        // change whether one item or all subitems should be projected
        $scope.changeItemTree = function (item) {
            var isProjected = item.isProjected(item.tree);
            if (isProjected > 0) {
                // Deactivate and reactivate
                item.project(isProjected, item.tree);
                item.project(isProjected, !item.tree);
            }
            item.tree = !item.tree;
        };
        // check if agenda is projected
        $scope.isAgendaProjected = function (tree) {
            // Returns the ids of all projectors with an element with
            // the name 'agenda/item-list'. Else returns an empty list.
            var predicate = function (element) {
                var value;
                if (tree) {
                    // tree with all agenda items
                    value = element.name == 'agenda/item-list' &&
                        typeof element.id === 'undefined' &&
                        element.tree;
                } else {
                    // only main agenda items
                    value = element.name == 'agenda/item-list' &&
                        typeof element.id === 'undefined' &&
                        !element.tree;
                }
                return value;
            };
            var projectorIds = [];
            $scope.projectors.forEach(function (projector) {
                if (typeof _.findKey(projector.elements, predicate) === 'string') {
                    projectorIds.push(projector.id);
                }
            });
            return projectorIds;
        };
    }
])

.controller('ItemDetailCtrl', [
    '$scope',
    '$filter',
    'Agenda',
    'itemId',
    'Projector',
    'ProjectionDefault',
    'ErrorMessage',
    function ($scope, $filter, Agenda, itemId, Projector, ProjectionDefault, ErrorMessage) {
        $scope.alert = {};

        $scope.$watch(function () {
            return Agenda.lastModified(itemId);
        }, function () {
            $scope.item = Agenda.get(itemId);
            // all speakers
            $scope.speakers = $filter('orderBy')($scope.item.speakers, 'weight');
            // next speakers
            $scope.nextSpeakers = $filter('filter')($scope.speakers, {'begin_time': null});
            // current speaker
            $scope.currentSpeaker = $filter('filter')($scope.speakers, {'begin_time': '!!', 'end_time': null});
            // last speakers
            $scope.lastSpeakers = $filter('filter')($scope.speakers, {'end_time': '!!'});
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var item_app_name = $scope.item.content_object.collection.split('/')[0];
            var projectiondefaultItem = ProjectionDefault.filter({name: item_app_name})[0];
            if (projectiondefaultItem) {
                $scope.defaultProjectorItemId = projectiondefaultItem.projector_id;
            }
            var projectiondefaultListOfSpeakers = ProjectionDefault.filter({name: 'agenda_list_of_speakers'})[0];
            if (projectiondefaultListOfSpeakers) {
                $scope.defaultProjectorListOfSpeakersId = projectiondefaultListOfSpeakers.projector_id;
            }
        });
    }
])

/* This is the controller for the list of speakers partial management template.
 * The parent controller needs to provide a $scope.item, $scope.speakers, $scope.nextSpeakers,
 * $scope.currentSpeakers, $scope.lastSpeakers. See (as example) ItemDetailCtrl. */
.controller('ListOfSpeakersManagementCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Agenda',
    'User',
    'operator',
    'ErrorMessage',
    function ($scope, $http, $filter, Agenda, User, operator, ErrorMessage) {
        User.bindAll({}, $scope, 'users');
        $scope.speakerSelectBox = {};

        // close/open list of speakers of current item
        $scope.closeList = function (listClosed) {
            $scope.item.speaker_list_closed = listClosed;
            Agenda.save($scope.item);
        };

        // add user to list of speakers
        $scope.addSpeaker = function (userId) {
            $http.post('/rest/agenda/item/' + $scope.item.id + '/manage_speaker/', {'user': userId}).then(
                function (success) {
                    $scope.alert.show = false;
                    $scope.speakers = $scope.item.speakers;
                    $scope.speakerSelectBox = {};
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                    $scope.speakerSelectBox = {};
                }
            );
        };

        // delete speaker(!) from list of speakers
        $scope.removeSpeaker = function (speakerId) {
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/manage_speaker/',
                {headers: {'Content-Type': 'application/json'},
                 data: JSON.stringify({speaker: speakerId})}
            )
            .then(function (success) {
                $scope.speakers = $scope.item.speakers;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
            $scope.speakers = $scope.item.speakers;
        };

        //delete all speakers from list of speakers
        $scope.removeAllSpeakers = function () {
            var speakersOnList = [];
            angular.forEach($scope.item.speakers, function (speaker) {
                speakersOnList.push(speaker.id);
            });
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/manage_speaker/',
                {headers: {'Content-Type': 'application/json'},
                 data: JSON.stringify({speaker: speakersOnList})}
            )
            .then(function (success) {
                $scope.speakers = $scope.item.speakers;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
            $scope.speakers = $scope.item.speakers;
        };

        // Return true if the requested user is allowed to do a specific action
        // and see the corresponding button (e.g. 'add me' or 'remove me').
        $scope.isAllowed = function (action) {
            var nextUsers = [];
            angular.forEach($scope.nextSpeakers, function (speaker) {
                nextUsers.push(speaker.user_id);
            });
            switch (action) {
                case 'add':
                    return (operator.hasPerms('agenda.can_be_speaker') &&
                            !$scope.item.speaker_list_closed &&
                            $.inArray(operator.user.id, nextUsers) == -1);
                case 'remove':
                    if (operator.user) {
                        return ($.inArray(operator.user.id, nextUsers) != -1);
                    }
                    return false;
                case 'removeAll':
                    return (operator.hasPerms('agenda.can_manage') &&
                            $scope.speakers.length > 0);
                case 'beginNextSpeech':
                    return (operator.hasPerms('agenda.can_manage') &&
                            $scope.nextSpeakers.length > 0);
                case 'endCurrentSpeech':
                    return (operator.hasPerms('agenda.can_manage') &&
                            $scope.currentSpeaker.length > 0);
                case 'showLastSpeakers':
                    return $scope.lastSpeakers.length > 0;
            }
        };

        // begin speech of selected/next speaker
        $scope.beginSpeech = function (speakerId) {
            $http.put('/rest/agenda/item/' + $scope.item.id + '/speak/', {'speaker': speakerId})
            .then(function (success) {
                $scope.alert.show = false;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // end speech of current speaker
        $scope.endSpeech = function () {
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/speak/',
                {headers: {'Content-Type': 'application/json'}, data: {}}
            ).then(
                function (success) {},
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // gets speech duration of selected speaker in seconds
        $scope.getDuration = function (speaker) {
            var beginTimestamp = new Date(speaker.begin_time).getTime();
            var endTimestamp = new Date(speaker.end_time).getTime();
            // calculate duration in seconds
            return Math.floor((endTimestamp - beginTimestamp) / 1000);

        };
        // save reordered list of speakers
        $scope.treeOptions = {
            dropped: function (event) {
                var sortedSpeakers = [];
                var nextSpeakers = $filter('filter')($scope.speakers, {'begin_time': null});
                angular.forEach(nextSpeakers, function (speaker) {
                    sortedSpeakers.push(speaker.id);
                });
                $http.post('/rest/agenda/item/' + $scope.item.id + '/sort_speakers/',
                    {speakers: sortedSpeakers}
                );
            }
        };
    }
])

.controller('AgendaSortCtrl', [
    '$scope',
    '$http',
    'Agenda',
    'AgendaTree',
    'ErrorMessage',
    function($scope, $http, Agenda, AgendaTree, ErrorMessage) {
        // Bind agenda tree to the scope
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            $scope.items = AgendaTree.getTree(Agenda.getAll());
        });
        $scope.showInternalItems = true;
        $scope.alert = {};

        // save parent and weight of moved agenda item (and all items on same level)
        $scope.treeOptions = {
            dropped: function(event) {
                var parentID = null;
                var droppedItemID = event.source.nodeScope.$modelValue.id;
                if (event.dest.nodesScope.item) {
                    parentID = event.dest.nodesScope.item.id;
                }
                $http.post('/rest/agenda/item/sort/', {
                    nodes: event.dest.nodesScope.$modelValue,
                    parent_id: parentID}
                ).then(
                    function(success) {},
                    function(error){
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.controller('CurrentListOfSpeakersViewCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Projector',
    'ProjectionDefault',
    'Agenda',
    'Config',
    'CurrentListOfSpeakersItem',
    'CurrentListOfSpeakersSlide',
    function($scope, $http, $filter, Projector, ProjectionDefault, Agenda, Config,
        CurrentListOfSpeakersItem, CurrentListOfSpeakersSlide) {
        $scope.alert = {};
        $scope.currentListOfSpeakers = CurrentListOfSpeakersSlide;

        // Watch for changes in the current list of speakers reference
        $scope.$watch(function () {
            return Config.lastModified('projector_currentListOfSpeakers_reference');
        }, function () {
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
            $scope.updateCurrentListOfSpeakersItem();
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function() {
            $scope.projectors = Projector.getAll();
            // If there is just one projector we provide just the overlay.
            if ($scope.projectors.length === 1) {
                $scope.currentListOfSpeakersAsOverlay = true;
            }
            $scope.updateCurrentListOfSpeakersItem();

            $scope.listOfSpeakersDefaultProjectorId = ProjectionDefault.filter({name: 'agenda_current_list_of_speakers'})[0].projector_id;
        });

        $scope.$watch(function () {
            return $scope.item ? Agenda.lastModified($scope.item.id) : void 0;
        }, function () {
            $scope.updateCurrentListOfSpeakersItem();
        });

        $scope.updateCurrentListOfSpeakersItem = function () {
            $scope.item = CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
            if ($scope.item) {
                // all speakers
                $scope.speakers = $filter('orderBy')($scope.item.speakers, 'weight');
                // next speakers
                $scope.nextSpeakers = $filter('filter')($scope.speakers, {'begin_time': null});
                // current speaker
                $scope.currentSpeaker = $filter('filter')($scope.speakers, {'begin_time': '!!', 'end_time': null});
                // last speakers
                $scope.lastSpeakers = $filter('filter')($scope.speakers, {'end_time': '!!'});
            } else {
                $scope.speakers = void 0;
                $scope.nextSpeakers = void 0;
                $scope.currentSpeaker = void 0;
                $scope.lastSpeakers = void 0;
            }
        };

        // Set the current overlay status
        if ($scope.currentListOfSpeakers.isProjected().length) {
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            $scope.currentListOfSpeakersAsOverlay = isProjected[0].overlay;
        } else {
            $scope.currentListOfSpeakersAsOverlay = true;
        }
        $scope.setOverlay = function (overlay) {
            $scope.currentListOfSpeakersAsOverlay = overlay;
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.overlay != overlay) { // change the overlay if it is different
                        $scope.currentListOfSpeakers.project(mapping.projectorId, overlay);
                    }
                });
            }
        };
    }
])

//mark all agenda config strings for translation with Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Numbering prefix for agenda items');
        gettext('This prefix will be set if you run the automatic agenda numbering.');
        gettext('Agenda');
        gettext('Invalid input.');
        gettext('Numeral system for agenda items');
        gettext('Arabic');
        gettext('Roman');
        gettext('Begin of event');
        gettext('Input format: DD.MM.YYYY HH:MM');
        gettext('Number of last speakers to be shown on the projector');
        gettext('List of speakers');
        gettext('Show orange countdown in the last x seconds of speaking time');
        gettext('Enter duration in seconds. Choose 0 to disable warning color.');
        gettext('Couple countdown with the list of speakers');
        gettext('[Begin speech] starts the countdown, [End speech] stops the ' +
                'countdown.');
    }
 ]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments', [])

.factory('AssignmentPollOption', [
    'DS',
    'jsDataModel',
    'gettextCatalog',
    'Config',
    'MajorityMethods',
    function (DS, jsDataModel, gettextCatalog, Config, MajorityMethods) {
        return DS.defineResource({
            name: 'assignments/polloption',
            useClass: jsDataModel,
            methods: {
                getVotes: function () {
                    if (!this.poll.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values for the option
                    var votes = [],
                        config = Config.get('assignments_poll_100_percent_base').value;

                    var base = this.poll.getPercentBase(config);
                    if (typeof base === 'object' && base !== null) {
                        // this.poll.pollmethod === 'yna'
                        base = base[this.id];
                    }

                    _.forEach(this.votes, function (vote) {
                        // Initial values for the vote
                        var order = '',
                            value = '',
                            percentStr = '',
                            percentNumber;

                        // Check for special value
                        switch (vote.weight) {
                            case -1:
                                value = gettextCatalog.getString('majority');
                                break;
                            case -2:
                                value = gettextCatalog.getString('undocumented');
                                break;
                            default:
                                if (vote.weight >= 0) {
                                    value = vote.weight;
                                } else {
                                    value = 0;  // Vote was not defined. Set value to 0.
                                }
                        }
                        switch (vote.value) {
                            case "Yes":
                                order = 1;
                                break;
                            case "No":
                                order = 2;
                                break;
                            case "Abstain":
                                order = 3;
                                break;
                            default:
                                order = 0;
                        }

                        // Special case where to skip percents
                        var skipPercents = config === 'YES_NO' && vote.value === 'Abstain';

                        if (base && !skipPercents) {
                            percentNumber = Math.round(vote.weight * 100 / base * 10) / 10;
                            percentStr = "(" + percentNumber + "%)";
                        }
                        votes.push({
                            'order': order,
                            'label': gettextCatalog.getString(vote.value),
                            'value': value,
                            'percentStr': percentStr,
                            'percentNumber': percentNumber
                        });
                    });
                    return _.sortBy(votes, 'order');
                },

                // Returns 0 or positive integer if quorum is reached or surpassed.
                // Returns negativ integer if quorum is not reached.
                // Returns undefined if we can not calculate the quorum.
                isReached: function (method) {
                    if (!this.poll.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }
                    var isReached;
                    var config = Config.get('assignments_poll_100_percent_base').value;
                    var base = this.poll.getPercentBase(config);
                    if (typeof base === 'object' && base !== null) {
                        // this.poll.pollmethod === 'yna'
                        base = base[this.id];
                    }
                    if (base) {
                        // Provide result only if base is not undefined and not 0.
                        isReached = MajorityMethods[method](this.getVoteYes(), base);
                    }
                    return isReached;
                },

                // Returns the weight for the vote or the vote 'yes' in case of YNA poll method.
                getVoteYes: function () {
                    var voteYes = 0;
                    if (this.poll.pollmethod === 'yna') {
                        var voteObj = _.find(this.votes, function (vote) {
                            return vote.value === 'Yes';
                        });
                        if (voteObj) {
                            voteYes = voteObj.weight;
                        }
                    } else {
                        // pollmethod === 'votes'
                        voteYes = this.votes[0].weight;
                    }
                    return voteYes;
                }
            },
            relations: {
                belongsTo: {
                    'assignments/poll': {
                        localField: 'poll',
                        localKey: 'poll_id',
                    },
                    'users/user': {
                        localField: 'candidate',
                        localKey: 'candidate_id',
                    }
                }
            },
        });
    }
])

.factory('AssignmentPoll', [
    '$http',
    'DS',
    'jsDataModel',
    'gettextCatalog',
    'AssignmentPollOption',
    'Config',
    function ($http, DS, jsDataModel, gettextCatalog, AssignmentPollOption, Config) {
        var name = 'assignments/poll';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            methods: {
                getResourceName: function () {
                    return name;
                },

                // Returns percent base. Returns undefined if calculation is not possible in general.
                getPercentBase: function (config, type) {
                    var base;
                    switch (config) {
                        case 'CAST':
                            if (this.votescast <= 0 || this.votesinvalid < 0) {
                                // It would be OK to check only this.votescast < 0 because 0
                                // is checked again later but this is a little bit faster.
                                break;
                            }
                            base = this.votescast;
                            /* falls through */
                        case 'VALID':
                            if (this.votesvalid < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid') {
                                base = this.votesvalid;
                            }
                            /* falls through */
                        case 'YES_NO_ABSTAIN':
                        case 'YES_NO':
                            if (this.pollmethod === 'yna') {
                                if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                    base = {};
                                    _.forEach(this.options, function (option) {
                                        var allVotes = option.votes;
                                        if (config === 'YES_NO') {
                                            allVotes = _.filter(allVotes, function (vote) {
                                                // Extract abstain votes in case of YES_NO percent base.
                                                // Do not extract abstain vote if it is set to majority so the predicate later
                                                // fails and therefor we get an undefined base. Reason: It should not be possible
                                                // to set abstain to majority and nevertheless calculate percents of yes and no.
                                                return vote.value !== 'Abstain' || vote.weight === -1;
                                            });
                                        }
                                        var predicate = function (vote) {
                                            return vote.weight < 0;
                                        };
                                        if (_.findIndex(allVotes, predicate) === -1) {
                                            base[option.id] = _.reduce(allVotes, function (sum, vote) {
                                                return sum + vote.weight;
                                            }, 0);
                                        }
                                    });
                                }
                            } else {
                                // this.pollmethod === 'votes'
                                var predicate = function (option) {
                                    return option.votes[0].weight < 0;
                                };
                                if (_.findIndex(this.options, predicate) !== -1) {
                                    base = void 0;
                                } else {
                                    if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                        base = _.reduce(this.options, function (sum, option) {
                                            return sum + option.votes[0].weight;
                                        }, 0);
                                    }
                                }
                            }
                    }
                    return base;
                },

                // Returns object with value and percent for this poll (for votes valid/invalid/cast only).
                getVote: function (type) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values
                    var value = '',
                        percentStr = '',
                        percentNumber,
                        vote,
                        config = Config.get('assignments_poll_100_percent_base').value;

                    switch (type) {
                        case 'votesinvalid':
                            vote = this.votesinvalid;
                            break;
                        case 'votesvalid':
                            vote = this.votesvalid;
                            break;
                        case 'votescast':
                            vote = this.votescast;
                            break;
                    }

                    // Check special values
                    switch (vote) {
                        case -1:
                            value = gettextCatalog.getString('majority');
                            break;
                        case -2:
                            value = gettextCatalog.getString('undocumented');
                            break;
                        default:
                            if (vote >= 0) {
                                value = vote;
                            } else {
                                value = 0; // value was not defined
                            }
                    }

                    // Calculate percent value
                    var base = this.getPercentBase(config, type);
                    if (base) {
                        percentNumber = Math.round(vote * 100 / (base) * 10) / 10;
                        percentStr = '(' + percentNumber + ' %)';
                    }
                    return {
                        'value': value,
                        'percentStr': percentStr,
                        'percentNumber': percentNumber,
                        'display': value + ' ' + percentStr
                    };
                }
            },
            relations: {
                belongsTo: {
                    'assignments/assignment': {
                        localField: 'assignment',
                        localKey: 'assignment_id',
                    }
                },
                hasMany: {
                    'assignments/polloption': {
                        localField: 'options',
                        foreignKey: 'poll_id',
                    }
                }
            },
        });
    }
])

.factory('AssignmentRelatedUser', [
    'DS',
    function (DS) {
        return DS.defineResource({
            name: 'assignments/relateduser',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

.factory('Assignment', [
    '$http',
    'DS',
    'Projector',
    'AssignmentRelatedUser',
    'AssignmentPoll',
    'jsDataModel',
    'gettext',
    function ($http, DS, Projector, AssignmentRelatedUser, AssignmentPoll, jsDataModel, gettext) {
        var name = 'assignments/assignment';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Election'),
            verboseNamePlural: gettext('Elections'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getAgendaTitle();
                },
                // return true if a specific relation matches for given searchquery
                // (here: related_users/candidates)
                hasSearchResult: function (results) {
                    var assignment = this;
                    // search for related users (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(assignment.assignment_related_users, {'user_id': result.id})) {
                                return true;
                            }
                        }
                    });
                },
                // override project function of jsDataModel factory
                project: function (projectorId, pollId) {
                    var isProjectedIds = this.isProjected(pollId);
                    _.forEach(isProjectedIds, function (id) {
                        $http.post('/rest/core/projector/' + id + '/clear_elements/');
                    });
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/prune_elements/',
                            [{name: 'assignments/assignment', id: this.id, poll: pollId}]
                        );
                    }
                },
                // override isProjected function of jsDataModel factory
                isProjected: function (poll_id) {
                    // Returns the ids of all projectors with an element
                    // with the name 'assignments/assignment'. Else returns an empty list.
                    var self = this;
                    var predicate = function (element) {
                        var value;
                        if (typeof poll_id === 'undefined') {
                            // Assignment detail slide without poll
                            value = element.name == 'assignments/assignment' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id &&
                                typeof element.poll === 'undefined';
                        } else {
                            // Assignment detail slide with specific poll
                            value = element.name == 'assignments/assignment' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id &&
                                typeof element.poll !== 'undefined' &&
                                element.poll == poll_id;
                        }
                        return value;
                    };
                    var isProjectedIds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjectedIds.push(projector.id);
                        }
                    });
                    return isProjectedIds;
                }
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'assignments/relateduser': {
                        localField: 'assignment_related_users',
                        foreignKey: 'assignment_id',
                    },
                    'assignments/poll': {
                        localField: 'polls',
                        foreignKey: 'assignment_id',
                    }
                }
            },
            beforeInject: function (resource, instance) {
                AssignmentRelatedUser.ejectAll({where: {assignment_id: {'==': instance.id}}});
            }
        });
    }
])

.run(['Assignment', function(Assignment) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.pdf', ['OpenSlidesApp.core.pdf'])

.factory('AssignmentContentProvider', [
    '$filter',
    'gettextCatalog',
    'PDFLayout',
    function($filter, gettextCatalog, PDFLayout) {

        var createInstance = function(assignment) {

            // page title
            var title = PDFLayout.createTitle(assignment.title);
            var isElectedSemaphore = false;

            // open posts
            var createPreamble = function() {
                var preambleText = gettextCatalog.getString("Number of persons to be elected") + ": ";
                var memberNumber = ""+assignment.open_posts;
                var preamble = {
                    text: [
                        {
                            text: preambleText,
                            bold: true,
                            style: 'textItem'
                        },
                        {
                            text: memberNumber,
                            style: 'textItem'
                        }
                    ]
                };
                return preamble;
            };

            // description
            var createDescription = function() {
                if (assignment.description) {
                    var descriptionText = gettextCatalog.getString("Description") + ":";
                    var description = [
                        {
                            text: descriptionText,
                            bold: true,
                            style: 'textItem'
                        },
                        {
                            text: assignment.description,
                            style: 'textItem',
                            margin: [10, 0, 0, 0]
                        }
                    ];
                    return description;
                } else {
                    return "";
                }
            };

            // show candidate list (if assignment phase is not 'finished')
            var createCandidateList = function() {
                if (assignment.phase != 2) {
                    var candidates = $filter('orderBy')(assignment.assignment_related_users, 'weight');
                    var candidatesText = gettextCatalog.getString("Candidates") + ": ";
                    var userList = [];

                    angular.forEach(candidates, function(assignmentsRelatedUser) {
                        userList.push({
                                text: assignmentsRelatedUser.user.get_full_name(),
                                margin: [0, 0, 0, 10],
                            }
                        );
                    });

                    var cadidateList = {
                        columns: [
                            {
                                text: candidatesText,
                                bold: true,
                                width: "25%",
                                style: 'textItem'
                            },
                            {
                                ul: userList,
                                style: 'textItem'
                            }
                        ]
                    };
                    return cadidateList;
                } else {
                    return "";
                }
            };

            // handles the case if a candidate is elected or not
            var electedCandidateLine = function(candidateName, pollOption, pollTableBody) {
                if (pollOption.is_elected) {
                    isElectedSemaphore = true;
                    return {
                        text: candidateName + "*",
                        bold: true,
                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                    };
                } else {
                    return {
                        text: candidateName,
                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                    };
                }
            };

            //creates the voting string for the result table and differentiates between special values
            var parseVoteValue = function(voteObject, printLabel) {
                var voteVal = "";

                if (printLabel) {
                    voteVal += voteObject.label + ": ";
                }

                voteVal += voteObject.value;

                if (voteObject.percentStr) {
                    voteVal += " " + voteObject.percentStr;
                }

                voteVal += "\n";
                return voteVal;
            };

            // creates the election result table
            var createPollResultTable = function() {
                var resultBody = [];
                angular.forEach(assignment.polls, function(poll, pollIndex) {
                    if (poll.published) {
                        var pollTableBody = [];

                        resultBody.push({
                            text: gettextCatalog.getString("Ballot") + " " + (pollIndex+1),
                            bold: true,
                            style: 'textItem',
                            margin: [0, 15, 0, 0]
                        });

                        pollTableBody.push([
                            {
                                text: gettextCatalog.getString("Candidates"),
                                style: 'tableHeader',
                            },
                            {
                                text: gettextCatalog.getString("Votes"),
                                style: 'tableHeader',
                            }
                        ]);

                        angular.forEach(poll.options, function(pollOption, optionIndex) {
                            var candidateName = pollOption.candidate.get_full_name();
                            var votes = pollOption.getVotes(); // 0 = yes, 1 = no, 2 = abstain
                            var tableLine = [];

                            tableLine.push(electedCandidateLine(candidateName, pollOption, pollTableBody));
                            if (poll.pollmethod == 'votes') {
                                tableLine.push(
                                    {
                                        text: parseVoteValue(votes[0], false),
                                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                                    }
                                );
                            } else {
                                var resultBlock = [];
                                angular.forEach(votes, function(vote) {
                                    resultBlock.push(parseVoteValue(vote, true));
                                });
                                tableLine.push({
                                        text: resultBlock,
                                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                                    }
                                );
                            }
                            pollTableBody.push(tableLine);
                        });

                        if (poll.votesvalid) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Valid ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votesvalid'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        if (poll.votesinvalid) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Invalid ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votesinvalid'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        if (poll.votescast) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Casted ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votescast'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        var resultTableJsonSting = {
                            table: {
                                widths: ['64%','33%'],
                                headerRows: 1,
                                body: pollTableBody,
                            },
                            layout: 'headerLineOnly',
                        };

                        resultBody.push(resultTableJsonSting);
                    }
                });

                // add the legend to the result body
                if (assignment.polls.length > 0 && isElectedSemaphore) {
                    resultBody.push({
                        text: "* = " + gettextCatalog.getString("is elected"),
                        margin: [0, 5, 0, 0],
                    });
                }

                return resultBody;
            };

            var getContent = function() {
                return [
                    title,
                    createPreamble(),
                    createDescription(),
                    createCandidateList(),
                    createPollResultTable()
                ];
            };

            return {
                getContent: getContent,
                title: assignment.title
            };
        };

    return {
        createInstance: createInstance
    };
}])

.factory('BallotContentProvider', [
    '$filter',
    'gettextCatalog',
    'PDFLayout',
    'Config',
    'User',
    function($filter, gettextCatalog, PDFLayout, Config, User) {
        var createInstance = function(scope, poll, pollNumber) {

            // page title
            var createTitle = function() {
                return {
                    text: scope.assignment.title,
                    style: 'title',
                };
            };

            // poll description
            var createPollHint = function() {
                var description = poll.description ? ': ' + poll.description : '';
                return {
                    text: gettextCatalog.getString("Ballot") + " " + pollNumber + description,
                    style: 'description',
                };
            };

            // election entries
            var createYNBallotEntry = function(decision) {
                var YNColumn = [
                    {
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("Yes"))
                        ]
                    },
                    {
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("No"))
                        ]
                    },
                ];

                if (poll.pollmethod == 'yna') {
                    YNColumn.push({
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("Abstain"))
                        ]
                    });
                }

                return [
                    {
                        text: decision,
                        margin: [40, 10, 0, 0],
                    },
                    {
                        columns: YNColumn
                    }
                ];
            };

            var createSelectionField = function() {
                var candidates = $filter('orderBy')(poll.options, 'weight');
                var candidateBallotList = [];

                if (poll.pollmethod == 'votes') {
                    angular.forEach(candidates, function(option) {
                        var candidate = option.candidate.get_full_name();
                        candidateBallotList.push(PDFLayout.createBallotEntry(candidate));
                    });
                } else {
                    angular.forEach(candidates, function(option) {
                        var candidate = option.candidate.get_full_name();
                        candidateBallotList.push(createYNBallotEntry(candidate));
                    });
                }
                return candidateBallotList;
            };

            var createSection = function(marginTop) {

                // since it is not possible to give a column a fixed height, we draw an "empty" column
                // with a one px width and a fixed top-margin
                return {
                    columns : [
                        {
                            width: 1,
                            margin: [0, marginTop],
                            text: ""
                        },
                        {
                            width: '*',
                            stack: [
                                createTitle(),
                                createPollHint(),
                                createSelectionField(),
                            ]
                        }
                    ]
                };
            };

            var createTableBody = function(numberOfRows, sheetend, maxballots) {
                var ballotstoprint = numberOfRows * 2;
                if (Number.isInteger(maxballots) && maxballots > 0 && maxballots < ballotstoprint) {
                    ballotstoprint = maxballots;
                }
                var tableBody = [];
                while (ballotstoprint > 1){
                    tableBody.push([createSection(sheetend), createSection(sheetend)]);
                    ballotstoprint -= 2;
                }
                if (ballotstoprint == 1) {
                    tableBody.push([createSection(sheetend), '']);
                }
                return tableBody;
            };

            var createContentTable = function() {
                // first, determine how many ballots we need
                var amount;
                var amount_method = Config.get('assignments_pdf_ballot_papers_selection').value;
                switch (amount_method) {
                    case 'NUMBER_OF_ALL_PARTICIPANTS':
                        amount = User.getAll().length;
                        break;
                    case 'NUMBER_OF_DELEGATES':
                        //TODO: assumption that DELEGATES is always group id 2. This may not be true
                        var group_id = 2;
                        amount = User.filter({where: {'groups_id': {contains:group_id} }}).length;
                        break;
                    case 'CUSTOM_NUMBER':
                        amount = Config.get('assignments_pdf_ballot_papers_number').value;
                        break;
                    default:
                        // should not happen.
                        amount = 0;
                }
                var tabledContent = [];
                var rowsperpage;
                var sheetend;
                if (poll.pollmethod == 'votes') {
                    if (poll.options.length <= 4) {
                        sheetend = 105;
                        rowsperpage = 4;
                    } else if (poll.options.length <= 8) {
                        sheetend = 140;
                        rowsperpage = 3;
                    } else if (poll.options.length <= 12) {
                        sheetend = 210;
                        rowsperpage = 2;
                    }
                    else { //works untill ~30 people
                        sheetend = 418;
                        rowsperpage = 1;
                    }
                } else {
                    if (poll.options.length <= 2) {
                        sheetend = 105;
                        rowsperpage = 4;
                    } else if (poll.options.length <= 4) {
                        sheetend = 140;
                        rowsperpage = 3;
                    } else if (poll.options.length <= 6) {
                        sheetend = 210;
                        rowsperpage = 2;
                    } else {
                        sheetend = 418;
                        rowsperpage = 1;
                    }
                }
                var page_entries = rowsperpage * 2;
                var fullpages = Math.floor(amount / page_entries);
                for (var i=0; i < fullpages; i++) {
                    tabledContent.push({
                        table: {
                            headerRows: 1,
                            widths: ['50%', '50%'],
                            body: createTableBody(rowsperpage, sheetend),
                            pageBreak: 'after'
                            },
                            layout: PDFLayout.getBallotLayoutLines(),
                            rowsperpage: rowsperpage
                        });
                }
                // fill the last page only partially
                var lastpage_ballots = amount - (fullpages * page_entries);
                if (lastpage_ballots < page_entries && lastpage_ballots > 0){
                    var partialpage = createTableBody(rowsperpage, sheetend, lastpage_ballots);
                    tabledContent.push({
                        table: {
                            headerRows: 1,
                            widths: ['50%', '50%'],
                            body: partialpage
                        },
                        layout: PDFLayout.getBallotLayoutLines(),
                        rowsperpage: rowsperpage
                    });
                }
                return tabledContent;
            };

            var getContent = function() {
                return createContentTable();
            };

            return {
                getContent: getContent
            };
        };

    return {
        createInstance: createInstance
    };
}])

.factory('AssignmentCatalogContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Config',
    function(gettextCatalog, PDFLayout, Config) {

        var createInstance = function(allAssignmnets) {

            var title = PDFLayout.createTitle(
                    Config.translate(Config.get('assignments_pdf_title').value)
            );

            var createPreamble = function() {
                var preambleText = Config.get('assignments_pdf_preamble').value;
                if (preambleText) {
                    return {
                        text: preambleText,
                        style: "preamble"
                    };
                } else {
                    return "";
                }
            };

            var createTOContent = function(assignmentTitles) {
                var heading = {
                    text: gettextCatalog.getString("Table of contents"),
                    style: "heading2",
                };

                var toc = [];
                angular.forEach(assignmentTitles, function(title) {
                    toc.push({
                        text: title,
                        style: "tableofcontent"
                    });
                });

                return [
                    heading,
                    toc,
                    PDFLayout.addPageBreak()
                ];
            };

            var getContent = function() {
                var content = [];
                var assignmentContent = [];
                var assignmentTitles = [];

                angular.forEach(allAssignmnets, function(assignment, key) {
                    assignmentTitles.push(assignment.title);
                    assignmentContent.push(assignment.getContent());
                    if (key < allAssignmnets.length - 1) {
                        assignmentContent.push(PDFLayout.addPageBreak());
                    }
                });

                content.push(title);
                content.push(createPreamble());
                content.push(createTOContent(assignmentTitles));
                content.push(assignmentContent);
                return content;
            };

            return {
                getContent: getContent
            };
        };

    return {
        createInstance: createInstance
    };
}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.projector', ['OpenSlidesApp.assignments'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('assignments/assignment', {
            template: 'static/templates/assignments/slide_assignment.html',
        });
    }
])

.controller('SlideAssignmentCtrl', [
    '$scope',
    'Assignment',
    'AssignmentPhases',
    'User',
    function($scope, Assignment, AssignmentPhases, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        $scope.showResult = $scope.element.poll;

        Assignment.bindOne(id, $scope, 'assignment');
        $scope.phases = AssignmentPhases;
        User.bindAll({}, $scope, 'users');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.site', [
    'OpenSlidesApp.assignments',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.assignments.pdf',
    'OpenSlidesApp.poll.majority'
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'assignments.assignment.list',
            'img_class': 'pie-chart',
            'title': gettext('Elections'),
            'weight': 400,
            'perm': 'assignments.can_see'
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Elections'),
            'collectionName': 'assignments/assignment',
            'urlDetailState': 'assignments.assignment.detail',
            'weight': 400,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('assignments', {
                url: '/assignments',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Elections'),
                    basePerm: 'assignments.can_see',
                },
            })
            .state('assignments.assignment', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('assignments.assignment.list', {})
            .state('assignments.assignment.detail', {
                controller: 'AssignmentDetailCtrl',
                resolve: {
                    assignmentId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to assignment detail and opens assignment edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from assignment controller use AssignmentForm factory instead to open dialog in front
            // of current view without redirect)
            .state('assignments.assignment.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/assignments/assignment-form.html',
                            controller: 'AssignmentUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                assignmentId: function() {
                                    return $stateParams.id;
                                },
                            },
                            preCloseCallback: function() {
                                $state.go('assignments.assignment.detail', {assignment: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ]
            });
    }
])

// Service for generic assignment form (create and update)
.factory('AssignmentForm', [
    'gettextCatalog',
    'operator',
    'Tag',
    'Assignment',
    'Agenda',
    'AgendaTree',
    function (gettextCatalog, operator, Tag, Assignment, Agenda, AgendaTree) {
        return {
            // ngDialog for assignment form
            getDialog: function (assignment) {
                return {
                    template: 'static/templates/assignments/assignment-form.html',
                    controller: (assignment) ? 'AssignmentUpdateCtrl' : 'AssignmentCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        assignmentId: function () {return assignment ? assignment.id : void 0;}
                    },
                };
            },
            // angular-formly fields for assignment form
            getFormFields: function (isCreateForm) {
                var formFields = [
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    key: 'description',
                    type: 'textarea',
                    templateOptions: {
                        label: gettextCatalog.getString('Description')
                    }
                },
                {
                    key: 'open_posts',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Number of persons to be elected'),
                        type: 'number',
                        min: 1,
                        required: true
                    }
                },
                {
                    key: 'poll_description_default',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Default comment on the ballot paper')
                    }
                },
                {
                    key: 'showAsAgendaItem',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Show as agenda item'),
                        description: gettextCatalog.getString('If deactivated the election appears as internal item on agenda.')
                    },
                    hide: !(operator.hasPerms('assignments.can_manage') && operator.hasPerms('agenda.can_manage'))
                }];

                // parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'agenda_parent_item_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }
                // more (with tags field)
                if (Tag.getAll().length > 0) {
                    formFields.push(
                        {
                            key: 'more',
                            type: 'checkbox',
                            templateOptions: {
                                label: gettextCatalog.getString('Show extended fields')
                            },
                            hide: !operator.hasPerms('assignments.can_manage')
                        },
                        {
                            template: '<hr class="smallhr">',
                            hideExpression: '!model.more'
                        },
                        {
                            key: 'tags_id',
                            type: 'select-multiple',
                            templateOptions: {
                                label: gettextCatalog.getString('Tags'),
                                options: Tag.getAll(),
                                ngOptions: 'option.id as option.name for option in to.options',
                                placeholder: gettextCatalog.getString('Select or search a tag ...')
                            },
                            hideExpression: '!model.more'
                        }
                    );
                }

                return formFields;
            }
        };
    }
])

// Cache for AssignmentPollDetailCtrl so that users choices are keeped during user actions (e. g. save poll form).
.value('AssignmentPollDetailCtrlCache', {})

// Child controller of AssignmentDetailCtrl for each single poll.
.controller('AssignmentPollDetailCtrl', [
    '$scope',
    'MajorityMethodChoices',
    'Config',
    'AssignmentPollDetailCtrlCache',
    'AssignmentPoll',
    function ($scope, MajorityMethodChoices, Config, AssignmentPollDetailCtrlCache, AssignmentPoll) {
        // Define choices.
        $scope.methodChoices = MajorityMethodChoices;
        // TODO: Get $scope.baseChoices from config_variables.py without copying them.

        // Setup empty cache with default values.
        if (typeof AssignmentPollDetailCtrlCache[$scope.poll.id] === 'undefined') {
            AssignmentPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.config('assignments_poll_default_majority_method'),
            };
        }

        // Fetch users choices from cache.
        $scope.method = AssignmentPollDetailCtrlCache[$scope.poll.id].method;

        $scope.recalculateMajorities = function (method) {
            $scope.method = method;
            _.forEach($scope.poll.options, function (option) {
                option.majorityReached = option.isReached(method);
            });
        };
        $scope.recalculateMajorities($scope.method);

        $scope.saveDescriptionChange = function (poll) {
            AssignmentPoll.save(poll);
        };

        // Save current values to cache on destroy of this controller.
        $scope.$on('$destroy', function() {
            AssignmentPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.method,
            };
        });
    }
])

.controller('AssignmentListCtrl', [
    '$scope',
    'ngDialog',
    'AssignmentForm',
    'Assignment',
    'Tag',
    'Agenda',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'AssignmentContentProvider',
    'AssignmentCatalogContentProvider',
    'PdfMakeDocumentProvider',
    'User',
    'osTableFilter',
    'osTableSort',
    'gettext',
    'PdfCreate',
    'AssignmentPhases',
    function($scope, ngDialog, AssignmentForm, Assignment, Tag, Agenda, Projector, ProjectionDefault,
        gettextCatalog, AssignmentContentProvider, AssignmentCatalogContentProvider, PdfMakeDocumentProvider,
        User, osTableFilter, osTableSort, gettext, PdfCreate, AssignmentPhases) {
        Assignment.bindAll({}, $scope, 'assignments');
        Tag.bindAll({}, $scope, 'tags');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'assignments'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.phases = AssignmentPhases;
        $scope.alert = {};

        // Filtering
        $scope.filter = osTableFilter.createInstance('AssignmentTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                tag: [],
                phase: [],
            };
        }
        $scope.filter.propertyList = ['title', 'description'];
        $scope.filter.propertyFunctionList = [
            function (assignment) {
                return gettextCatalog.getString($scope.phases[assignment.phase].display_name);
            },
        ];
        $scope.filter.propertyDict = {
            'assignment_related_users': function (candidate) {
                return candidate.user.get_short_name();
            },
            'tags': function (tag) {
                return tag.name;
            },
        };
        $scope.getItemId = {
            tag: function (assignment) {return assignment.tags_id;},
            phase: function (assignment) {return assignment.phase;},
        };
        // Sorting
        $scope.sort = osTableSort.createInstance();
        $scope.sort.column = 'title';
        $scope.sortOptions = [
            {name: 'agenda_item.getItemNumberWithAncestors()',
             display_name: gettext('Item')},
            {name: 'title',
             display_name: gettext('Title')},
            {name: 'phase',
             display_name: gettext('Phase')},
            {name: 'assignment_related_users.length',
             display_name: gettext('Number of candidates')},
        ];
        $scope.hasTag = function (assignment, tag) {
            return _.indexOf(assignment.tags_id, tag.id) > -1;
        };
        $scope.toggleTag = function (assignment, tag) {
            if ($scope.hasTag(assignment, tag)) {
                assignment.tags_id = _.filter(assignment.tags_id, function (tag_id){
                    return tag_id != tag.id;
                });
            } else {
                assignment.tags_id.push(tag.id);
            }
            Assignment.save(assignment);
        };
        // update phase
        $scope.updatePhase = function (assignment, phase_id) {
            assignment.phase = phase_id;
            Assignment.save(assignment);
        };
        // open new/edit dialog
        $scope.openDialog = function (assignment) {
            ngDialog.open(AssignmentForm.getDialog(assignment));
        };
        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            angular.forEach($scope.assignments, function (assignment) {
                assignment.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.assignments, function (assignment) {
                    assignment.selected = false;
                });
            }
        };
        // delete all selected assignments
        $scope.deleteMultiple = function () {
            angular.forEach($scope.assignments, function (assignment) {
                if (assignment.selected)
                    Assignment.destroy(assignment.id);
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete single assignment
        $scope.delete = function (assignment) {
            Assignment.destroy(assignment.id);
        };
        // create the PDF List
        $scope.makePDF_assignmentList = function () {
            var filename = gettextCatalog.getString("Elections") + ".pdf";
            var assignmentContentProviderArray = [];

            //convert the filtered assignments to content providers
            angular.forEach($scope.assignmentsFiltered, function(assignment) {
                assignmentContentProviderArray.push(AssignmentContentProvider.createInstance(assignment));
            });

            var assignmentCatalogContentProvider =
                AssignmentCatalogContentProvider.createInstance(assignmentContentProviderArray);
            var documentProvider =
                PdfMakeDocumentProvider.createInstance(assignmentCatalogContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };
    }
])

.controller('AssignmentDetailCtrl', [
    '$scope',
    '$http',
    '$filter',
    '$timeout',
    'filterFilter',
    'gettext',
    'ngDialog',
    'AssignmentForm',
    'operator',
    'Assignment',
    'User',
    'assignmentId',
    'Projector',
    'ProjectionDefault',
    'AssignmentContentProvider',
    'BallotContentProvider',
    'PdfMakeDocumentProvider',
    'PdfMakeBallotPaperProvider',
    'gettextCatalog',
    'PdfCreate',
    'AssignmentPhases',
    'ErrorMessage',
    function($scope, $http, $filter, $timeout, filterFilter, gettext, ngDialog, AssignmentForm, operator, Assignment,
        User, assignmentId, Projector, ProjectionDefault, AssignmentContentProvider, BallotContentProvider,
        PdfMakeDocumentProvider, PdfMakeBallotPaperProvider, gettextCatalog, PdfCreate, AssignmentPhases,
        ErrorMessage) {
        User.bindAll({}, $scope, 'users');
        var assignment = Assignment.get(assignmentId);
        Assignment.loadRelations(assignment, 'agenda_item');
        // This flag is for setting 'activeTab' to recently added (last) ballot tab.
        // Set this flag, if ballots are added/removed. When the next autoupdate comes
        // in, the tabset will be updated.
        var updateBallotTabsFlag = true;
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'assignments'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.$watch(function () {
            return Assignment.lastModified(assignmentId);
        }, function () {
            // setup sorting of candidates
            $scope.relatedUsersSorted = $filter('orderBy')(assignment.assignment_related_users, 'weight');
            $scope.assignment = Assignment.get(assignment.id);
            if (updateBallotTabsFlag) {
                $scope.activeTab = $scope.assignment.polls.length - 1;
                updateBallotTabsFlag = false;
            }
        });
        $scope.candidateSelectBox = {};
        $scope.phases = AssignmentPhases;
        $scope.alert = {};

        // open edit dialog
        $scope.openDialog = function () {
            ngDialog.open(AssignmentForm.getDialog($scope.assignment));
        };
        // add (nominate) candidate
        $scope.addCandidate = function (userId) {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/candidature_other/', {'user': userId})
                .then(function (success){
                    $scope.alert.show = false;
                    $scope.candidateSelectBox = {};
                }, function (error){
                    $scope.alert = ErrorMessage.forAlert(error);
                    $scope.candidateSelectBox = {};
                });
        };
        // remove candidate
        $scope.removeCandidate = function (userId) {
            $http.delete('/rest/assignments/assignment/' + assignmentId + '/candidature_other/',
                    {headers: {'Content-Type': 'application/json'},
                     data: JSON.stringify({user: userId})})
                .then(function (success) {},
                    function (error) {
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
        };
        // add me (nominate self as candidate)
        $scope.addMe = function () {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/candidature_self/', {}).then(
                function (success) {
                    $scope.alert.show = false;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // remove me (withdraw own candidature)
        $scope.removeMe = function () {
            $http.delete('/rest/assignments/assignment/' + assignmentId + '/candidature_self/').then(
                function (success) {
                    $scope.alert.show = false;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // check if current user is already a candidate (elected==false)
        $scope.isCandidate = function () {
            var check = $scope.assignment.assignment_related_users.map(function(candidate) {
                if (!candidate.elected) {
                    return candidate.user_id;
                }
            }).indexOf(operator.user.id);
            if (check > -1) {
                return true;
            } else {
                return false;
            }
        };
        // Sort all candidates
        $scope.treeOptions = {
            dropped: function () {
                var sortedCandidates = [];
                _.forEach($scope.relatedUsersSorted, function (user) {
                    sortedCandidates.push(user.id);
                });
                $http.post('/rest/assignments/assignment/' + assignmentId + '/sort_related_users/',
                    {related_users: sortedCandidates}
                );
            }
        };
        // update phase
        $scope.updatePhase = function (phase_id) {
            $scope.assignment.phase = phase_id;
            Assignment.save($scope.assignment);
        };
        // create new ballot
        $scope.createBallot = function () {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/create_poll/').then(
                function (success) {
                    $scope.alert.show = false;
                    if (assignment.phase === 0) {
                        $scope.updatePhase(1);
                    }
                    updateBallotTabsFlag = true;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // delete ballot
        $scope.deleteBallot = function (poll) {
            poll.DSDestroy().then(
                function (success) {
                    $scope.activeTab = $scope.activeTab - 1;
                    updateBallotTabsFlag = true;
                }
            );
        };
        // edit poll dialog
        $scope.editPollDialog = function (poll, ballot) {
            ngDialog.open({
                template: 'static/templates/assignments/assignmentpoll-form.html',
                controller: 'AssignmentPollUpdateCtrl',
                className: 'ngdialog-theme-default',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    assignmentpollId: function () {return poll.id;},
                    ballot: function () {return ballot;},
                }
            });
        };
        // publish ballot
        $scope.togglePublishBallot = function (poll) {
            poll.DSUpdate({
                    assignment_id: assignmentId,
                    published: !poll.published,
            })
            .then(function (success) {
                $scope.alert.show = false;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // mark candidate as (not) elected
        $scope.markElected = function (user, reverse) {
            if (reverse) {
                $http.delete(
                    '/rest/assignments/assignment/' + assignmentId + '/mark_elected/', {
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        data: JSON.stringify({user: user})
                });
            } else {
                $http.post('/rest/assignments/assignment/' + assignmentId + '/mark_elected/', {'user': user});
            }

        };

        //creates the document as pdf
        $scope.makePDF_singleAssignment = function() {
            var filename = gettextCatalog.getString("Election") + "_" + $scope.assignment.title + ".pdf";
            var assignmentContentProvider = AssignmentContentProvider.createInstance($scope.assignment);
            var documentProvider = PdfMakeDocumentProvider.createInstance(assignmentContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };

        //creates the ballotpaper as pdf
        $scope.makePDF_assignmentpoll = function(pollID) {
            var thePoll;
            var pollNumber;
            angular.forEach($scope.assignment.polls, function(poll, pollIndex) {
                if (poll.id == pollID) {
                    thePoll = poll;
                    pollNumber = pollIndex+1;
                }
            });
            var filename = gettextCatalog.getString("Ballot") + "_" + pollNumber + "_" + $scope.assignment.title + ".pdf";
            var ballotContentProvider = BallotContentProvider.createInstance($scope, thePoll, pollNumber);
            var documentProvider = PdfMakeBallotPaperProvider.createInstance(ballotContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };

        // Just mark some vote value strings for translation.
        gettext('Yes');
        gettext('No');
        gettext('Abstain');
    }
])

.controller('AssignmentCreateCtrl', [
    '$scope',
    '$state',
    'Assignment',
    'AssignmentForm',
    'Agenda',
    'AgendaUpdate',
    'ErrorMessage',
    function($scope, $state, Assignment, AssignmentForm, Agenda, AgendaUpdate, ErrorMessage) {
        $scope.model = {};
        // set default value for open posts form field
        $scope.model.open_posts = 1;
        // get all form fields
        $scope.formFields = AssignmentForm.getFormFields(true);
        // save assignment
        $scope.save = function(assignment, gotoDetailView) {
            Assignment.create(assignment).then(
                function (success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (assignment.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: assignment.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    if (gotoDetailView) {
                        $state.go('assignments.assignment.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('AssignmentUpdateCtrl', [
    '$scope',
    '$state',
    'Assignment',
    'AssignmentForm',
    'Agenda',
    'AgendaUpdate',
    'assignmentId',
    'ErrorMessage',
    function($scope, $state, Assignment, AssignmentForm, Agenda, AgendaUpdate, assignmentId, ErrorMessage) {
        var assignment = Assignment.get(assignmentId);
        $scope.alert = {};
        // set initial values for form model by create deep copy of assignment object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(assignment);
        // get all form fields
        $scope.formFields = AssignmentForm.getFormFields();
        var agenda_item = Agenda.get(assignment.agenda_item_id);
        for (var i = 0; i < $scope.formFields.length; i++) {
            if ($scope.formFields[i].key == "showAsAgendaItem") {
                // get state from agenda item (hidden/internal or agenda item)
                $scope.formFields[i].defaultValue = !assignment.agenda_item.is_hidden;
            } else if($scope.formFields[i].key == 'agenda_parent_item_id') {
                $scope.formFields[i].defaultValue = agenda_item.parent_id;
            }
        }

        // save assignment
        $scope.save = function (assignment, gotoDetailView) {
            // inject the changed assignment (copy) object back into DS store
            Assignment.inject(assignment);
            // save change assignment object on server
            Assignment.save(assignment).then(
                function(success) {
                    var changes = [{key: 'type', value: (assignment.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: assignment.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    if (gotoDetailView) {
                        $state.go('assignments.assignment.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original assignment object from server
                    Assignment.refresh(assignment);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('AssignmentPollUpdateCtrl', [
    '$scope',
    '$filter',
    'gettextCatalog',
    'AssignmentPoll',
    'assignmentpollId',
    'ballot',
    'ErrorMessage',
    function($scope, $filter, gettextCatalog, AssignmentPoll, assignmentpollId, ballot, ErrorMessage) {
        // set initial values for form model by create deep copy of assignmentpoll object
        // so detail view is not updated while editing poll
        var assignmentpoll = angular.copy(AssignmentPoll.get(assignmentpollId));
        $scope.model = assignmentpoll;
        $scope.ballot = ballot;
        $scope.formFields = [];
        $scope.alert = {};

        // add dynamic form fields
        var options = $filter('orderBy')(assignmentpoll.options, 'weight');
        options.forEach(function(option) {
            var defaultValue;
            if (assignmentpoll.pollmethod == 'yna' || assignmentpoll.pollmethod == 'yn') {
                defaultValue = {};
                _.forEach(option.votes, function (vote) {
                    defaultValue[vote.value.toLowerCase()] = vote.weight;
                });

                $scope.formFields.push(
                    {
                        noFormControl: true,
                        template: '<strong>' + option.candidate.get_full_name() + '</strong>'
                    },
                    {
                        key: 'yes_' + option.candidate_id,
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Yes'),
                            type: 'number',
                            required: true
                        },
                        defaultValue: defaultValue.yes
                    },
                    {
                        key: 'no_' + option.candidate_id,
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('No'),
                            type: 'number',
                            required: true
                        },
                        defaultValue: defaultValue.no
                    }
                );
                if (assignmentpoll.pollmethod == 'yna'){
                    $scope.formFields.push(
                    {
                        key:'abstain_' + option.candidate_id,
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Abstain'),
                            type: 'number',
                            required: true
                        },
                        defaultValue: defaultValue.abstain
                    });
                }
            } else { // votes method
                if (option.votes.length) {
                    defaultValue = option.votes[0].weight;
                }
                $scope.formFields.push(
                    {
                        key: 'vote_' + option.candidate_id,
                        type: 'input',
                        templateOptions: {
                            label: option.candidate.get_full_name(),
                            type: 'number',
                            required: true
                        },
                        defaultValue: defaultValue
                    });
            }
        });
        // add general form fields
        $scope.formFields.push(
                {
                    key: 'votesvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Valid ballots'),
                        type: 'number'
                    }
                },
                {
                    key: 'votesinvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Invalid ballots'),
                        type: 'number'
                    }
                },
                {
                    key: 'votescast',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Casted ballots'),
                        type: 'number'
                    }
                }
        );

        // save assignmentpoll
        $scope.save = function (poll) {
            var votes = [];
            if (assignmentpoll.pollmethod == 'yna') {
                assignmentpoll.options.forEach(function(option) {
                    votes.push({
                        "Yes": poll['yes_' + option.candidate_id],
                        "No": poll['no_' + option.candidate_id],
                        "Abstain": poll['abstain_' + option.candidate_id]
                    });
                });
            } else if (assignmentpoll.pollmethod == 'yn') {
                    assignmentpoll.options.forEach(function(option) {
                        votes.push({
                            "Yes": poll['yes_' + option.candidate_id],
                            "No": poll['no_' + option.candidate_id]
                            });
                        });
            } else {
                assignmentpoll.options.forEach(function(option) {
                    votes.push({
                        "Votes": poll['vote_' + option.candidate_id],
                    });
                });
            }
            // save change poll object on server
            poll.DSUpdate({
                assignment_id: poll.assignment_id,
                votes: votes,
                votesvalid: poll.votesvalid,
                votesinvalid: poll.votesinvalid,
                votescast: poll.votescast
            })
            .then(function(success) {
                $scope.alert.show = false;
                $scope.closeThisDialog();
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

//mark all assignment config strings for translation with Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Election method');
        gettext('Automatic assign of method');
        gettext('Always one option per candidate');
        gettext('Always Yes-No-Abstain per candidate');
        gettext('Always Yes/No per candidate');
        gettext('Elections');
        gettext('Ballot and ballot papers');
        gettext('The 100-%-base of an election result consists of');
        gettext('For Yes/No/Abstain per candidate and Yes/No per candidate the 100-%-base ' +
                'depends on the election method: If there is only one option per candidate, ' +
                'the sum of all votes of all candidates is 100 %. Otherwise for each ' +
                'candidate the sum of all votes is 100 %.');
        gettext('Yes/No/Abstain per candidate');
        gettext('Yes/No per candidate');
        gettext('All valid ballots');
        gettext('All casted ballots');
        gettext('Disabled (no percents)');
        gettext('Number of ballot papers (selection)');
        gettext('Number of all delegates');
        gettext('Number of all participants');
        gettext('Use the following custom number');
        gettext('Custom number of ballot papers');
        gettext('Required majority');
        gettext('Default method to check whether a candidate has reached the required majority.');
        gettext('Simple majority');
        gettext('Two-thirds majority');
        gettext('Three-quarters majority');
        gettext('Disabled');
        gettext('Title for PDF document (all elections)');
        gettext('Preamble text for PDF document (all elections)');
        //other translations
        gettext('Searching for candidates');
        gettext('Voting');
        gettext('Finished');
    }
]);

}());

(function () {

'use strict';

// The core module used for the OpenSlides site and the projector
angular.module('OpenSlidesApp.core', [
    'js-data',
    'gettext',
    'ngAnimate',
    'ngBootbox',
    'ngSanitize',  // TODO: only use this in functions that need it.
    'ui.bootstrap',
    'ui.bootstrap.datetimepicker',
    'ui.tree',
    'pdf',
    'OpenSlidesApp-templates',
])

.config([
    'DSProvider',
    'DSHttpAdapterProvider',
    function(DSProvider, DSHttpAdapterProvider) {
        DSProvider.defaults.reapAction = 'none';
        DSProvider.defaults.basePath = '/rest';
        DSProvider.defaults.afterReap = function(model, items) {
            if (items.length > 5) {
                model.findAll({}, {bypassCache: true});
            } else {
                _.forEach(items, function (item) {
                    model.refresh(item[model.idAttribute]);
                });
            }
        };
        DSHttpAdapterProvider.defaults.forceTrailingSlash = true;
    }
])

.factory('ProjectorID', [
    function () {
        return function () {
            return /projector\/(\d+)\//.exec(location.pathname)[1];
        };
    }
])

.factory('autoupdate', [
    'DS',
    'REALM',
    'ProjectorID',
    '$q',
    'ErrorMessage',
    function (DS, REALM, ProjectorID, $q, ErrorMessage) {
        var socket = null;
        var recInterval = null;

        var websocketProtocol;
        if (location.protocol == 'https:') {
            websocketProtocol = 'wss:';
        } else {
            websocketProtocol = 'ws:';
        }

        var websocketPath;
        if (REALM === 'site') {
          websocketPath = '/ws/site/';
        } else if (REALM === 'projector') {
          websocketPath = '/ws/projector/' + ProjectorID() + '/';
        } else {
          console.error('The constant REALM is not set properly.');
        }

        var Autoupdate = {};
        Autoupdate.messageReceivers = [];
        // We use later a promise to defer the first message of the established ws connection.
        Autoupdate.firstMessageDeferred = $q.defer();
        Autoupdate.onMessage = function (receiver) {
            Autoupdate.messageReceivers.push(receiver);
        };
        Autoupdate.reconnect = function () {
            if (socket) {
                socket.close();
            }
        };
        Autoupdate.newConnect = function () {
            socket = new WebSocket(websocketProtocol + '//' + location.host + websocketPath);
            clearInterval(recInterval);
            socket.onclose = function (event) {
                socket = null;
                recInterval = setInterval(function () {
                    Autoupdate.newConnect();
                }, 1000);
                ErrorMessage.setConnectionError();
            };
            socket.onmessage = function (event) {
                _.forEach(Autoupdate.messageReceivers, function (receiver) {
                    receiver(event.data);
                });
                // Check if the promise is not resolved yet.
                if (Autoupdate.firstMessageDeferred.promise.$$state.status === 0) {
                    Autoupdate.firstMessageDeferred.resolve();
                }
                ErrorMessage.clearConnectionError();
            };
        };
        return Autoupdate;
    }
])

.factory('operator', [
    'User',
    'Group',
    function (User, Group) {
        var operator = {
            user: null,
            perms: [],
            isAuthenticated: function () {
                return !!this.user;
            },
            setUser: function(user_id, user_data) {
                if (user_id && user_data) {
                    operator.user = User.inject(user_data);
                } else {
                    operator.user = null;
                }
            },
            // Returns true if the operator has at least one perm of the perms-list.
            hasPerms: function(perms) {
                if (typeof perms === 'string') {
                    perms = perms.split(' ');
                }
                return _.intersection(perms, operator.perms).length > 0;
            },
            reloadPerms: function () {
                if (operator.user) {
                    operator.perms = operator.user.getPerms();
                } else {
                    var defaultGroup = Group.get(1);
                    operator.perms = defaultGroup ? defaultGroup.permissions : [];
                }
            },
            // Returns true if the operator is a member of group.
            isInGroup: function(group) {
                return _.indexOf(operator.user.groups_id, group.id) > -1;
            },
        };
        return operator;
    }
])

// gets all in OpenSlides available languages
.factory('Languages', [
    'gettext',
    'gettextCatalog',
    'OpenSlidesPlugins',
    '$ngBootbox',
    function (gettext, gettextCatalog, OpenSlidesPlugins, $ngBootbox) {
        return {
            // get all available languages
            getLanguages: function () {
                var current = gettextCatalog.getCurrentLanguage();
                // Define here new languages...
                var languages = [
                    { code: 'en', name: 'English' },
                    { code: 'de', name: 'Deutsch' },
                    { code: 'fr', name: 'Français' },
                    { code: 'es', name: 'Español' },
                    { code: 'pt', name: 'Português' },
                    { code: 'cs', name: 'Čeština'},
                    { code: 'ru', name: 'русский'},
                ];
                angular.forEach(languages, function (language) {
                    if (language.code == current)
                        language.selected = true;
                });
                return languages;
            },
            // get detected browser language code
            getBrowserLanguage: function () {
                var lang = navigator.language || navigator.userLanguage;
                if (!navigator.language && !navigator.userLanguage) {
                    lang = 'en';
                } else {
                    if (lang.indexOf('-') !== -1)
                        lang = lang.split('-')[0];
                    if (lang.indexOf('_') !== -1)
                        lang = lang.split('_')[0];
                }
                return lang;
            },
            // set current language and return updated languages object array
            setCurrentLanguage: function (lang) {
                var languages = this.getLanguages();
                var plugins = OpenSlidesPlugins.getAll();
                angular.forEach(languages, function (language) {
                    language.selected = false;
                    if (language.code == lang) {
                        language.selected = true;
                        gettextCatalog.setCurrentLanguage(lang);
                        // Plugins
                        if (lang != 'en') {
                            gettextCatalog.loadRemote("static/i18n/" + lang + ".json").then(function (success) {
                                // translate ng-bootbox directives when the translations are available.
                                $ngBootbox.addLocale(lang, {
                                    OK: gettextCatalog.getString('OK'),
                                    CANCEL: gettextCatalog.getString('Cancel'),
                                    CONFIRM: gettextCatalog.getString('OK'), // Yes, 'OK' is the original string.
                                });
                                $ngBootbox.setLocale(lang);
                            });
                            // load language files from plugins
                            angular.forEach(plugins, function (plugin) {
                                if (plugin.languages.indexOf(lang) != -1) {
                                    gettextCatalog.loadRemote("static/i18n/" + plugin.name + '/' + lang + ".json");
                                }
                            });
                        }
                    }
                });
                return languages;
            }
        };
    }
])

// set browser language as default language for OpenSlides
.run([
    'gettextCatalog',
    'Languages',
    function(gettextCatalog, Languages) {
        // set detected browser language as default language (fallback: 'en')
        Languages.setCurrentLanguage(Languages.getBrowserLanguage());

        // Set this to true for debug. Helps to find untranslated strings by
        // adding "[MISSING]:".
        gettextCatalog.debug = false;
    }
])

.factory('dsEject', [
    'DS',
    function (DS) {
        return function (collection, instance) {
            var Resource = DS.definitions[collection];
            if (Resource.relationList) {
                Resource.relationList.forEach(function (relationDef) {
                    if (relationDef.foreignKey && !relationDef.osProtectedRelation) {
                        var query = {};
                        query[relationDef.foreignKey] = instance[Resource.idAttribute];
                        Resource.getResource(relationDef.relation).ejectAll(query);
                    }
                });
            }
        };
    }
])

.run([
    'DS',
    'autoupdate',
    'dsEject',
    function (DS, autoupdate, dsEject) {
        autoupdate.onMessage(function(json) {
            // TODO: when MODEL.find() is called after this
            //       a new request is fired. This could be a bug in DS
            var dataList = [];
            try {
                 dataList = JSON.parse(json);
            } catch(err) {
                console.error(json);
            }

            var dataListByCollection = _.groupBy(dataList, 'collection');
            _.forEach(dataListByCollection, function(list, key) {
                var changedElements = [];
                var deletedElements = [];
                var collectionString = key;
                _.forEach(list, function(data) {
                    // Uncomment this line for debugging to log all autoupdates:
                    // console.log("Received object: " + data.collection + ", " + data.id);

                    // remove (=eject) object from local DS store
                    var instance = DS.get(data.collection, data.id);
                    if (instance) {
                        dsEject(data.collection, instance);
                    }
                    // check if object changed or deleted
                    if (data.action === 'changed') {
                        changedElements.push(data.data);
                    } else if (data.action === 'deleted') {
                        deletedElements.push(data.id);
                    } else {
                        console.error('Error: Undefined action for received object' +
                            '(' + data.collection + ', ' + data.id + ')');
                    }
                });
                // add (=inject) all given objects into local DS store
                if (changedElements.length > 0) {
                    DS.inject(collectionString, changedElements);
                }
                // delete (=eject) all given objects from local DS store
                // (note: js-data does not provide 'bulk eject' as for DS.inject)
                _.forEach(deletedElements, function(id) {
                    DS.eject(collectionString, id);
                });
            });
        });
    }
])

// Save the server time to the rootscope.
.run([
    '$http',
    '$rootScope',
    function ($http, $rootScope) {
        // Loads server time and calculates server offset
        $rootScope.serverOffset = 0;
        $http.get('/core/servertime/')
        .then(function(data) {
            $rootScope.serverOffset = Math.floor(Date.now() / 1000 - data.data);
        });
    }
])

.run([
    'Config',
    '$rootScope',
    function (Config, $rootScope) {
        $rootScope.config = function (key) {
            try {
                return Config.get(key).value;
            }
            catch(err) {
                return '';
            }
        };
    }
])

// Make the indexOf available in every scope; needed for the projectorbuttons
.run([
    '$rootScope',
    function ($rootScope) {
        $rootScope.inArray = function (array, value) {
            return _.indexOf(array, value) > -1;
        };
    }
])

// Template hooks
.factory('templateHooks', [
    function () {
        var hooks = {};
        return {
            hooks: hooks,
            registerHook: function (hook) {
                if (hooks[hook.Id] === undefined) {
                    hooks[hook.Id] = [];
                }
                hooks[hook.Id].push(hook);
            }
        };
    }
])

.directive('templateHook', [
    '$compile',
    'templateHooks',
    function ($compile, templateHooks) {
        return {
            restrict: 'E',
            template: '',
            link: function (scope, iElement, iAttr) {
                var hooks = templateHooks.hooks[iAttr.hookName];
                var html;
                if (hooks) {
                    html = hooks.map(function (hook) {
                        return '<div>' + hook.template + '</div>';
                    }).join('');
                } else {
                    html = '';
                }
                iElement.append($compile(html)(scope));
            }
        };
    }
])

/*
 * This places a projector button in the document.
 *
 * Example: <projector-button model="motion" default-projector.id="defPrId"
 *           arg="2" content="{{ 'project' | translate }}"></projector-button>
 * This button references to model (in this example 'motion'). Also a defaultProjectionId
 * has to be given. In the example it's a scope variable. The next two parameters are additional:
 *   - arg: Then the model.project and model.isProjected will be called with
 *          this argument (e. g.: model.project(2))
 *   - content: A text placed behind the projector symbol.
 */
.directive('projectorButton', [
    'Projector',
    function (Projector) {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/projector-button.html',
            link: function (scope, element, attributes) {
                if (!attributes.model) {
                    throw 'A model has to be given!';
                } else if (!attributes.defaultProjectorId) {
                    throw 'A default-projector-id has to be given!';
                }

                Projector.bindAll({}, scope, 'projectors');

                scope.$watch(attributes.model, function (model) {
                    scope.model = model;
                });

                scope.$watch(attributes.defaultProjectorId, function (defaultProjectorId) {
                    scope.defaultProjectorId = defaultProjectorId;
                });

                if (attributes.arg) {
                    scope.$watch(attributes.arg, function (arg) {
                        scope.arg = arg;
                    });
                }

                if (attributes.content) {
                    attributes.$observe('content', function (content) {
                        scope.content = content;
                    });
                }
            }
        };
    }
])

.factory('jsDataModel', [
    '$http',
    'Projector',
    function($http, Projector) {
        var BaseModel = function() {};
        BaseModel.prototype.project = function(projectorId) {
            // if this object is already projected on projectorId, delete this element from this projector
            var isProjectedIds = this.isProjected();
            _.forEach(isProjectedIds, function (id) {
                $http.post('/rest/core/projector/' + id + '/clear_elements/');
            });
            // Show the element, if it was not projected before on the given projector
            if (_.indexOf(isProjectedIds, projectorId) == -1) {
                return $http.post(
                    '/rest/core/projector/' + projectorId + '/prune_elements/',
                    [{name: this.getResourceName(), id: this.id}]
                );
            }
        };
        BaseModel.prototype.isProjected = function() {
            // Returns the ids of all projectors if there is a projector element
            // with the same name and the same id. Else returns an empty list.
            var self = this;
            var predicate = function (element) {
                return element.name == self.getResourceName() &&
                    typeof element.id !== 'undefined' &&
                    element.id == self.id;
            };
            var isProjectedIds = [];
            Projector.getAll().forEach(function (projector) {
                if (typeof _.findKey(projector.elements, predicate) === 'string') {
                    isProjectedIds.push(projector.id);
                }
            });
            return isProjectedIds;
        };
        return BaseModel;
    }
])

.factory('ErrorMessage', [
    '$timeout',
    'gettextCatalog',
    'Messaging',
    function ($timeout, gettextCatalog, Messaging) {
        return {
            forAlert: function (error) {
                var message = gettextCatalog.getString('Error') + ': ';

                if (!error.data) {
                    message += gettextCatalog.getString("The server didn't respond.");
                } else if (error.data.detail) {
                    message += error.data.detail;
                } else {
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                }
                return { type: 'danger', msg: message, show: true };
            },
            setConnectionError: function () {
                $timeout(function () {
                    Messaging.createOrEditMessage(
                        'connectionLostMessage',
                        gettextCatalog.getString('Connection lost. You are not connected to the server anymore.'),
                        'error',
                        {noClose: true});
                }, 1);
            },
            clearConnectionError: function () {
                $timeout(function () {
                    Messaging.deleteMessage('connectionLostMessage');
                }, 1);
            },
        };
    }
])

/* Messaging factory. The text is html-binded into the document, so you can
 * provide also html markup for the messages. There are 4 types: 'info',
 * 'success', 'warning', 'error'. The timeout is for autodeleting the message.
 * Args that could be provided:
 * - timeout: Milliseconds until autoclose the message
 * - noClose: Whether to show the close button*/
.factory('Messaging', [
    '$timeout',
    function($timeout) {
        var callbackList = [],
            messages = {},
            idCounter = 0;

        var onChange = function () {
            _.forEach(callbackList, function (callback) {
                callback();
            });
        };

        return {
            addMessage: function (text, type, args) {
                var id = idCounter++;
                return this.createOrEditMessage(id, text, type, args);
            },
            createOrEditMessage: function (id, text, type, args) {
                if (!args) {
                    args = {};
                }
                if (messages[id] && messages[id].timeout) {
                    $timeout.cancel(messages[id].timeout);
                }
                messages[id] = {
                    text: text,
                    type: type,
                    id: id,
                    args: args,
                };
                if (typeof args.timeout === 'number' && args.timeout > 0) {
                    var self = this;
                    messages[id].timeout = $timeout(function () {
                        self.deleteMessage(id);
                    }, args.timeout);
                }
                onChange();
                return id;
            },
            deleteMessage: function (id) {
                delete messages[id];
                onChange();
            },
            getMessages: function () {
                return messages;
            },
            registerMessageChangeCallback: function (fn) {
                if (typeof fn === 'function') {
                    callbackList.push(fn);
                } else {
                    throw 'fn has to be a function';
                }
            },
        };
    }
])

.factory('Tag', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/tag',
        });
    }
])

.factory('Config', [
    '$http',
    'gettextCatalog',
    'DS',
    function($http, gettextCatalog, DS) {
        var configOptions;
        return DS.defineResource({
            name: 'core/config',
            idAttribute: 'key',
            configOptions: configOptions,
            getConfigOptions: function () {
                if (!this.configOptions) {
                    this.configOptions = $http({ 'method': 'OPTIONS', 'url': '/rest/core/config/' });
                }
                return this.configOptions;
            },
            translate: function (value) {
                return gettextCatalog.getString(value);
            }
        });
    }
])

.factory('ChatMessage', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/chat-message',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

/*
 * Provides a function for plugins to register as new plugin.
 *
 * Get all registerd plugins via 'OpenSlidesPlugins.getAll()'.
 *
 * Example code for plugins:
 *
 *  .config([
 *      'OpenSlidesPluginsProvider',
 *       function(OpenSlidesPluginsProvider) {
 *          OpenSlidesPluginsProvider.registerPlugin({
 *              name: 'openslides_votecollector',
 *              display_name: 'VoteCollector',
 *              languages: ['de']
 *          });
 *      }
 *  ])
 */
.provider('OpenSlidesPlugins', [
    function () {
        var provider = this;
        provider.plugins = [];
        provider.registerPlugin = function (plugin) {
            provider.plugins.push(plugin);
        };
        provider.$get = [
            function () {
                return {
                    getAll: function () {
                        return provider.plugins;
                    }
                };
            }
        ];
    }
])


// Configs for CKEditor which has to set while startup of OpenSlides
.config(
    function() {
        CKEDITOR.disableAutoInline = true;
    }
)

// Options for CKEditor used in various create and edit views.
// Required in core/base.js because MotionComment factory which used this
// factory has to placed in motions/base.js.
.factory('Editor', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getOptions: function (images) {
                return {
                    on: {
                        instanceReady: function() {
                            // This adds a listener to ckeditor to remove unwanted blank lines on import.
                            // Clipboard content varies heavily in structure and html code, depending on the "sender".
                            // Here it is first parsed into a pseudo-DOM (two lines taken from a ckeditor
                            // paste example on the ckeditor site).
                            this.on('paste', function(evt) {
                                if (evt.data.type == 'html') {
                                    var fragment = CKEDITOR.htmlParser.fragment.fromHtml(evt.data.dataValue);
                                    var writer = new CKEDITOR.htmlParser.basicWriter();
                                    // html content will now be in a dom-like structure inside 'fragment'.
                                    this.filter.applyTo(fragment);
                                    if (fragment.children) {
                                        // If this fragment is DOM-like, it may contain nested properties
                                        // (being html nodes). Traverse the children and check if it is a
                                        // child only containing empty <br> or <p>.
                                        // new_content_children will finally contain all nodes that are
                                        // not empty.
                                        var new_content_children = [];
                                        _.forEach(fragment.children, function (child) {
                                            var empty = true;
                                            if (child.children){
                                                _.forEach(child.children, function(grandchild) {
                                                    if (grandchild.name != 'p' && grandchild.name != 'br') {
                                                        empty = false;
                                                    } else if (grandchild.isEmpty !== true) {
                                                        empty = false;
                                                    }
                                                });
                                                if (empty === false) {
                                                    new_content_children.push(child);
                                                }
                                            } else {
                                                if (child.name != 'p' && child.name != 'br' &&
                                                    child.isEmpty !== true){
                                                    new_content_children.push(child);
                                                }
                                            }
                                        });
                                        fragment.children = new_content_children;
                                    }
                                    fragment.writeHtml(writer);
                                    // Return the re-created fragment without the empty <p> and <br> into the
                                    // editor import processing (same as at the begin of the function: by ckeditor)
                                    evt.data.dataValue = writer.getHtml();
                                }
                            });
                        }
                    },
                    customConfig: '',
                    disableNativeSpellChecker: false,
                    language_list: [
                        'fr:français',
                        'es:español',
                        'pt:português',
                        'en:english',
                        'de:deutsch',
                        'cs:čeština'],
                    language: gettextCatalog.getCurrentLanguage(),
                    allowedContent:
                        'h1 h2 h3 b i u strike sup sub strong em;' +
                        'blockquote p pre table' +
                        '(text-align-left,text-align-center,text-align-right,text-align-justify){text-align};' +
                        'a[!href];' +
                        'img[!src,alt]{width,height,float};' +
                        'tr th td caption;' +
                        'li; ol[start]{list-style-type};' +
                        'ul{list-style};' +
                        'span[data-line-number,contenteditable]{color,background-color}(os-line-number,line-number-*);' +
                        'br(os-line-break);',

                    // there seems to be an error in CKeditor that parses spaces in extraPlugins as part of the plugin name.
                    extraPlugins: 'colorbutton,find,sourcedialog,justify,showblocks',
                    removePlugins: 'wsc,scayt,a11yhelp,filebrowser,sourcearea,liststyle,tabletools,contextmenu',
                    removeButtons: 'Scayt,Anchor,Styles,HorizontalRule',
                    toolbarGroups: [
                        { name: 'clipboard', groups: [ 'clipboard', 'undo' ] },
                        { name: 'editing', groups: [ 'find', 'selection', 'spellchecker', 'editing' ] },
                        { name: 'links', groups: [ 'links' ] },
                        { name: 'insert', groups: [ 'insert' ] },
                        { name: 'tools', groups: [ 'tools' ] },
                        { name: 'document', groups: [ 'mode' ] },
                        '/',
                        { name: 'styles', groups: [ 'styles' ] },
                        { name: 'basicstyles', groups: [ 'basicstyles', 'cleanup' ] },
                        { name: 'colors', groups: [ 'colors' ] },
                        { name: 'paragraph', groups: [ 'list', 'indent' ] },
                        { name: 'align'},
                        { name: 'paragraph', groups: [ 'blocks' ] }
                    ]
                };
            }
        };
    }
])

/* Model for a projector.
 *
 * At the moment we use only one projector, so there will be only one object
 * in this model. It has the id 1. For later releases there will be multiple
 * projector objects.
 *
 * This model uses onConfilict: 'replace' instead of 'merge'. This is necessary
 * because the keys of the projector objects can change and old keys have to
 * be removed. See http://www.js-data.io/docs/dsdefaults#onconflict for
 * more information.
 */
.factory('Projector', [
    'DS',
    '$http',
    'EditForm',
    'Config',
    function(DS, $http, EditForm, Config) {
        return DS.defineResource({
            name: 'core/projector',
            onConflict: 'replace',
            relations: {
                hasMany: {
                    'core/projection-default': {
                        localField: 'projectiondefaults',
                        foreignKey: 'projector_id',
                    }
                },
            },
            methods: {
                controlProjector: function(action, direction) {
                    $http.post('/rest/core/projector/' + this.id + '/control_view/',
                            {"action": action, "direction": direction}
                    );
                },
                getFormOrStateForCurrentSlide: function () {
                    var return_dict;
                    angular.forEach(this.elements, function(value, key) {
                        if (value.name == 'agenda/list-of-speakers') {
                            return_dict = {
                                state: 'agenda.item.detail',
                                id: value.id,
                            };
                        } else if (
                            // TODO:
                            // Find generic solution for whitelist in getFormOrStateForCurrentSlide
                            // see https://github.com/OpenSlides/OpenSlides/issues/3130
                            value.name === 'topics/topic' ||
                            value.name === 'motions/motion' ||
                            value.name === 'motions/motion-block' ||
                            value.name === 'assignments/assignment' ||
                            value.name === 'mediafiles/mediafile' ||
                            value.name === 'users/user') {
                                return_dict = {
                                    form: EditForm.fromCollectionString(value.name),
                                    id: value.id,
                                };
                        }
                    });
                    return return_dict;
                },
                toggleBlank: function () {
                    $http.post('/rest/core/projector/' + this.id + '/control_blank/',
                        !this.blank
                    );
                },
                toggleBroadcast: function () {
                    $http.post('/rest/core/projector/' + this.id + '/broadcast/');
                }
            },
        });
    }
])

/* Model for all projection defaults */
.factory('ProjectionDefault', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/projection-default',
            relations: {
                belongsTo: {
                    'core/projector': {
                        localField: 'projector',
                        localKey: 'projector_id',
                    }
                }
            }
        });
    }
])

/* Model for ProjectorMessages */
.factory('ProjectorMessage', [
    'DS',
    'jsDataModel',
    'gettext',
    '$http',
    'Projector',
    function(DS, jsDataModel, gettext, $http, Projector) {
        var name = 'core/projector-message';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Message'),
            verbosenamePlural: gettext('Messages'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                // Override the BaseModel.project function
                project: function(projectorId) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var isProjectedIds = this.isProjected();
                    var self = this;
                    var predicate = function (element) {
                        return element.name === name && element.id === self.id;
                    };
                    _.forEach(isProjectedIds, function (id) {
                        var uuid = _.findKey(Projector.get(id).elements, predicate);
                        $http.post('/rest/core/projector/' + id + '/deactivate_elements/', [uuid]);
                    });
                    // if it was the same projector before, just delete it but not show again
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        // Now check whether other messages are already projected and delete them
                        var elements = Projector.get(projectorId).elements;
                        _.forEach(elements, function (element, uuid) {
                            if (element.name === name) {
                                $http.post('/rest/core/projector/' + projectorId + '/deactivate_elements/', [uuid]);
                            }
                        });
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                            [{name: name, id: self.id, stable: true}]
                        );
                    }
                },
            }
        });
    }
])

/* Model for Countdowns */
.factory('Countdown', [
    'DS',
    'jsDataModel',
    'gettext',
    '$rootScope',
    '$http',
    'Projector',
    function(DS, jsDataModel, gettext, $rootScope, $http, Projector) {
        var name = 'core/countdown';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Countdown'),
            verbosenamePlural: gettext('Countdowns'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                start: function () {
                    // calculate end point of countdown (in seconds!)
                    var endTimestamp = Date.now() / 1000 - $rootScope.serverOffset + this.countdown_time;
                    this.running = true;
                    this.countdown_time = endTimestamp;
                    DS.save(name, this.id);
                },
                stop: function () {
                    // calculate rest duration of countdown (in seconds!)
                    var newDuration = Math.floor( this.countdown_time - Date.now() / 1000 + $rootScope.serverOffset );
                    this.running = false;
                    this.countdown_time = newDuration;
                    DS.save(name, this.id);
                },
                reset: function () {
                    this.running = false;
                    this.countdown_time = this.default_time;
                    DS.save(name, this.id);
                },
                // Override the BaseModel.project function
                project: function(projectorId) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var isProjectedIds = this.isProjected();
                    var self = this;
                    var predicate = function (element) {
                        return element.name == name && element.id == self.id;
                    };
                    _.forEach(isProjectedIds, function (id) {
                        var uuid = _.findKey(Projector.get(id).elements, predicate);
                        $http.post('/rest/core/projector/' + id + '/deactivate_elements/', [uuid]);
                    });
                    // if it was the same projector before, just delete it but not show again
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                            [{name: name, id: self.id, stable: true}]
                        );
                    }
                },
            },
        });
    }
])

/* Two functions to convert between time duration in seconds <-> human readable time span.
 * E.g. 90 sec <-> 1:30 (min), 3661 sec <-> 1:01:01 (h)
 *
 * secondsToHumanTime: Expects seconds and give [h*:]mm[:ss]. The minutes part is always given, the hours
 *      and minutes could be controlled. The default are forced seconds and hours just if it is not 0.
 *      - seconds ('enabled', 'auto', 'disabled'): Whether to show seconds (Default 'enabled')
 *      - hours ('enabled', 'auto', 'disabled'): Whether to show hours (Default 'auto')
 *
 * humanTimeToSeconds: Expects [h*:]m*[:s*] with each part could have a variable length. The parsed time is
 *      in seconds. Minutes have to be given and hours and seconds are optional. One have to set 'seconds' or
 *      'hours' to true toparse these.
 *
 * params could be an object with the given settings, e.g. {ignoreHours: true}
 */
.factory('HumanTimeConverter', [
    function () {
        return {
            secondsToHumanTime: function (seconds, params) {
                if (!params) {
                    params = {seconds: 'enabled', hours: 'auto'};
                }
                if (!params.seconds) {
                    params.seconds = 'enabled';

                }
                if (!params.hours) {
                    params.hours = 'auto';
                }
                var time;
                // floor returns the largest integer of the absolut value of seconds
                var total = Math.floor(Math.abs(seconds));
                var h = Math.floor(total / 3600);
                var m = Math.floor(total % 3600 / 60);
                var s = Math.floor(total % 60);
                // Add leading "0" for double digit values
                time = ('0'+m).slice(-2); //minutes
                if ((params.seconds == 'auto' && s > 0) || params.seconds == 'enabled') {
                    s = ('0'+s).slice(-2);
                    time =  time + ':' + s;
                }
                if ((params.hours == 'auto' && h > 0) || params.hours == 'enabled') {
                    time = h + ':' + time;
                }
                if (seconds < 0) {
                    time = '-'+time;
                }
                return time;
            },
            humanTimeToSeconds: function (data, params) {
                if (!params) {
                    params = {seconds: false, hours: false};
                }
                var minLength = 1;
                if (params.seconds) {
                    minLength++;
                }
                if (params.hours){
                    minLength++;
                }

                var negative = data.charAt(0) == '-';
                var time = data.split(':');
                data = 0;
                if (time.length >= minLength) {
                    for (var i = 0; i < minLength; i++) {
                        data = data*60;
                        if (!isNaN(+time[i])) {
                            data += (+time[i]);
                        }
                    }
                    if (!params.seconds) { // the last field was minutes (e.g. h:mm)
                        data *= 60;
                    }
                    if (negative) {
                        data = -data;
                    }
                }
                return data;
            },
        };
    }
])

/* Converts a snake-case string to camelCase. Example:
 * 'motion-block-config' -> 'motionBlockConfig' */
.factory('CamelCase', [
    function () {
        return function (str) {
            return str.replace(/-([a-z])/g, function (match) {
                return match[1].toUpperCase();
            });
        };
    }
])

/* Return the specific EditForm for a given model. */
.factory('EditForm', [
    '$injector',
    'CamelCase',
    function ($injector, CamelCase) {
        return {
            fromCollectionString: function (collection) {
                var modelName = CamelCase(collection).split('/')[1];
                // Convert modelModel to ModelModelForm
                var formName = modelName.charAt(0).toUpperCase() + modelName.slice(1) + 'Form';
                return $injector.get(formName);
            },
        };
    }
])

/* Converts number of seconds into string "h:mm:ss" or "mm:ss" */
.filter('osSecondsToTime', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return function (seconds) {
            return HumanTimeConverter.secondsToHumanTime(seconds);
        };
    }
])

/* Converts number of minutes into string "h:mm" or "hh:mm" */
.filter('osMinutesToTime', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return function (minutes) {
            return HumanTimeConverter.secondsToHumanTime(minutes*60,
                { seconds: 'disabled',
                    hours: 'enabled' }
            );
        };
    }
])

// mark HTML as "trusted"
.filter('trusted', [
    '$sce',
    function ($sce) {
        return function(text) {
            return $sce.trustAsHtml(text);
        };
    }
])

// filters the requesting object (id=selfid) from a list of input objects
.filter('notself', function() {
    return function(input, selfid) {
        var result;
        if (selfid) {
            result = [];
            for (var key in input){
                var obj = input[key];
                if (selfid != obj.id) {
                    result.push(obj);
                }
            }
        } else {
            result = input;
        }
        return result;
    };
})

// Make sure that the DS factories are loaded by making them a dependency
.run([
    'ChatMessage',
    'Config',
    'Countdown',
    'ProjectorMessage',
    'Projector',
    'ProjectionDefault',
    'Tag',
    function (ChatMessage, Config, Countdown, ProjectorMessage, Projector, ProjectionDefault, Tag) {}
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.csv', [])

.factory('CsvDownload', [
    'Config',
    'FileSaver',
    function (Config, FileSaver) {
        var utf8_BOM = decodeURIComponent('%EF%BB%BF');
        return function (contentRows, filename) {
            var separator = Config.get('general_csv_separator').value;
            var rows = _.map(contentRows, function (row) {
                return row.join(separator);
            });
            var blob = new Blob([utf8_BOM + rows.join('\n')]);
            FileSaver.saveAs(blob, filename);
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.pdf', [])

/*
 * General layout functions for building PDFs with pdfmake.
 */
.factory('PDFLayout', [
    function() {
        var PDFLayout = {};
        var BallotCircleDimensions = {
            yDistance: 6,
            size: 8
        };

        // page title
        PDFLayout.createTitle = function(title) {
            return {
                text: title,
                style: "title"
            };
        };

        // page subtitle
        PDFLayout.createSubtitle = function(subtitle) {
            return {
                text: subtitle.join('\n'),
                style: "subtitle"
            };
        };

        // pagebreak
        PDFLayout.addPageBreak = function() {
            return [
                {
                    text: '',
                    pageBreak: 'after'
                }
            ];
        };

        // table row style
        PDFLayout.flipTableRowStyle = function(currentTableSize) {
            if (currentTableSize % 2 === 0) {
                return "tableEven";
            } else {
                return "tableOdd";
            }
        };

        // draws a circle
        PDFLayout.drawCircle = function(y, size) {
            return [
                {
                    type: 'ellipse',
                    x: 0,
                    y: y,
                    lineColor: 'black',
                    r1: size,
                    r2: size
                }
            ];
        };

        // returns an entry in the ballot with a circle to draw into
        PDFLayout.createBallotEntry = function(decision) {
            return {
                margin: [40+BallotCircleDimensions.size, 10, 0, 0],
                columns: [
                    {
                        width: 15,
                        canvas: PDFLayout.drawCircle(BallotCircleDimensions.yDistance,
                                BallotCircleDimensions.size)
                    },
                    {
                        width: "auto",
                        text: decision
                    }
                ],
            };
        };

        // crop marks for ballot papers
        PDFLayout.getBallotLayoutLines = function() {
            return '{{ballot-placeholder-to-insert-functions-here}}';
        };

        return PDFLayout;
    }
])


.factory('HTMLValidizer', function() {
    var HTMLValidizer = {};

    //checks if str is valid HTML. Returns valid HTML if not,
    //return emptystring if empty
    HTMLValidizer.validize = function(str) {
        if (str) {
            var a = document.createElement('div');
            a.innerHTML = str;
            angular.forEach(a.childNodes, function (child) {
                if (child.nodeType == 1) {
                    return str;
                }
            });
            return "<p>" + str + "</p>";
        } else {
            return ""; //needed for blank "reaons" field
        }
    };
    return HTMLValidizer;
})


.factory('PdfMakeDocumentProvider', [
    'Config',
    'PDFLayout',
    function(Config, PDFLayout) {
        /**
         * Provides the global document
         * @constructor
         * @param {object} contentProvider - Object with on method `getContent`, which
         * returns an array for content
         */
        var createInstance = function(contentProvider) {
            // PDF header
            var getHeader = function() {
                var columns = [];

                // add here your custom logo (which has to be added to a custom vfs_fonts.js)
                // see https://github.com/pdfmake/pdfmake/wiki/Custom-Fonts---client-side
                /*
                columns.push({
                    image: 'logo.png',
                    fit: [180,40]
                });*/

                var line1 = [
                    Config.translate(Config.get('general_event_name').value),
                    Config.translate(Config.get('general_event_description').value)
                ].filter(Boolean).join(' – ');
                var line2 = [
                    Config.get('general_event_location').value,
                    Config.get('general_event_date').value
                ].filter(Boolean).join(', ');
                var text = [line1, line2].join('\n');
                columns.push({
                    text: text,
                    fontSize:10,
                    width: '100%'
                });
                return {
                    color: '#555',
                    fontSize: 9,
                    margin: [75, 30, 75, 10], // [left, top, right, bottom]
                    columns: columns,
                    columnGap: 10
                };
            };


            // PDF footer
            // Used placeholder for currentPage and pageCount which
            // are replaced by dynamic footer function in pdf-worker.js.
            var getFooter = function() {
                return {
                    alignment: 'right',
                    margin: [0, 15, 75, 0],
                    fontSize: 9,
                    color: '#555',
                    text: '{{currentPage}} / {{pageCount}}'
                };
            };
            // Generates the document(definition) for pdfMake
            var getDocument = function(noFooter) {
                var content = contentProvider.getContent();
                return {
                    pageSize: 'A4',
                    pageMargins: [75, 90, 75, 75],
                    defaultStyle: {
                        font: 'PdfFont',
                        fontSize: 10
                    },
                    header: getHeader(),
                    footerTpl: noFooter ? '' : getFooter(),
                    content: content,
                    styles: {
                        title: {
                            fontSize: 18,
                            margin: [0,0,0,20],
                            bold: true
                        },
                        subtitle: {
                            fontSize: 9,
                            margin: [0,-20,0,20],
                            color: 'grey'
                        },
                        preamble: {
                            fontSize: 10,
                            margin: [0,0,0,10],
                        },
                        userDataTitle: {
                            fontSize: 26,
                            margin: [0,0,0,0],
                            bold: true
                        },
                        textItem: {
                            fontSize: 11,
                            margin: [0,7]
                        },
                        heading2: {
                            fontSize: 14,
                            margin: [0,0,0,10],
                            bold: true
                        },
                        heading3: {
                            fontSize: 12,
                            margin: [0,10,0,0],
                            bold: true
                        },
                        userDataHeading: {
                            fontSize: 14,
                            margin: [0,10],
                            bold: true
                        },
                        userDataTopic: {
                            fontSize: 12,
                            margin: [0,5]
                        },
                        userDataValue: {
                            fontSize: 12,
                            margin: [15,5]
                        },
                        tableofcontent: {
                            fontSize: 12,
                            margin: [0,3]
                        },
                        listParent: {
                            fontSize: 12,
                            margin: [0,5]
                        },
                        listChild: {
                            fontSize: 10,
                            margin: [0,5]
                        },
                        tableHeader: {
                            bold: true,
                            fillColor: 'white'
                        },
                        tableEven: {
                            fillColor: 'white'
                        },
                        tableOdd: {
                            fillColor: '#eee'
                        },
                        tableConclude: {
                            fillColor: '#ddd',
                            bold: true
                        },
                        grey: {
                            fillColor: '#ddd',
                        },
                        lightgrey: {
                            fillColor: '#aaa',
                        },
                        bold: {
                            bold: true,
                        },
                        small: {
                            fontSize: 8,
                        }
                    }
                };
            };

            return {
                getDocument: getDocument
            };
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('PdfMakeBallotPaperProvider', [
    'PDFLayout',
    function(PDFLayout) {
        /**
         * Provides the global Document
         * @constructor
         * @param {object} contentProvider - Object with on method `getContent`, which returns an array for content
         */
        var createInstance = function(contentProvider) {
            /**
             * Generates the document(definition) for pdfMake
             * @function
             */
            var getDocument = function() {
                var content = contentProvider.getContent();
                return {
                    pageSize: 'A4',
                    pageMargins: [0, 0, 0, 0],
                    defaultStyle: {
                        font: 'PdfFont',
                        fontSize: 10
                    },
                    content: content,
                    styles: {
                        title: {
                            fontSize: 14,
                            bold: true,
                            margin: [30, 30, 0, 0]
                        },
                        description: {
                            fontSize: 11,
                            margin: [30, 0, 0, 0]
                        }
                    }
                };
            };
            return {
                getDocument: getDocument
            };
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('PdfMakeConverter', [
    'HTMLValidizer',
    function(HTMLValidizer) {
        /**
         * Converter component for HTML->JSON for pdfMake
         * @constructor
         * @param {object} images   - Key-Value structure representing image.src/BASE64 of images
         */
        var createInstance = function(images) {
            var slice = Function.prototype.call.bind([].slice),
                map = Function.prototype.call.bind([].map),

                DIFF_MODE_NORMAL = 0,
                DIFF_MODE_INSERT = 1,
                DIFF_MODE_DELETE = 2,

                /**
                 * Convertes HTML for use with pdfMake
                 * @function
                 * @param {object} html - html
                 * @param {string} lineNumberMode - [inline, outside, none]
                 */
                convertHTML = function(html, lineNumberMode) {
                    var elementStyles = {
                            "b": ["font-weight:bold"],
                            "strong": ["font-weight:bold"],
                            "u": ["text-decoration:underline"],
                            "em": ["font-style:italic"],
                            "i": ["font-style:italic"],
                            "h1": ["font-size:14", "font-weight:bold"],
                            "h2": ["font-size:12", "font-weight:bold"],
                            "h3": ["font-size:10", "font-weight:bold"],
                            "h4": ["font-size:10", "font-style:italic"],
                            "h5": ["font-size:10"],
                            "h6": ["font-size:10"],
                            "a": ["color:blue", "text-decoration:underline"],
                            "strike": ["text-decoration:line-through"],
                            "del": ["color:red", "text-decoration:line-through"],
                            "ins": ["color:green", "text-decoration:underline"]
                        },
                        classStyles = {
                            "delete": ["color:red", "text-decoration:line-through"],
                            "insert": ["color:green", "text-decoration:underline"]
                        },
                        /**
                         * Removes all line number nodes (not line-breaks)
                         * and returns an array containing the reoved numbers (as integer, not as node)
                         *
                         * @function
                         * @param {object} element
                         */
                        extractLineNumbers = function(element) {
                            var foundLineNumbers = [];
                            if (element.nodeName == 'SPAN' && element.getAttribute('class') && element.getAttribute('class').indexOf('os-line-number') > -1) {
                                foundLineNumbers.push(element.getAttribute('data-line-number'));
                                element.parentNode.removeChild(element);
                            } else {
                                var children = element.childNodes,
                                    childrenLength = children.length;
                                for (var i = 0; i < children.length; i++) {
                                    foundLineNumbers = _.union(foundLineNumbers, extractLineNumbers(children[i]));
                                    if (children.length < childrenLength) {
                                        i -= (childrenLength - children.length);
                                        childrenLength = children.length;
                                    }
                                }
                            }
                            return foundLineNumbers;
                        },
                        /**
                         * Parses Children of the current paragraph
                         * @function
                         * @param {object} converted  -
                         * @param {object} element   -
                         * @param {object} currentParagraph -
                         * @param {object} styles -
                         * @param {number} diff_mode
                         */
                        parseChildren = function(converted, element, currentParagraph, styles, diff_mode) {
                            var elements = [];
                            var children = element.childNodes;
                            if (children.length !== 0) {
                                _.forEach(children, function(child) {
                                    currentParagraph = ParseElement(elements, child, currentParagraph, styles, diff_mode);
                                });
                            }
                            if (elements.length !== 0) {
                                _.forEach(elements, function(el) {
                                    converted.push(el);
                                });
                            }
                            return currentParagraph;
                        },
                        /**
                         * Extracts the style from an object
                         * @function
                         * @param {object} o       - the current object
                         * @param {object} styles  - an array with styles
                         */
                        ComputeStyle = function(o, styles) {
                            styles.forEach(function(singleStyle) {
                                var styleDefinition = singleStyle.trim().toLowerCase().split(":");
                                var style = styleDefinition[0];
                                var value = styleDefinition[1];
                                if (styleDefinition.length == 2) {
                                    switch (style) {
                                        case "padding-left":
                                            o.margin = [parseInt(value), 0, 0, 0];
                                            break;
                                        case "font-size":
                                            o.fontSize = parseInt(value);
                                            break;
                                        case "text-align":
                                            switch (value) {
                                                case "right":
                                                case "center":
                                                case "justify":
                                                    o.alignment = value;
                                                    break;
                                            }
                                            break;
                                        case "font-weight":
                                            switch (value) {
                                                case "bold":
                                                    o.bold = true;
                                                    break;
                                            }
                                            break;
                                        case "text-decoration":
                                            switch (value) {
                                                case "underline":
                                                    o.decoration = "underline";
                                                    break;
                                                case "line-through":
                                                    o.decoration = "lineThrough";
                                                    break;
                                            }
                                            break;
                                        case "font-style":
                                            switch (value) {
                                                case "italic":
                                                    o.italics = true;
                                                    break;
                                            }
                                            break;
                                        case "color":
                                            o.color = value;
                                            break;
                                        case "background-color":
                                            o.background = value;
                                            break;
                                    }
                                }
                            });
                        },
                        /**
                         * Parses a single HTML element
                         * @function
                         * @param {object} alreadyConverted  -
                         * @param {object} element   -
                         * @param {object} currentParagraph -
                         * @param {object} styles -
                         * @param {number} diff_mode
                         */
                        ParseElement = function(alreadyConverted, element, currentParagraph, styles, diff_mode) {
                            styles = styles || [];
                            var classes = [];
                            if (element.getAttribute) {
                                styles = [];
                                var nodeStyle = element.getAttribute("style");
                                if (nodeStyle) {
                                    nodeStyle.split(";").forEach(function(nodeStyle) {
                                        var tmp = nodeStyle.replace(/\s/g, '');
                                        styles.push(tmp);
                                    });
                                }
                                var nodeClass = element.getAttribute("class");
                                if (nodeClass) {
                                    classes = nodeClass.toLowerCase().split(" ");
                                    classes.forEach(function(nodeClass) {
                                        if (typeof(classStyles[nodeClass]) != 'undefined') {
                                            classStyles[nodeClass].forEach(function(style) {
                                                styles.push(style);
                                            });
                                        }
                                        if (nodeClass == 'insert') {
                                            diff_mode = DIFF_MODE_INSERT;
                                        }
                                        if (nodeClass == 'delete') {
                                            diff_mode = DIFF_MODE_DELETE;
                                        }
                                    });
                                }
                            }
                            var nodeName = element.nodeName.toLowerCase();
                            switch (nodeName) {
                                case "h1":
                                case "h2":
                                case "h3":
                                case "h4":
                                case "h5":
                                case "h6":
                                    // Special case quick fix to handle the dirty HTML format*/
                                    // see following issue: https://github.com/OpenSlides/OpenSlides/issues/3025
                                    if (lineNumberMode === "outside") {
                                        var HeaderOutsideLineNumber = {
                                            width: 20,
                                            text: element.childNodes[0].getAttribute("data-line-number"),
                                            color: "gray",
                                            fontSize: 8,
                                            margin: [0, 2, 0, 0]
                                        };
                                        var HeaderOutsideLineNumberText = {
                                            text: element.childNodes[1].textContent,
                                        };
                                        ComputeStyle(HeaderOutsideLineNumberText, elementStyles[nodeName]);
                                        var HeaderOutsideLineNumberColumns = {
                                            columns: [
                                                HeaderOutsideLineNumber,
                                                HeaderOutsideLineNumberText
                                            ]
                                        };
                                        alreadyConverted.push(HeaderOutsideLineNumberColumns);
                                    } else {
                                        currentParagraph = create("text");
                                        currentParagraph.marginBottom = 4;
                                        currentParagraph.marginTop = 10;
                                        currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles.concat(elementStyles[nodeName]), diff_mode);
                                        alreadyConverted.push(currentParagraph);
                                    }
                                    break;
                                case "a":
                                case "b":
                                case "strong":
                                case "u":
                                case "em":
                                case "i":
                                case "ins":
                                case "del":
                                case "strike":
                                    currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles.concat(elementStyles[nodeName]), diff_mode);
                                    break;
                                case "table":
                                    var t = create("table", {
                                        widths: [],
                                        body: []
                                    });
                                    var border = element.getAttribute("border");
                                    var isBorder = false;
                                    if (border)
                                        if (parseInt(border) == 1) isBorder = true;
                                    if (!isBorder) t.layout = 'noBorders';
                                    currentParagraph = parseChildren(t.table.body, element, currentParagraph, styles, diff_mode);
                                    var widths = element.getAttribute("widths");
                                    if (!widths) {
                                        if (t.table.body.length !== 0) {
                                            if (t.table.body[0].length !== 0)
                                                for (var k = 0; k < t.table.body[0].length; k++)
                                                    t.table.widths.push("*");
                                        }
                                    } else {
                                        var w = widths.split(",");
                                        for (var ko = 0; ko < w.length; ko++) t.table.widths.push(w[ko]);
                                    }
                                    alreadyConverted.push(t);
                                    break;
                                case "tbody":
                                    currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles, diff_mode);
                                    break;
                                case "tr":
                                    var row = [];
                                    currentParagraph = parseChildren(row, element, currentParagraph, styles, diff_mode);
                                    alreadyConverted.push(row);
                                    break;
                                case "td":
                                    currentParagraph = create("text");
                                    var st = create("stack");
                                    st.stack.push(currentParagraph);
                                    var rspan = element.getAttribute("rowspan");
                                    if (rspan)
                                        st.rowSpan = parseInt(rspan);
                                    var cspan = element.getAttribute("colspan");
                                    if (cspan)
                                        st.colSpan = parseInt(cspan);
                                    currentParagraph = parseChildren(st.stack, element, currentParagraph, styles, diff_mode);
                                    alreadyConverted.push(st);
                                    break;
                                case "span":
                                    if (element.getAttribute("data-line-number")) {
                                        if (lineNumberMode == "inline") {
                                            if (diff_mode != DIFF_MODE_INSERT) {
                                                var lineNumberInline = element.getAttribute("data-line-number"),
                                                    lineNumberObjInline = {
                                                        text: lineNumberInline,
                                                        color: "gray",
                                                        fontSize: 5
                                                    };
                                                currentParagraph.text.push(lineNumberObjInline);
                                            }
                                        } else if (lineNumberMode == "outside") {
                                            var lineNumberOutline;
                                            if (diff_mode == DIFF_MODE_INSERT) {
                                                lineNumberOutline = "";
                                            } else {
                                                lineNumberOutline = element.getAttribute("data-line-number");
                                            }
                                            var lineNumberObject = {
                                                    width: 20,
                                                    text: lineNumberOutline,
                                                    color: "gray",
                                                    fontSize: 8,
                                                    margin: [0, 2, 0, 0]
                                            },
                                                col = {
                                                    columns: [
                                                        lineNumberObject,
                                                    ]
                                            };
                                            currentParagraph = create("text");
                                            currentParagraph.lineHeight = 1.25;
                                            col.columns.push(currentParagraph);
                                            alreadyConverted.push(col);
                                        }
                                    }
                                    else {
                                        currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles, diff_mode);
                                    }
                                    break;
                                case "br":
                                    //in case of inline-line-numbers and the os-line-break class ignore the break
                                    if ((lineNumberMode == "inline" && element.getAttribute("class") == "os-line-break") ||
                                        (lineNumberMode == "outside" && element.getAttribute("class") == "os-line-break" && element.parentNode.tagName == "INS") ||
                                        (lineNumberMode == "outside" && element.getAttribute("class") == "os-line-break" && element.parentNode.getAttribute("class") == "merge-before")) {
                                        break;
                                    } else {
                                        currentParagraph = create("text");
                                        currentParagraph.lineHeight = 1.25;
                                        alreadyConverted.push(currentParagraph);
                                    }
                                    break;
                                case "li":
                                case "div":
                                    currentParagraph = create("text");
                                    currentParagraph.lineHeight = 1.25;
                                    var stackDiv = create("stack");
                                    stackDiv.stack.push(currentParagraph);
                                    ComputeStyle(stackDiv, styles);
                                    currentParagraph = parseChildren(stackDiv.stack, element, currentParagraph, [], diff_mode);
                                    alreadyConverted.push(stackDiv);
                                    break;
                                case "p":
                                    var pObjectToPush; //determine what to push later
                                    currentParagraph = create("text");
                                    if (classes.indexOf("merge-before") > -1) {
                                        currentParagraph.marginTop = 0;
                                    } else {
                                        currentParagraph.marginTop = 8;
                                    }
                                    currentParagraph.lineHeight = 1.25;
                                    var stackP = create("stack");
                                    stackP.stack.push(currentParagraph);
                                    ComputeStyle(stackP, styles);
                                    currentParagraph = parseChildren(stackP.stack, element, currentParagraph, [], diff_mode);
                                    pObjectToPush = stackP; //usually we want to push stackP
                                    if (lineNumberMode === "outside") {
                                        if (element.childNodes.length > 0) { //if we hit = 0, the code would fail
                                            // add empty line number column for inline diff or pragraph diff mode
                                            if (element.childNodes[0].tagName === "INS" ||
                                                element.getAttribute("class") === "insert") {
                                                var pLineNumberPlaceholder = {
                                                    width: 20,
                                                    text: "",
                                                    fontSize: 8,
                                                    margin: [0, 2, 0, 0]
                                                };
                                                var pLineNumberPlaceholderCol = {
                                                    columns: [
                                                        pLineNumberPlaceholder,
                                                        stackP
                                                    ]
                                                };
                                                pObjectToPush = pLineNumberPlaceholderCol; //overwrite the object to push
                                            }
                                        }
                                    }
                                    alreadyConverted.push(pObjectToPush);
                                    break;
                                case "img":
                                    // TODO: need a proper way to calculate the space
                                    // left on the page.
                                    // This requires further information
                                    // A4 in 72dpi: 595px x 842px
                                    var maxResolution = {
                                        width: 435,
                                        height: 830
                                    },
                                        width = parseInt(element.getAttribute("width")),
                                        height = parseInt(element.getAttribute("height"));

                                    if (width > maxResolution.width) {
                                        var scaleByWidth = maxResolution.width/width;
                                        width *= scaleByWidth;
                                        height *= scaleByWidth;
                                    }
                                    if (height > maxResolution.height) {
                                        var scaleByHeight = maxResolution.height/height;
                                        width *= scaleByHeight;
                                        height *= scaleByHeight;
                                    }
                                    alreadyConverted.push({
                                        image: BaseMap[element.getAttribute("src")],
                                        width: width,
                                        height: height
                                    });
                                    break;
                                case "ul":
                                case "ol":
                                    var list = create(nodeName);
                                    if (lineNumberMode == "outside") {
                                        var lines = extractLineNumbers(element);
                                        currentParagraph = parseChildren(list[nodeName], element, currentParagraph, styles, diff_mode);
                                        if (lines.length > 0) {
                                            var listCol = {
                                                    columns: [{
                                                        width: 20,
                                                        stack: []
                                                    }]
                                                };
                                            _.forEach(lines, function(line) {
                                                listCol.columns[0].stack.push({
                                                    width: 20,
                                                    text: line,
                                                    color: "gray",
                                                    fontSize: 8,
                                                    lineHeight: 1.25,
                                                    margin: [0, 2.85, 0, 0]
                                                });
                                            });
                                            listCol.columns.push(list);
                                            listCol.margin = [0,10,0,0];
                                            alreadyConverted.push(listCol);
                                        } else {
                                            alreadyConverted.push(list);
                                        }
                                    } else {
                                        currentParagraph = parseChildren(list[nodeName], element, currentParagraph, styles, diff_mode);
                                        alreadyConverted.push(list);
                                    }
                                    break;
                                default:
                                    var defaultText = create("text", element.textContent.replace(/\n/g, ""));
                                    ComputeStyle(defaultText, styles);
                                    if (!currentParagraph) {
                                        currentParagraph = {};
                                        currentParagraph.text = [];
                                    }
                                    currentParagraph.text.push(defaultText);
                                    break;
                            }
                            return currentParagraph;
                        },
                        /**
                         * Parses HTML
                         * @function
                         * @param {string} converted      -
                         * @param {object} htmlText   -
                         */
                        ParseHtml = function(converted, htmlText) {
                            var html = HTMLValidizer.validize(htmlText);
                            html = $(html.replace(/\t/g, "").replace(/\n/g, ""));
                            var emptyParagraph = create("text");
                            slice(html).forEach(function(element) {
                                ParseElement(converted, element, null, [], DIFF_MODE_NORMAL);
                            });
                        },
                        content = [];
                    ParseHtml(content, html);
                    return content;
                },
                BaseMap = images,
                /**
                 * Creates containerelements for pdfMake
                 * e.g create("text":"MyText") result in { text: "MyText" }
                 * or complex objects create("stack", [{text:"MyText"}, {text:"MyText2"}])
                 *for units / paragraphs of text
                 *
                 * @function
                 * @param {string} name      - name of the attribute holding content
                 * @param {object} content   - the actual content (maybe empty)
                 */
                create = function(name, content) {
                    var o = {};
                    content = content || [];
                    o[name] = content;
                    return o;
                };
            return {
                convertHTML: convertHTML,
                createElement: create
            };
        };
        return {
            createInstance: createInstance
        };
}])

.factory('PdfCreate', [
    '$timeout',
    'gettextCatalog',
    'FileSaver',
    'Messaging',
    function ($timeout, gettextCatalog, FileSaver, Messaging) {
        var filenameMessageMap = {};
        var b64toBlob = function(b64Data) {
            var byteCharacters = atob(b64Data);
            var byteNumbers = new Array(byteCharacters.length);
            for (var i = 0; i < byteCharacters.length; i++) {
                byteNumbers[i] = byteCharacters.charCodeAt(i);
            }
            var byteArray = new Uint8Array(byteNumbers);
            var blob = new Blob([byteArray]);
            return blob;
        };
        var stateChange = function (state, filename, error) {
            var text, timeout;
            switch (state) {
                case 'info':
                    text = '<i class="fa fa-spinner fa-pulse fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('Generating PDF file') + ' (' + filename + ') ...';
                    break;
                case 'success':
                    text = '<i class="fa fa-check fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('PDF successfully generated.');
                    timeout = 3000;
                    break;
                case 'error':
                    text = '<i class="fa fa-exclamation-triangle fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('Error while generating PDF file') +
                        ' (' + filename + '): <code>' + error + '</code>';
                    break;
            }
            $timeout(function () {
                filenameMessageMap[filename] = Messaging.createOrEditMessage(
                    filenameMessageMap[filename], text, state, {timeout: timeout});
            }, 1);
        };
        return {
            download: function (pdfDocument, filename) {
                stateChange('info', filename);
                var pdfWorker = new Worker('/static/js/workers/pdf-worker.js');

                pdfWorker.addEventListener('message', function (event) {
                    var blob = b64toBlob(event.data);
                    stateChange('success', filename);
                    FileSaver.saveAs(blob, filename);
                });
                pdfWorker.addEventListener('error', function (event) {
                    stateChange('error', filename, event.message);
                });
                pdfWorker.postMessage(JSON.stringify(pdfDocument));
            },
        };
    }
]);

}());

(function () {

'use strict';

// The core module for the OpenSlides projector
angular.module('OpenSlidesApp.core.projector', ['OpenSlidesApp.core'])

// Can be used to find out if the projector or the side is used
.constant('REALM', 'projector')

.run([
    'autoupdate',
    function (autoupdate) {
        autoupdate.newConnect();
    }
])

// Provider to register slides in a .config() statement.
.provider('slides', [
    function() {
        var slidesMap = {};

        this.registerSlide = function(name, config) {
            slidesMap[name] = config;
            return this;
        };

        this.$get = function($templateRequest, $q) {
            var self = this;
            return {
                getElements: function(projector) {
                    var elements = [];
                    var factory = this;
                    _.forEach(projector.elements, function(element) {
                        if (element.name in slidesMap) {
                            element.template = slidesMap[element.name].template;
                            elements.push(element);
                        } else {
                            console.error("Unknown slide: " + element.name);
                        }
                    });
                    return elements;
                }
            };
        };
    }
])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('core/clock', {
            template: 'static/templates/core/slide_clock.html',
        });

        slidesProvider.registerSlide('core/countdown', {
            template: 'static/templates/core/slide_countdown.html',
        });

        slidesProvider.registerSlide('core/projector-message', {
            template: 'static/templates/core/slide_message.html',
        });
    }
])

.controller('LanguageCtrl', [
    '$scope',
    'Languages',
    'Config',
    'ProjectorID',
    function ($scope, Languages, Config, ProjectorID) {
        // for the dynamic title
        $scope.projectorId = ProjectorID();

        $scope.$watch(function () {
            return Config.lastModified('projector_language');
        }, function () {
            var lang = Config.get('projector_language');
            if (!lang || lang.value == 'browser') {
                $scope.selectedLanguage = Languages.getBrowserLanguage();
            } else {
                $scope.selectedLanguage = lang.value;
            }
            Languages.setCurrentLanguage($scope.selectedLanguage);
        });
    }
])

// Projector Container Controller
.controller('ProjectorContainerCtrl', [
    '$scope',
    '$location',
    'gettext',
    'Projector',
    function($scope, $location, gettext, Projector) {
        $scope.error = '';

        // watch for changes in Projector
        $scope.$watch(function () {
            return Projector.lastModified($scope.projectorId);
        }, function () {
            var projector = Projector.get($scope.projectorId);
            if (projector) {
                $scope.error = '';
                $scope.projectorWidth = projector.width;
                $scope.projectorHeight = projector.height;
                $scope.recalculateIframe();
            } else {
                $scope.error = gettext('Can not open the projector.');
            }
        });

        // recalculate the actual Iframesize and scale
        $scope.recalculateIframe = function () {
            var scale_width = window.innerWidth / $scope.projectorWidth;
            var scale_height = window.innerHeight / $scope.projectorHeight;

            // Iframe has to be scaled down or saceUp is activated
            if (scale_width <= scale_height) {
                // width is the reference
                $scope.iframeWidth = window.innerWidth;
                $scope.scale = scale_width;
                $scope.iframeHeight = $scope.projectorHeight * scale_width;
            } else {
                // height is the reference
                $scope.iframeHeight = window.innerHeight;
                $scope.scale = scale_height;
                $scope.iframeWidth = $scope.projectorWidth * scale_height;
            }
        };

        // watch for changes in the windowsize
        $(window).on("resize.doResize", function () {
            $scope.$apply(function() {
                $scope.recalculateIframe();
            });
        });

        $scope.$on("$destroy",function (){
            $(window).off("resize.doResize");
        });
    }
])

.controller('ProjectorCtrl', [
    '$scope',
    '$location',
    '$timeout',
    'Projector',
    'slides',
    'Config',
    'ProjectorID',
    function($scope, $location, $timeout, Projector, slides, Config, ProjectorID) {
        var projectorId = ProjectorID();

        $scope.broadcast = 0;

        var setElements = function (projector) {
            $scope.elements = [];
            _.forEach(slides.getElements(projector), function(element) {
                if (!element.error) {
                    $scope.elements.push(element);
                } else {
                    console.error("Error for slide " + element.name + ": " + element.error);
                }
            });
        };

        // This function scrolls the projector smoothly. It scrolls is steps calling each
        // step with a little timeout.
        var STEPS = 5;
        $scope.scroll = 0;
        var setScroll = function (scroll) {
            scroll = -250 * scroll;
            if ($scope.scrollTimeout) {
                $timeout.cancel($scope.scrollTimeout);
            }
            var oldScroll = $scope.scroll;
            var diff = scroll - oldScroll;
            var count = 0;
            if (scroll !== oldScroll) {
                var scrollFunction = function () {
                    $scope.scroll += diff/STEPS;
                    count++;
                    if (count < STEPS) {
                        $scope.scrollTimeout = $timeout(scrollFunction, 1);
                    }
                };
                scrollFunction();
            }
        };

        $scope.$watch(function () {
            return Projector.lastModified(projectorId);
        }, function () {
            $scope.projector = Projector.get(projectorId);
            if ($scope.projector) {
                if ($scope.broadcast === 0) {
                    setElements($scope.projector);
                    $scope.blank = $scope.projector.blank;
                }
                setScroll($scope.projector.scroll);
            } else {
                // Blank projector on error
                $scope.elements = [];
                $scope.projector = {
                    scale: 0,
                    blank: true
                };
                setScroll(0);
            }
        });

        $scope.$watch(function () {
            return Config.lastModified('projector_broadcast');
        }, function () {
            var bc = Config.get('projector_broadcast');
            if (bc) {
                if ($scope.broadcast != bc.value) {
                    $scope.broadcast = bc.value;
                    if ($scope.broadcastDeregister) {
                        // revert to original $scope.projector
                        $scope.broadcastDeregister();
                        $scope.broadcastDeregister = null;
                        setElements($scope.projector);
                        $scope.blank = $scope.projector.blank;
                    }
                }
                if ($scope.broadcast > 0) {
                    // get elements and blank from broadcast projector
                    $scope.broadcastDeregister = $scope.$watch(function () {
                        return Projector.lastModified($scope.broadcast);
                    }, function () {
                        if ($scope.broadcast > 0) {
                            var broadcast_projector = Projector.get($scope.broadcast);
                            if (broadcast_projector) {
                                setElements(broadcast_projector);
                                $scope.blank = broadcast_projector.blank;
                            }
                        }
                    });
                }
            }
        });

        $scope.$on('$destroy', function() {
            if ($scope.broadcastDeregister) {
                $scope.broadcastDeregister();
                $scope.broadcastDeregister = null;
            }
        });
    }
])

.controller('SlideClockCtrl', [
    '$scope',
    '$interval',
    function($scope, $interval) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        $scope.servertime = ( Date.now() / 1000 - $scope.serverOffset ) * 1000;
        var interval = $interval(function () {
            $scope.servertime = ( Date.now() / 1000 - $scope.serverOffset ) * 1000;
        }, 30000); // Update the clock every 30 seconds

        $scope.$on('$destroy', function() {
            if (interval) {
                $interval.cancel(interval);
            }
        });
    }
])

.controller('SlideCountdownCtrl', [
    '$scope',
    '$interval',
    'Countdown',
    function($scope, $interval, Countdown) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        var interval;
        var calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( $scope.countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        $scope.$watch(function () {
            return Countdown.lastModified(id);
        }, function () {
            $scope.countdown = Countdown.get(id);
            if (interval) {
                $interval.cancel(interval);
            }
            if ($scope.countdown) {
                if ($scope.countdown.running) {
                    calculateCountdownTime($scope.countdown);
                    interval = $interval(function () { calculateCountdownTime($scope.countdown); }, 1000);
                } else {
                    $scope.countdown.seconds = $scope.countdown.countdown_time;
                }
            }
        });
        $scope.$on('$destroy', function() {
            // Cancel the interval if the controller is destroyed
            if (interval) {
                $interval.cancel(interval);
            }
        });
    }
])

.controller('SlideMessageCtrl', [
    '$scope',
    'ProjectorMessage',
    'Projector',
    'ProjectorID',
    'gettextCatalog',
    function($scope, ProjectorMessage, Projector, ProjectorID, gettextCatalog) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;

        if ($scope.element.identify) {
            var projector = Projector.get(ProjectorID());
            $scope.identifyMessage = gettextCatalog.getString('Projector') + ' ' + projector.id + ': ' + gettextCatalog.getString(projector.name);
        } else {
            $scope.message = ProjectorMessage.get(id);
            ProjectorMessage.bindOne(id, $scope, 'message');
        }
    }
]);

}());

(function () {

'use strict';

// The core module for the OpenSlides site
angular.module('OpenSlidesApp.core.site', [
    'OpenSlidesApp.core',
    'OpenSlidesApp.core.start',
    'OpenSlidesApp.core.csv',
    'OpenSlidesApp.poll.majority',
    'ui.router',
    'colorpicker.module',
    'formly',
    'formlyBootstrap',
    'localytics.directives',
    'ngDialog',
    'ngFileSaver',
    'ngMessages',
    'ngStorage',
    'ckeditor',
    'luegg.directives',
    'xeditable',
])

// Can be used to find out if the projector or the side is used
.constant('REALM', 'site')

.factory('DateTimePickerTranslation', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getButtons: function () {
                return {
                    show: true,
                    now: {
                        show: true,
                        text: gettextCatalog.getString('now')
                    },
                    today: {
                        show: true,
                        text: gettextCatalog.getString('today')
                    },
                    clear: {
                        show: true,
                        text: gettextCatalog.getString('clear')
                    },
                    date: {
                        show: true,
                        text: gettextCatalog.getString('date')
                    },
                    time: {
                        show: true,
                        text: gettextCatalog.getString('time')
                    },
                    close: {
                        show: true,
                        text: gettextCatalog.getString('close')
                    }
                };
            }
        };
    }

])

// Provider to register entries for the main menu.
.provider('mainMenu', [
    function() {
        var mainMenuList = [];
        var scope;

        this.register = function(config) {
            mainMenuList.push(config);
        };

        this.$get = ['operator', function(operator) {
            return {
                registerScope: function (scope) {
                    this.scope = scope;
                },
                updateMainMenu: function () {
                    this.scope.elements = this.getElements();
                },
                getElements: function() {
                    var elements = mainMenuList.filter(function (element) {
                        return typeof element.perm === "undefined" || operator.hasPerms(element.perm);
                    });

                    elements.sort(function (a, b) {
                        return a.weight - b.weight;
                    });
                    return elements;
                }
            };
        }];
    }
])

// Provider to register a searchable module/app.
.provider('Search', [
    function() {
        var searchModules = [];

        this.register = function(module) {
            searchModules.push(module);
        };

        this.$get = [
            function () {
                return {
                    getAll: function () {
                        return searchModules;
                    }
                };
            }
        ];
    }
])

.run([
    'editableOptions',
    'gettext',
    function (editableOptions, gettext) {
        editableOptions.theme = 'bs3';
        editableOptions.cancelButtonAriaLabel = gettext('Cancel');
        editableOptions.cancelButtonTitle = gettext('Cancel');
        editableOptions.clearButtonAriaLabel = gettext('Clear');
        editableOptions.clearButtonTitle = gettext('Clear');
        editableOptions.submitButtonAriaLabel = gettext('Submit');
        editableOptions.submitButtonTitle = gettext('Submit');
    }
])

// Set up the activeAppTitle for the title from the webpage
.run([
    '$rootScope',
    'gettextCatalog',
    'operator',
    function ($rootScope, gettextCatalog, operator) {
        $rootScope.activeAppTitle = '';
        $rootScope.$on('$stateChangeSuccess', function(event, toState) {
            if (toState.data) {
                $rootScope.activeAppTitle = toState.data.title || '';
                $rootScope.baseViewPermissionsGranted = toState.data.basePerm ?
                    operator.hasPerms(toState.data.basePerm) : true;
            } else {
                $rootScope.activeAppTitle = '';
                $rootScope.baseViewPermissionsGranted = true;
            }
        });
    }
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'home',
            'img_class': 'home',
            'title': gettext('Home'),
            'weight': 100,
            'perm': 'core.can_see_frontpage',
        });

        mainMenuProvider.register({
            'ui_sref': 'config',
            'img_class': 'cog',
            'title': gettext('Settings'),
            'weight': 1000,
            'perm': 'core.can_manage_config',
        });
    }
])

.config([
    '$urlRouterProvider',
    '$locationProvider',
    function($urlRouterProvider, $locationProvider) {
        // define fallback url and html5Mode
        $urlRouterProvider.otherwise('/');
        $locationProvider.html5Mode(true);
    }
])

.config([
    '$httpProvider',
    function($httpProvider) {
        // Combine the django csrf system with the angular csrf system
        $httpProvider.defaults.xsrfCookieName = 'OpenSlidesCsrfToken';
        $httpProvider.defaults.xsrfHeaderName = 'X-CSRFToken';
    }
])


.config([
    '$stateProvider',
    '$urlMatcherFactoryProvider',
    function($stateProvider, $urlMatcherFactoryProvider) {
        // Make the trailing slash optional
        $urlMatcherFactoryProvider.strictMode(false);

        // Use stateProvider.decorator to give default values to our states
        $stateProvider.decorator('views', function(state, parent) {
            var result = {},
                views = parent(state);

            if (state.abstract || state.data && state.data.extern) {
                return views;
            }

            angular.forEach(views, function(config, name) {
                // Sets additional default values for templateUrl
                var templateUrl,
                    controller,
                    defaultControllers = {
                        create: 'CreateCtrl',
                        update: 'UpdateCtrl',
                        list: 'ListCtrl',
                        detail: 'DetailCtrl',
                    };

                // Split up state name
                // example: "motions.motion.detail.update" -> ['motions', 'motion', 'detail', 'update']
                var patterns = state.name.split('.');

                // set app and module name from state
                // - appName: patterns[0] (e.g. "motions")
                // - moduleNames: patterns without first element (e.g. ["motion", "detail", "update"])
                var appName = '';
                var moduleName = '';
                var moduleNames = [];
                if (patterns.length > 0) {
                    appName = patterns[0];
                    moduleNames = patterns.slice(1);
                }
                if (moduleNames.length > 0) {
                    // convert from camcelcase to dash notation
                    // example: ["motionBlock", "detail"] -> ["motion-block", "detail"]
                    for (var i = 0; i < moduleNames.length; i++) {
                        moduleNames[i] =  moduleNames[i].replace(/([a-z\d])([A-Z])/g, '$1-$2').toLowerCase();
                    }

                    // use special templateUrl for create and update view
                    // example: ["motion", "detail", "update"] -> "motion-form"
                    if (_.last(moduleNames).match(/(create|update)/)) {
                        moduleName = '/' + moduleNames[0] + '-form';
                    } else {
                        // convert modelNames array to url string
                        // example: ["motion-block", "detail"] -> "motion-block-detail"
                        moduleName = '/' + moduleNames.join('-');
                    }
                }
                templateUrl = 'static/templates/' + appName + moduleName + '.html';
                config.templateUrl = state.templateUrl || templateUrl;

                // controller
                if (patterns.length >= 3) {
                    controller = _.upperFirst(patterns[1]) + defaultControllers[_.last(patterns)];
                    config.controller = state.controller || controller;
                }
                result[name] = config;
            });
            return result;
        })

        .decorator('url', function(state, parent) {
            var defaultUrl;

            if (state.abstract) {
                defaultUrl = '';
            } else {
                var patterns = state.name.split('.'),
                    defaultUrls = {
                        create: '/new',
                        update: '/edit',
                        list: '',
                        // The id is expected to be an integer, if not, the url has to
                        // be defined manually
                        detail: '/{id:int}',
                    };

                defaultUrl = defaultUrls[_.last(patterns)];
            }

            state.url = state.url || defaultUrl;
            return parent(state);
        });
    }
])

.config([
    '$stateProvider',
    '$locationProvider',
    'gettext',
    function($stateProvider, $locationProvider, gettext) {
        // Core urls
        $stateProvider
            .state('home', {
                url: '/',
                templateUrl: 'static/templates/home.html',
                data: {
                    title: gettext('Home'),
                    basePerm: 'core.can_see_frontpage',
                },
            })
            .state('projector', {
                url: '/projector/{id:int}/',
                templateUrl: 'static/templates/projector-container.html',
                data: {extern: true},
                onEnter: function($window) {
                    $window.location.href = this.url;
                }
            })
            .state('real-projector', {
                url: '/real-projector/{id:int}/',
                templateUrl: 'static/templates/projector.html',
                data: {extern: true},
                onEnter: function($window) {
                    $window.location.href = this.url;
                }
            })
            .state('manage-projectors', {
                url: '/manage-projectors',
                templateUrl: 'static/templates/core/manage-projectors.html',
                controller: 'ManageProjectorsCtrl',
                data: {
                    title: gettext('Manage projectors'),
                    basePerm: 'core.can_manage_projector',
                },
            })
            .state('core', {
                url: '/core',
                abstract: true,
                template: "<ui-view/>",
            })

            // legal notice and version
            .state('legalnotice', {
                url: '/legalnotice',
                controller: 'LegalNoticeCtrl',
                data: {
                    title: gettext('Legal notice'),
                },
            })

            //config
            .state('config', {
                url: '/config',
                controller: 'ConfigCtrl',
                resolve: {
                    configOptions: function(Config) {
                        return Config.getConfigOptions();
                    }
                },
                data: {
                    title: gettext('Settings'),
                    basePerm: 'core.can_manage_config',
                },
            })

            // search
            .state('search', {
                url: '/search?q',
                controller: 'SearchCtrl',
                templateUrl: 'static/templates/search.html',
                data: {
                    title: gettext('Search'),
                },
            })

            // tag
            .state('core.tag', {
                url: '/tag',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Tags'),
                    basePerm: 'core.can_manage_tags',
                },
            })
            .state('core.tag.list', {});

        $locationProvider.html5Mode(true);
    }
])

.config([
    '$sessionStorageProvider',
    function ($sessionStorageProvider) {
        $sessionStorageProvider.setKeyPrefix('OpenSlides');
    }
])

.factory('ProjectorMessageForm', [
    'Editor',
    'gettextCatalog',
    function (Editor, gettextCatalog) {
        return {
            getDialog: function (message) {
                return {
                    template: 'static/templates/core/projector-message-form.html',
                    controller: 'ProjectorMessageEditCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        projectorMessageId: function () {
                            return message.id;
                        }
                    },
                };
            },
            getFormFields: function () {
                return [
                    {
                        key: 'message',
                        type: 'editor',
                        templateOptions: {
                            label: gettextCatalog.getString('Message'),
                        },
                        data: {
                            ckeditorOptions: Editor.getOptions()
                        }
                    },
                ];
            },
        };
    }
])

.factory('TagForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getDialog: function (tag) {
                return {
                    template: 'static/templates/core/tag-form.html',
                    controller: (tag) ? 'TagUpdateCtrl' : 'TagCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        tagId: function () {return tag ? tag.id : void 0;},
                    },
                };
            },
            getFormFields: function() {
                return [
                    {
                        key: 'name',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Name'),
                            required: true
                        }
                    },
                ];
            },
        };
    }
])

/* This factory handles the filtering of the OS-data-tables. It contains
 * all logic needed for the table header filtering. Things to configure:
 * - multiselectFilters: A dict associating the filter name to a list (empty per default). E.g.
 *       { tag: [],
 *         category: [], }
 * - booleanFilters: A dict containing a dict for every filter. The value property is a must.
 *   For displaying properties like displayName, choiceYes and choiceNo could be usefull. E.g.
 *      { isPresent: {
 *          value: undefined,
 *          displayName: gettext('Is present'), } }
 * - propertyList, propertyFunctionList, propertyDict: See function getObjectQueryString
 */
.factory('osTableFilter', [
    '$sessionStorage',
    function ($sessionStorage) {
        var createInstance = function (tableName) {
            var self = {
                multiselectFilters: {},
                booleanFilters: {},
                filterString: '',
            };
            var existsStorageEntry = function () {
                return $sessionStorage[tableName];
            };
            var storage = existsStorageEntry();
            if (storage) {
                self = storage;
            }

            self.existsStorageEntry = existsStorageEntry;
            self.save = function () {
                $sessionStorage[tableName] = self;
            };
            self.areFiltersSet = function () {
                var areFiltersSet = _.find(self.multiselectFilters, function (filterList) {
                    return filterList.length > 0;
                });
                areFiltersSet = areFiltersSet || _.find(self.booleanFilters, function (filterDict) {
                    return filterDict.value !== undefined;
                });
                areFiltersSet = areFiltersSet || (self.filterString !== '');
                return areFiltersSet !== false;
            };
            self.reset = function () {
                _.forEach(self.multiselectFilters, function (filterList, filter) {
                    self.multiselectFilters[filter] = [];
                });
                _.forEach(self.booleanFilters, function (filterDict, filter) {
                    self.booleanFilters[filter].value = undefined;
                });
                self.filterString = '';
                self.save();
            };
            self.operateMultiselectFilter = function (filter, id, danger) {
                if (!danger) {
                    if (_.indexOf(self.multiselectFilters[filter], id) > -1) {
                        // remove id
                        self.multiselectFilters[filter].splice(_.indexOf(self.multiselectFilters[filter], id), 1);
                    } else {
                        // add id
                        self.multiselectFilters[filter].push(id);
                    }
                    self.save();
                }
            };
            /* Three things are could be given to create the query string:
             * - propertyList: Just a list of object's properties like ['title', 'name']
             * - propertyFunktionList: A list of functions returning a property (e.g. [function(motion) {return motion.getTitle();}] for retrieving the motions title)
             * - propertyDict: A dict association properties that are lists to functions on how to handle them.
             *   E.g.: {'tags': function (tag) {return tag.name;}, }
             *         The list of tags will be mapped with this function to a list of strings (tag names).
             */
            self.getObjectQueryString = function (obj) {
                var stringList = [];
                _.forEach(self.propertyList, function (property) {
                    stringList.push(obj[property]);
                });
                _.forEach(self.propertyFunctionList, function (fn) {
                    stringList.push(fn(obj));
                });
                _.forEach(self.propertyDict, function (idFunction, property) {
                    stringList.push(_.map(obj[property], idFunction).join(' '));
                });
                return stringList.join(' ');
            };
            return self;
        };

        return {
            createInstance: createInstance
        };
    }
])

/* This factory takes care of the sorting of OS-data-tables. Things to configure:
 * - column: the default column which is the list sorted by (e.g.
 *   instance.column='title')
 */
.factory('osTableSort', [
    function () {
        var createInstance = function () {
            var self = {
                column: '',
                reverse: false,
            };
            self.toggle = function (column) {
                if (self.column === column) {
                    self.reverse = !self.reverse;
                }
                self.column = column;
            };
            return self;
        };

        return {
            createInstance: createInstance
        };
    }
])

/*
 * This filter filters all items in an array. If the filterArray is empty, the
 * array is passed. The filterArray contains numbers of the multiselect, e. g. [1, 3, 4].
 * Then, all items in the array are passed, if the item_id (get with id_function) matches
 * one of the ids in filterArray. id_function could also return a list of ids. Example:
 * Item 1 has two tags with ids [1, 4]. filterArray == [3, 4] --> match
 *
 * If -1 is in the array items without an id will not be filtered. This is for implementing
 * a filter option like: "All items without a category"
 */
.filter('MultiselectFilter', [
    function () {
        return function (array, filterArray, idFunction) {
            if (filterArray.length === 0) {
                return array;
            }
            var itemsWithoutProperty = _.indexOf(filterArray, -1) > -1;
            return Array.prototype.filter.call(array, function (item) {
                var id = idFunction(item);
                if (typeof id === 'number') {
                    id = [id];
                } else if (id === null || !id.length) {
                    return itemsWithoutProperty;
                }
                return _.intersection(id, filterArray).length > 0;
            });
        };
    }
])

.filter('osFilter', [
    function () {
        return function (array, string, getFilterString) {
            if (!string) {
                return array;
            }
            return Array.prototype.filter.call(array, function (item) {
                return getFilterString(item).toLowerCase().indexOf(string.toLowerCase()) > -1;
            });
        };
    }
])

// angular formly config options
.run([
    'formlyConfig',
    function (formlyConfig) {
        // NOTE: This next line is highly recommended. Otherwise Chrome's autocomplete
        // will appear over your options!
        formlyConfig.extras.removeChromeAutoComplete = true;

        // Configure custom types
        formlyConfig.setType({
            name: 'editor',
            extends: 'textarea',
            templateUrl: 'static/templates/core/editor.html',
        });
        formlyConfig.setType({
            name: 'password',
            extends: 'input',
            templateUrl: 'static/templates/core/password.html',
        });
        formlyConfig.setType({
            name: 'checkbox',
            templateUrl: 'static/templates/core/checkbox.html',
            overwriteOk: true,
        });
        formlyConfig.setType({
            name: 'select-single',
            extends: 'select',
            templateUrl: 'static/templates/core/select-single.html'
        });
        formlyConfig.setType({
            name: 'select-multiple',
            extends: 'select',
            templateUrl: 'static/templates/core/select-multiple.html'
        });
        formlyConfig.setType({
            name: 'radio-buttons',
            templateUrl: 'static/templates/core/radio-buttons.html',
            wrapper: ['bootstrapHasError'],
            defaultOptions: {
                noFormControl: false
            }
        });
        formlyConfig.setType({
            name: 'file',
            extends: 'input',
            templateUrl: 'static/templates/core/file.html',
        });
    }
])

// html-tag os-form-field to generate generic from fields
// TODO: make it possible to use other fields then config fields
.directive('osFormField', [
    '$parse',
    'Config',
    'gettextCatalog',
    function($parse, Config, gettextCatalog) {
        var getHtmlType = function (type) {
            return {
                string: 'text',
                text: 'textarea',
                markupText: 'editor',
                integer: 'number',
                boolean: 'checkbox',
                choice: 'choice',
                comments: 'comments',
                colorpicker: 'colorpicker',
                datetimepicker: 'datetimepicker',
                majorityMethod: 'choice',
            }[type];
        };

        return {
            restrict: 'E',
            scope: true,
            templateUrl: 'static/templates/config-form-field.html',
            link: function ($scope, iElement, iAttrs, controller, transcludeFn) {
                var field = $parse(iAttrs.field)($scope);
                var config = Config.get(field.key);
                $scope.type = getHtmlType(field.input_type);
                if ($scope.type == 'choice') {
                    $scope.choices = field.choices;
                    $scope.value = config.value;
                } else {
                    $scope.value = gettextCatalog.getString(config.value);
                }
                $scope.label = field.label;
                $scope.key = 'field-' + field.key;
                $scope.help_text = field.help_text;
                $scope.default_value = field.default_value;
                $scope.reset = function () {
                    if ($scope.type == 'choice') {
                        $scope.value = $scope.default_value;
                    } else {
                        $scope.value = gettextCatalog.getString($scope.default_value);
                    }
                    $scope.save(field, $scope.value);
                };
            }
        };
    }
])

/* This directive provides a csv import template.
 * Papa Parse is used to parse the csv file. Accepted attributes:
 * * change:
 *   Callback if file changes. The one parameter is csv passing the parsed file
 * * config (optional):
 *   - accept: String with extensions: default '.csv .txt'
 *   - encodingOptions: List with encodings. Default ['UTF-8', 'ISO-8859-1']
 *   - parseConfig: a dict passed to PapaParse
 */
.directive('csvImport', [
    function () {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/csv-import.html',
            scope: {
                change: '&',
                config: '=?',
            },
            controller: function ($scope, $element, $attrs, $location) {
                // set config if it is not given
                if (!$scope.config) {
                    $scope.config = {};
                }
                if (!$scope.config.parseConfig) {
                    $scope.config.parseConfig = {};
                }

                $scope.inputElement = angular.element($element[0].querySelector('#csvFileSelector'));

                // set accept and encoding
                $scope.accept = $scope.config.accept || '.csv';
                $scope.encodingOptions = $scope.config.encodingOptions || ['UTF-8'];
                $scope.encoding = $scope.encodingOptions[0];

                $scope.parse = function () {
                    var inputElement = $scope.inputElement[0];
                    if (!inputElement.files.length) {
                        $scope.change({csv: {data: {}}});
                    } else {
                        var parseConfig = _.defaults(_.clone($scope.config.parseConfig), {
                            delimiter: $scope.delimiter,
                            encoding: $scope.encoding,
                            header: false, // we do not want to have dicts in result
                            complete: function (csv) {
                                if (csv.data.length) {
                                    csv.meta.fields = csv.data[0];
                                }
                                else {
                                    csv.meta.fields = [];
                                }
                                csv.data = csv.data.splice(1); // do not interpret the header as data
                                $scope.$apply(function () {
                                    if (csv.meta.delimiter) {
                                        $scope.autodelimiter = csv.meta.delimiter;
                                    }
                                    $scope.change({csv: csv});
                                });
                            },
                            error: function () {
                                $scope.$apply(function () {
                                    $scope.change({csv: {data: {}}});
                                });
                            },
                        });

                        Papa.parse(inputElement.files[0], parseConfig);
                    }
                };

                $scope.clearFile = function () {
                    $scope.inputElement[0].value = '';
                    $scope.selectedFile = undefined;
                    $scope.parse();
                };

                $scope.inputElement.on('change', function () {
                    $scope.selectedFile = _.last($scope.inputElement[0].value.split('\\'));
                    $scope.parse();
                });
            },
        };
    }
])

.directive('messaging', [
    '$timeout',
    'Messaging',
    function ($timeout, Messaging) {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/messaging.html',
            scope: {},
            controller: function ($scope, $element, $attrs, $location) {
                $scope.messages = {};

                var update = function () {
                    $scope.messages = Messaging.getMessages();
                };
                Messaging.registerMessageChangeCallback(update);

                $scope.close = function (id) {
                    Messaging.deleteMessage(id);
                };
            },
        };
    }
])

.controller('MainMenuCtrl', [
    '$scope',
    'mainMenu',
    function ($scope, mainMenu) {
        mainMenu.registerScope($scope);
        $scope.isMenuOpen = false;
        $scope.closeMenu = function () {
            $scope.isMenuOpen = false;
        };
    }
])

.controller('LanguageCtrl', [
    '$scope',
    'gettextCatalog',
    'Languages',
    'filterFilter',
    function ($scope, gettextCatalog, Languages, filterFilter) {
        $scope.languages = Languages.getLanguages();
        $scope.selectedLanguage = filterFilter($scope.languages, {selected: true});
        // controller to switch app language
        $scope.switchLanguage = function (lang) {
            $scope.languages = Languages.setCurrentLanguage(lang);
            $scope.selectedLanguage = filterFilter($scope.languages, {selected: true});
        };
    }
])

// Projector Sidebar Controller
.controller('ProjectorSidebarCtrl', [
    '$scope',
    '$document',
    '$window',
    function ($scope, $document, $window) {
        $scope.isProjectorSidebar = false;
        $scope.showProjectorSidebar = function (show) {
            $scope.isProjectorSidebar = show;
        };

        // Sidebar scroll
        var marginTop = 20, // margin-top from #content
            marginBottom = 30, // 30px + 20px sidebar margin-bottom = 50px from footer
            sidebar;

        var sidebarScroll = function () {
            var sidebarHeight = sidebar.height(),
                sidebarOffset = sidebar.offset().top,
                sidebarMinOffset = $('#header').height() + $('#nav').height() + marginTop,
                documentHeight = $document.height(),
                windowHeight = $window.innerHeight,
                scrollTop = $window.pageYOffset;

            // First, check if there is a need to scroll: scroll if the sidebar is smaller then the content
            if (sidebarHeight < $('.col1').height()) {
                if ((scrollTop + marginTop + sidebarHeight) > (documentHeight - marginBottom)) {
                    // Stick to the bottom
                    var bottom = marginBottom + scrollTop + windowHeight - documentHeight;
                    sidebar.css({'position': 'fixed', 'top': '', 'bottom': bottom});
                } else if ((scrollTop + marginTop) > sidebarMinOffset) {
                    // scroll with the user
                    sidebar.css({'position': 'fixed', 'top': marginTop, 'bottom': ''});
                } else {
                    // Stick to the top
                    sidebar.css({'position': 'relative', 'top': 0, 'bottom': ''});
                }
            } else {
                // Stick to the top, if the sidebar is larger then the content
                sidebar.css({'position': 'relative', 'top': 0, 'bottom': ''});
            }
        };

        $scope.initSidebar = function () {
            sidebar = $('#sidebar');
            $scope.$watch(function () {
                return sidebar.height();
            }, sidebarScroll);
            angular.element($window).bind('scroll', sidebarScroll);
        };

    }
])

// Legal Notice Controller
.controller('LegalNoticeCtrl', [
    '$scope',
    '$http',
    function ($scope, $http) {
        $http.get('/core/version/').then(function (success) {
            $scope.core_version = success.data.openslides_version;
            $scope.plugins = success.data.plugins;
        });
    }
])

// Config Controller
.controller('ConfigCtrl', [
    '$scope',
    '$timeout',
    'MajorityMethodChoices',
    'Config',
    'configOptions',
    'gettextCatalog',
    'DateTimePickerTranslation',
    'Editor',
    function($scope, $timeout, MajorityMethodChoices, Config, configOptions,
        gettextCatalog, DateTimePickerTranslation, Editor) {
        Config.bindAll({}, $scope, 'configs');
        $scope.configGroups = configOptions.data.config_groups;
        $scope.dateTimePickerTranslatedButtons = DateTimePickerTranslation.getButtons();

        $scope.ckeditorOptions = Editor.getOptions();
        $scope.ckeditorOptions.on.change = function (event) {
            // we could just retrieve the key, but we need the configOption object.
            var configOption_key = event.editor.element.$.id;

            // find configOption object
            var subgroups = _.flatMap($scope.configGroups, function (group) {
                return group.subgroups;
            });
            var items = _.flatMap(subgroups, function (subgroup) {
                return subgroup.items;
            });
            var configOption = _.find(items, function (_item) {
                return _item.key === configOption_key;
            });

            var editor = this;
            // The $timeout executes the given function in an angular context. Because
            // this is a standard JS event, all changes may not happen in the digist-cylce.
            // By using $timeout angular calls $apply for us that we do not have to care
            // about starting the digist-cycle.
            $timeout(function () {
                $scope.save(configOption, editor.getData());
            }, 1);
        };

        // save changed config value
        $scope.save = function(configOption, value) {
            Config.get(configOption.key).value = value;
            Config.save(configOption.key).then(function (success) {
                configOption.success = true;
                // fade out the success symbol after 2 seconds.
                $timeout(function () {
                    var element = $('#success-field-' + configOption.key);
                    element.fadeOut(800, function () {
                        configOption.success = void 0;
                    });
                }, 2000);
            }, function (error) {
                configOption.success = false;
                configOption.errorMessage = error.data.detail;
            });
        };

        // For comments input
        $scope.addComment = function (key, parent) {
            parent.value.push({
                name: gettextCatalog.getString('New'),
                public: false,
            });
            $scope.save(key, parent.value);
        };
        $scope.removeComment = function (key, parent, index) {
            parent.value.splice(index, 1);
            $scope.save(key, parent.value);
        };

        // For majority method
        angular.forEach(
            _.filter($scope.configGroups, function (configGroup) {
                return configGroup.name === 'Motions' || configGroup.name === 'Elections';
            }),
            function (configGroup) {
                var configItem;
                _.forEach(configGroup.subgroups, function (subgroup) {
                    configItem = _.find(subgroup.items, ['input_type', 'majorityMethod']);
                    if (configItem !== undefined) {
                        // Break the forEach loop if we found something.
                        return false;
                    }
                });
                if (configItem !== undefined) {
                    configItem.choices = MajorityMethodChoices;
                }
            }
        );
    }
])

// Search Bar Controller
.controller('SearchBarCtrl', [
    '$scope',
    '$state',
    '$sanitize',
    function ($scope, $state, $sanitize) {
        $scope.search = function() {
            var query = _.escape($scope.querybar);
            $scope.querybar = '';
            $state.go('search', {q: query});
        };
    }
])

// Search Controller
.controller('SearchCtrl', [
    '$scope',
    '$filter',
    '$stateParams',
    'Search',
    'DS',
    'Motion',
    function ($scope, $filter, $stateParams, Search, DS, Motion) {
        $scope.searchresults = [];
        var searchModules = Search.getAll();

        // search function
        $scope.search = function() {
            $scope.results = [];
            var foundObjects = [];
            // search in rest properties of all defined searchModule
            // (does not found any related objects, e.g. speakers of items)
            _.forEach(searchModules, function(searchModule) {
                var result = {};
                result.verboseName = searchModule.verboseName;
                result.collectionName = searchModule.collectionName;
                result.urlDetailState = searchModule.urlDetailState;
                result.weight = searchModule.weight;
                result.checked = true;
                result.elements = $filter('filter')(DS.getAll(searchModule.collectionName), $scope.searchquery);
                $scope.results.push(result);
                _.forEach(result.elements, function(element) {
                    foundObjects.push(element);
                });
            });
            // search additionally in specific releations of all defined searchModules
            _.forEach(searchModules, function(searchModule) {
                _.forEach(DS.getAll(searchModule.collectionName), function(object) {
                    if (_.isFunction(object.hasSearchResult)) {
                        if (object.hasSearchResult(foundObjects, $scope.searchquery)) {
                            // releation found, check if object is not yet in search results
                            _.forEach($scope.results, function(result) {
                                if ((object.getResourceName() === result.collectionName) &&
                                        _.findIndex(result.elements, {'id': object.id}) === -1) {
                                    result.elements.push(object);
                                }
                            });
                        }
                    } else {
                        return false;
                    }
                });
            });
        };

        //get search string from parameters submitted from outside the scope
        if ($stateParams.q) {
            $scope.searchquery = $stateParams.q;
            $scope.search();
        }
    }
])

// Projector Control Controller
.controller('ProjectorControlCtrl', [
    '$scope',
    '$http',
    '$interval',
    '$state',
    '$q',
    '$filter',
    'Config',
    'Projector',
    'CurrentListOfSpeakersItem',
    'CurrentListOfSpeakersSlide',
    'ProjectionDefault',
    'ProjectorMessage',
    'Countdown',
    'gettextCatalog',
    'ngDialog',
    'ProjectorMessageForm',
    function($scope, $http, $interval, $state, $q, $filter, Config, Projector, CurrentListOfSpeakersItem,
        CurrentListOfSpeakersSlide, ProjectionDefault, ProjectorMessage, Countdown, gettextCatalog,
        ngDialog, ProjectorMessageForm) {
        ProjectorMessage.bindAll({}, $scope, 'messages');

        var intervals = [];
        var calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        var cancelIntervalTimers = function () {
            intervals.forEach(function (interval) {
                $interval.cancel(interval);
            });
        };
        $scope.$watch(function () {
            return Countdown.lastModified();
        }, function () {
            $scope.countdowns = Countdown.getAll();

            // stop ALL interval timer
            cancelIntervalTimers();
            $scope.countdowns.forEach(function (countdown) {
                if (countdown.running) {
                    calculateCountdownTime(countdown);
                    intervals.push($interval(function () { calculateCountdownTime(countdown); }, 1000));
                } else {
                    countdown.seconds = countdown.countdown_time;
                }
            });
        });
        $scope.$on('$destroy', function() {
            // Cancel all intervals if the controller is destroyed
            cancelIntervalTimers();
        });

        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            if (!$scope.active_projector) {
                $scope.active_projector = $filter('orderBy')($scope.projectors, 'id')[0];
            }
            $scope.setIframeSize($scope.active_projector);
            if ($scope.projectors.length === 1) {
                $scope.currentListOfSpeakersAsOverlay = true;
            }

            $scope.messageDefaultProjectorId = ProjectionDefault.filter({name: 'messages'})[0].projector_id;
            $scope.countdownDefaultProjectorId = ProjectionDefault.filter({name: 'countdowns'})[0].projector_id;
            $scope.listOfSpeakersDefaultProjectorId = ProjectionDefault.filter({name: 'agenda_current_list_of_speakers'})[0].projector_id;
        });
        // watch for changes in projector_broadcast and currentListOfSpeakersReference
        var last_broadcast;
        $scope.$watch(function () {
            return Config.lastModified();
        }, function () {
            var broadcast = Config.get('projector_broadcast').value;
            if (!last_broadcast || last_broadcast != broadcast) {
                last_broadcast = broadcast;
                $scope.broadcast = broadcast;
            }
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
        });

        $scope.changeProjector = function (projector) {
            $scope.active_projector = projector;
            $scope.setIframeSize(projector);
        };
        $scope.setIframeSize = function (projector) {
            $scope.scale = 256.0 / projector.width;
            $scope.iframeHeight = $scope.scale * projector.height;
        };

        $scope.editCurrentSlide = function (projector) {
            var data = projector.getFormOrStateForCurrentSlide();
            if (data) {
                if (data.form) {
                    ngDialog.open(data.form.getDialog({id: data.id}));
                } else {
                    $state.go(data.state, {id: data.id});
                }
            }
        };

        // *** countdown functions ***
        $scope.calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        $scope.editCountdown = function (countdown) {
            countdown.editFlag = false;
            countdown.description = countdown.new_description;
            Countdown.save(countdown);
            if (!countdown.running) {
                countdown.reset();
            }
        };
        $scope.addCountdown = function () {
            var default_time = parseInt($scope.config('projector_default_countdown'));
            var countdown = {
                description: '',
                default_time: default_time,
                countdown_time: default_time,
                running: false,
            };
            Countdown.create(countdown);
        };
        $scope.removeCountdown = function (countdown) {
            var isProjectedIds = countdown.isProjected();
            _.forEach(isProjectedIds, function(id) {
                countdown.project(id);
            });
            Countdown.destroy(countdown.id);
        };

        // *** message functions ***
        $scope.editMessage = function (message) {
            ngDialog.open(ProjectorMessageForm.getDialog(message));
        };
        $scope.addMessage = function () {
            var message = {message: ''};
            ProjectorMessage.create(message);
        };
        $scope.removeMessage = function (message) {
            var isProjectedIds = message.isProjected();
            _.forEach(isProjectedIds, function(id) {
                message.project(id);
            });
            ProjectorMessage.destroy(message.id);
        };

        /* Current list of speakers */
        $scope.currentListOfSpeakers = CurrentListOfSpeakersSlide;
        // Set the current overlay status
        if ($scope.currentListOfSpeakers.isProjected().length) {
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            $scope.currentListOfSpeakersAsOverlay = isProjected[0].overlay;
        } else {
            $scope.currentListOfSpeakersAsOverlay = true;
        }
        // go to the list of speakers(management) of the currently displayed list of speakers reference slide
        $scope.goToListOfSpeakers = function() {
            var item = $scope.currentListOfSpeakersItem();
            if (item) {
                $state.go('agenda.item.detail', {id: item.id});
            }
        };
        $scope.currentListOfSpeakersItem = function () {
            return CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
        };
        $scope.setOverlay = function (overlay) {
            $scope.currentListOfSpeakersAsOverlay = overlay;
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.overlay != overlay) { // change the overlay if it is different
                        $scope.currentListOfSpeakers.project(mapping.projectorId, overlay);
                    }
                });
            }
        };
    }
])

.controller('ProjectorMessageEditCtrl', [
    '$scope',
    'projectorMessageId',
    'ProjectorMessage',
    'ProjectorMessageForm',
    function ($scope, projectorMessageId, ProjectorMessage, ProjectorMessageForm) {
        $scope.formFields = ProjectorMessageForm.getFormFields();
        $scope.model = angular.copy(ProjectorMessage.get(projectorMessageId));

        $scope.save = function (message) {
            ProjectorMessage.inject(message);
            ProjectorMessage.save(message);
            $scope.closeThisDialog();
        };
    }
])

.controller('ManageProjectorsCtrl', [
    '$scope',
    '$http',
    '$timeout',
    'Projector',
    'ProjectionDefault',
    'Config',
    'ProjectorMessage',
    'ngDialog',
    function ($scope, $http, $timeout, Projector, ProjectionDefault, Config,
        ProjectorMessage, ngDialog) {
        ProjectionDefault.bindAll({}, $scope, 'projectiondefaults');

        // watch for changes in projector_broadcast
        // and projector_currentListOfSpeakers_reference
        var last_broadcast, last_clos;
        $scope.$watch(function () {
            return Config.lastModified();
        }, function () {
            var broadcast = $scope.config('projector_broadcast'),
            currentListOfSpeakers = $scope.config('projector_currentListOfSpeakers_reference');
            if (!last_broadcast || last_broadcast != broadcast) {
                last_broadcast = broadcast;
                $scope.broadcast = broadcast;
            }
            if (!last_clos || last_clos != currentListOfSpeakers) {
                last_clos = currentListOfSpeakers;
                $scope.currentListOfSpeakers = currentListOfSpeakers;
            }
        });

        // watch for changes in Projector, and recalc scale and iframeHeight
        var first_watch = true;
        $scope.resolutions = [];
        $scope.edit = [];
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            $scope.projectors.forEach(function (projector) {
                projector.iframeScale = 256.0 / projector.width;
                projector.iframeHeight = projector.iframeScale * projector.height;
                if (first_watch) {
                    $scope.resolutions[projector.id] = {
                        width: projector.width,
                        height: projector.height
                    };
                    $scope.edit[projector.id] = false;
                }
            });
            if ($scope.projectors.length) {
                first_watch = false;
            }
        });

        // Set list of speakers reference
        $scope.setListOfSpeakers = function (projector) {
            Config.get('projector_currentListOfSpeakers_reference').value = projector.id;
            Config.save('projector_currentListOfSpeakers_reference');
        };

        // Projector functions
        $scope.setProjectionDefault = function (projector, projectiondefault) {
            if (projectiondefault.projector_id !== projector.id) {
                $http.post('/rest/core/projector/' + projector.id + '/set_projectiondefault/', projectiondefault.id);
            }
        };
        $scope.createProjector = function (name) {
            var projector = {
                name: name,
                config: {},
                scale: 0,
                scroll: 0,
                blank: false,
                projectiondefaults: [],
            };
            Projector.create(projector).then(function (projector) {
                $http.post('/rest/core/projector/' + projector.id + '/activate_elements/', [{
                    name: 'core/clock',
                    stable: true
                }]);
                $scope.resolutions[projector.id] = {
                    width: projector.width,
                    height: projector.height
                };
            });
        };
        $scope.deleteProjector = function (projector) {
            if (projector.id != 1) {
                Projector.destroy(projector.id);
            }
        };
        $scope.editCurrentSlide = function (projector) {
            var data = projector.getFormOrStateForCurrentSlide();
            if (data) {
                if (data.form) {
                    ngDialog.open(data.form.getDialog({id: data.id}));
                } else {
                    $state.go(data.state, {id: data.id});
                }
            }
        };
        $scope.editName = function (projector) {
            projector.config = projector.elements;
            Projector.save(projector);
        };
        $scope.changeResolution = function (projectorId) {
            $http.post(
                '/rest/core/projector/' + projectorId + '/set_resolution/',
                $scope.resolutions[projectorId]
            ).then(function (success) {
                $scope.resolutions[projectorId].error = null;
            }, function (error) {
                $scope.resolutions[projectorId].error = error.data.detail;
            });
        };

        // Identify projectors
        $scope.identifyProjectors = function () {
            if ($scope.identifyPromise) {
                $timeout.cancel($scope.identifyPromise);
                $scope.removeIdentifierMessages();
            } else {
                // Create new Message
                var message = {
                    message: '',
                };
                ProjectorMessage.create(message).then(function(message){
                    $scope.projectors.forEach(function (projector) {
                        $http.post('/rest/core/projector/' + projector.id + '/activate_elements/', [{
                            name: 'core/projector-message',
                            stable: true,
                            id: message.id,
                            identify: true,
                        }]);
                    });
                    $scope.identifierMessage = message;
                });
                $scope.identifyPromise = $timeout($scope.removeIdentifierMessages, 3000);
            }
        };
        $scope.removeIdentifierMessages = function () {
            Projector.getAll().forEach(function (projector) {
                _.forEach(projector.elements, function (element, uuid) {
                    if (element.name === 'core/projector-message' && element.id === $scope.identifierMessage.id) {
                        $http.post('/rest/core/projector/' + projector.id + '/deactivate_elements/', [uuid]);
                    }
                });
            });
            ProjectorMessage.destroy($scope.identifierMessage.id);
            $scope.identifyPromise = null;
        };
    }
])

// Tag Controller
.controller('TagListCtrl', [
    '$scope',
    'Tag',
    'ngDialog',
    'TagForm',
    'gettext',
    'ErrorMessage',
    function($scope, Tag, ngDialog, TagForm, gettext, ErrorMessage) {
        Tag.bindAll({}, $scope, 'tags');
        $scope.alert = {};

        // setup table sorting
        $scope.sortColumn = 'name';
        $scope.reverse = false;
        // function to sort by clicked column
        $scope.toggleSort = function ( column ) {
            if ( $scope.sortColumn === column ) {
                $scope.reverse = !$scope.reverse;
            }
            $scope.sortColumn = column;
        };
        $scope.delete = function (tag) {
            Tag.destroy(tag.id).then(
                function(success) {
                    $scope.alert = {
                        type: 'success',
                        msg: gettext('The delete was successful.'),
                        show: true,
                    };
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        $scope.editOrCreate = function (tag) {
            ngDialog.open(TagForm.getDialog(tag));
        };
    }
])

.controller('TagCreateCtrl', [
    '$scope',
    'Tag',
    'TagForm',
    'ErrorMessage',
    function($scope, Tag, TagForm, ErrorMessage) {
        $scope.model = {};
        $scope.alert = {};
        $scope.formFields = TagForm.getFormFields();
        $scope.save = function (tag) {
            Tag.create(tag).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TagUpdateCtrl', [
    '$scope',
    'Tag',
    'tagId',
    'TagForm',
    'ErrorMessage',
    function($scope, Tag, tagId, TagForm, ErrorMessage) {
        $scope.model = angular.copy(Tag.get(tagId));
        $scope.alert = {};
        $scope.formFields = TagForm.getFormFields();
        $scope.save = function (tag) {
            Tag.inject(tag);
            Tag.save(tag).then(function(success) {
                $scope.closeThisDialog();
            }, function (error) {
                // save error: revert all changes by restore
                // the original object
                Tag.refresh(tag);
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

// counter of new (unread) chat messages
.value('NewChatMessages', [])

// ChatMessage Controller
.controller('ChatMessageCtrl', [
    '$scope',
    '$http',
    '$timeout',
    'ChatMessage',
    'NewChatMessages',
    function ($scope, $http, $timeout, ChatMessage, NewChatMessages) {
        ChatMessage.bindAll({}, $scope, 'chatmessages');
        $scope.unreadMessages = NewChatMessages.length;
        $scope.chatboxIsCollapsed = true;
        $scope.openChatbox = function () {
            $scope.chatboxIsCollapsed = !$scope.chatboxIsCollapsed;
            NewChatMessages = [];
            $scope.unreadMessages = NewChatMessages.length;
            $timeout(function () {
                angular.element('#messageInput').focus();
            }, 0);
        };
        $scope.sendMessage = function () {
            angular.element('#messageSendButton').addClass('disabled');
            angular.element('#messageInput').attr('disabled', '');
            $http.post(
                '/rest/core/chat-message/',
                {message: $scope.newMessage}
            ).then(function (success) {
                $scope.newMessage = '';
                angular.element('#messageSendButton').removeClass('disabled');
                angular.element('#messageInput').removeAttr('disabled');
                $timeout(function () {
                    angular.element('#messageInput').focus();
                }, 0);
            }, function (error) {
                angular.element('#messageSendButton').removeClass('disabled');
                angular.element('#messageInput').removeAttr('disabled');
            });
        };
        // increment unread messages counter for each new message
        $scope.$watch('chatmessages', function (newVal, oldVal) {
            // add new message id if there is really a new message which is not yet tracked
            if (oldVal.length > 0 && newVal.length > 0) {
                if ((oldVal[oldVal.length-1].id != newVal[newVal.length-1].id) &&
                    ($.inArray(newVal[newVal.length-1].id, NewChatMessages) == -1)) {
                    NewChatMessages.push(newVal[newVal.length-1].id);
                    $scope.unreadMessages = NewChatMessages.length;
                }
            } else if (newVal.length === 0) {
                NewChatMessages = [];
                $scope.unreadMessages = 0;
            }
        });

        $scope.clearChatHistory = function () {
            $http.post('/rest/core/chat-message/clear/');
        };
    }
])

// format time string for model ("s") and view format ("h:mm:ss" or "mm:ss")
.directive('minSecFormat', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return {
            require: 'ngModel',
            link: function(scope, element, attrs, ngModelController) {
                ngModelController.$parsers.push(function(data) {
                    //convert data from view format (mm:ss) to model format (s)
                    return HumanTimeConverter.humanTimeToSeconds(data, {seconds: true});
                });

                ngModelController.$formatters.push(function(data) {
                    //convert data from model format (s) to view format (mm:ss)
                    return HumanTimeConverter.secondsToHumanTime(data);
                });
            }
        };
    }
])

// format time string for model ("m") and view format ("h:mm" or "hh:mm")
.directive('hourMinFormat', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return {
            require: 'ngModel',
            link: function(scope, element, attrs, ngModelController) {
                ngModelController.$parsers.push(function(data) {
                    //convert data from view format (hh:mm) to model format (m)
                    return HumanTimeConverter.humanTimeToSeconds(data, {hours: true})/60;
                });

                ngModelController.$formatters.push(function(data) {
                    //convert data from model format (m) to view format (hh:mm)
                    return HumanTimeConverter.secondsToHumanTime(data*60,
                        { seconds: 'disabled',
                            hours: 'enabled' }
                    );
                });
            }
        };
    }
])

.directive('osFocusMe', [
    '$timeout',
    function ($timeout) {
        return {
            link: function (scope, element, attrs, model) {
                $timeout(function () {
                    element[0].focus();
                });
            }
        };
    }
])

.filter('toArray', function(){
    /*
     * Transforms an object to an array. Items of the array are the values of
     * the object elements.
     */
    return function(obj) {
        var result = [];
        angular.forEach(obj, function(val, key) {
            result.push(val);
        });
        return result;
    };
})

//Mark all core config strings for translation in Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Presentation and assembly system');
        gettext('Event name');
        gettext('<a href="http://www.openslides.org">OpenSlides</a> is a free ' +
                'web based presentation and assembly system for visualizing ' +
                'and controlling agenda, motions and elections of an ' +
                'assembly.');
        gettext('General');
        gettext('Event');
        gettext('Short description of event');
        gettext('Event date');
        gettext('Event location');
        gettext('Event organizer');
        gettext('Legal notice');
        gettext('Front page title');
        gettext('Welcome to OpenSlides');
        gettext('Front page text');
        gettext('[Space for your welcome text.]');
        gettext('Allow access for anonymous guest users');
        gettext('Show this text on the login page');
        gettext('Separator used for all csv exports and examples');
        gettext('Show logo on projector');
        gettext('You can replace the logo. Just copy a file to ' +
                '"static/img/logo-projector.png" in your OpenSlides data ' +
                'path.');
        gettext('Projector');
        gettext('Projector language');
        gettext('Current browser language');
        gettext('Show title and description of event on projector');
        gettext('Background color of projector header and footer');
        gettext('Font color of projector header and footer');
        gettext('Font color of projector headline');
        gettext('Predefined seconds of new countdowns');
        gettext('Color for blanked projector');
        gettext('List of speakers overlay');

        // Mark the string 'Default projector' here, because it does not appear in the templates.
        gettext('Default projector');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.start', [])

.run([
    '$http',
    '$rootScope',
    '$state',
    'autoupdate',
    'operator',
    'Group',
    'mainMenu',
    function($http, $rootScope, $state, autoupdate, operator, Group, mainMenu) {
        $rootScope.openslidesBootstrapDone = false;
        $http.get('/users/whoami/').then(function (success) {
            $rootScope.guest_enabled = success.data.guest_enabled;
            if (success.data.user_id === null && !success.data.guest_enabled) {
                // Redirect to login dialog if user is not logged in.
                $state.go('login', {guest_enabled: success.data.guest_enabled});
            } else {
                autoupdate.newConnect();
                autoupdate.firstMessageDeferred.promise.then(function () {
                    operator.setUser(success.data.user_id, success.data.user);
                    $rootScope.operator = operator;
                    mainMenu.updateMainMenu();
                    $rootScope.openslidesBootstrapDone = true;
                });
            }
        });
    }
])

.run([
    '$rootScope',
    '$state',
    'operator',
    'User',
    'Group',
    'mainMenu',
    function ($rootScope, $state, operator, User, Group, mainMenu) {
        var permissionChangeCallback = function () {
            operator.reloadPerms();
            mainMenu.updateMainMenu();
            var stateData = $state.current.data;
            var basePerm = stateData ? stateData.basePerm : '';
            $rootScope.baseViewPermissionsGranted = basePerm ?
                operator.hasPerms(basePerm) : true;
        };

        $rootScope.$watch(function () {
            return Group.lastModified();
        }, function () {
            if (Group.getAll().length) {
                permissionChangeCallback();
            }
        });

        $rootScope.$watch(function () {
            return operator.user ? User.lastModified(operator.user.id) : true;
        }, function () {
            permissionChangeCallback();
        });
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.create', [
    'OpenSlidesApp.mediafiles.forms',
])

.controller('MediafileCreateCtrl', [
    '$scope',
    'MediafileForm',
    'ErrorMessage',
    function ($scope, MediafileForm, ErrorMessage) {
        $scope.model = {};
        $scope.alert = {};
        $scope.formFields = MediafileForm.getFormFields(true);

        // upload and save mediafile
        $scope.save = function (mediafile) {
            if (typeof mediafile.getFile === 'function') {
                $scope.activeUpload = MediafileForm.uploadFile(mediafile).then(
                    function (success) {
                        $scope.closeThisDialog();
                    },
                    function (error) {
                        $scope.activeUpload = void 0;
                        $scope.alert = ErrorMessage.forAlert(error);
                    },
                    function (progress) {
                        $scope.progress = parseInt(100.0 * progress.loaded / progress.total);
                    }
                );
            }
        };
        $scope.close = function () {
            // TODO: abort() is not a function. But it is documented in the docs.
            // See https://github.com/danialfarid/ng-file-upload/issues/1844
            /*if ($scope.activeUpload) {
                $scope.activeUpload.abort();
            }*/
            $scope.closeThisDialog();
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.forms', [
    'gettext',
    'ngFileUpload',
    'ui.router',
    //TODO: Add deps for operator, User
])

// Service for mediafile form
.factory('MediafileForm', [
    'gettextCatalog',
    'Upload',
    'operator',
    'User',
    function (gettextCatalog, Upload, operator, User) {
        return {
            // ngDialog for mediafile form
            getDialog: function (mediafile) {
                return {
                    template: 'static/templates/mediafiles/mediafile-form.html',
                    controller: (mediafile) ? 'MediafileUpdateCtrl' : 'MediafileCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        mediafileId: function () {return mediafile ? mediafile.id : void 0;}
                    },
                };
            },
            // upload selected file (used by create view only)
            uploadFile: function (mediafile) {
                var file = mediafile.getFile();
                if (!mediafile.title) {
                    mediafile.title = file.name;
                }
                if (!mediafile.uploader_id) {
                    mediafile.uploader_id = operator.user.id;
                }
                return Upload.upload({
                    url: '/rest/mediafiles/mediafile/',
                    method: 'POST',
                    data: {mediafile: file, title: mediafile.title, uploader_id: mediafile.uploader_id, hidden: mediafile.hidden}
                });
            },
            getFormFields: function (isCreateForm) {
                return [
                    {
                        key: 'newFile',
                        type: 'file',
                        templateOptions: {
                            label: gettextCatalog.getString('File'),
                            required: true,
                            change: function (model, files, event, rejectedFiles) {
                                var file = files ? files[0] : void 0;
                                model.getFile = function () {
                                    return file;
                                };
                                model.newFile = file ? file.name : void 0;
                            },
                        },
                        hide: !isCreateForm,
                    },
                    {
                        key: 'title',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Title'),
                        },
                    },
                    {
                        key: 'hidden',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Hidden'),
                            description: gettextCatalog.getString('This does not protect the ' +
                                'file but hides it for non authorized users.'),
                        },
                        hide: !operator.hasPerms('mediafiles.can_see_hidden'),
                    },
                    {
                        key: 'uploader_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Uploaded by'),
                            options: User.getAll(),
                            ngOptions: 'option.id as option.full_name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a participant ...')
                        },
                        hide: !operator.hasPerms('mediafiles.can_manage')
                    },
                ];

            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.list', [
    'gettext',
    'ngDialog',
    'OpenSlidesApp.mediafiles.forms',
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for operator, User, Projector, ProjectionDefault, osTableFilter, osTableSort,
])

.controller('MediafileListCtrl', [
    '$http',
    '$scope',
    'gettext',
    'ngDialog',
    'osTableFilter',
    'osTableSort',
    'ProjectionDefault',
    'Projector',
    'User',
    'Mediafile',
    'MediafileForm',
    function ($http, $scope, gettext, ngDialog, osTableFilter, osTableSort,
              ProjectionDefault, Projector, User, Mediafile, MediafileForm) {
        Mediafile.bindAll({}, $scope, 'mediafiles');
        User.bindAll({}, $scope, 'users');
        $scope.$watch(function() {
            return Projector.lastModified();
        }, function() {
            $scope.projectors = Projector.getAll();
            updatePresentedMediafiles();
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'mediafiles'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });

        function updatePresentedMediafiles () {
            $scope.presentedMediafiles = [];
            Projector.getAll().forEach(function (projector) {
                var projectorElements = _.map(projector.elements, function(element) { return element; });
                var mediaElements = _.filter(projectorElements, function (element) {
                    return element.name === 'mediafiles/mediafile';
                });
                mediaElements.forEach(function (element) {
                    $scope.presentedMediafiles.push(element);
                });
            });
            if ($scope.presentedMediafiles.length) {
                $scope.isMeta = false;
            } else {
                $scope.isMeta = true;
            }
        }

        updatePresentedMediafiles();

        // Filtering
        $scope.filter = osTableFilter.createInstance('MediafilesTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.booleanFilters = {
                isHidden: {
                    value: undefined,
                    displayName: gettext('Hidden'),
                    choiceYes: gettext('Is hidden'),
                    choiceNo: gettext('Is not hidden'),
                    needExtraPermission: true,
                },
                isPdf: {
                    value: undefined,
                    displayName: gettext('Is PDF'),
                    choiceYes: gettext('Is PDF file'),
                    choiceNo: gettext('Is no PDF file'),
                },
            };
        }
        $scope.filter.propertyList = ['title_or_filename'];
        $scope.filter.propertyFunctionList = [
            function (mediafile) {return mediafile.uploader.get_short_name();},
            function (mediafile) {return mediafile.mediafile.type;},
            function (mediafile) {return mediafile.mediafile.name;},
        ];
        // Sorting
        $scope.sort = osTableSort.createInstance();
        $scope.sort.column = 'title_or_filename';
        $scope.sortOptions = [
            {name: 'title_or_filename',
             display_name: gettext('Title')},
            {name: 'mediafile.type',
             display_name: gettext('Type')},
            {name: 'filesize',
             display_name: gettext('File size')},
            {name: 'timestamp',
             display_name: gettext('Upload time')},
            {name: 'uploader.get_short_name()',
             display_name: gettext('Uploaded by')},
        ];

        // open new/edit dialog
        $scope.openDialog = function (mediafile) {
            ngDialog.open(MediafileForm.getDialog(mediafile));
        };

        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            angular.forEach($scope.mediafiles, function (mediafile) {
                mediafile.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if SelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.mediafiles, function (mediafile) {
                    mediafile.selected = false;
                });
            }
        };
        // delete all selected mediafiles
        $scope.deleteMultiple = function () {
            angular.forEach($scope.mediafiles, function (mediafile) {
                if (mediafile.selected)
                    Mediafile.destroy(mediafile.id);
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete single mediafile
        $scope.delete = function (mediafile) {
            Mediafile.destroy(mediafile.id);
        };

        // ** PDF presentation functions **/
        // show document on projector
        $scope.showMediafile = function (projectorId, mediafile) {
            var isProjectedIds = mediafile.isProjected();
            _.forEach(isProjectedIds, function (id) {
                $http.post('/rest/core/projector/' + id + '/clear_elements/');
            });
            if (_.indexOf(isProjectedIds, projectorId) == -1) {
                var postUrl = '/rest/core/projector/' + projectorId + '/prune_elements/';
                var data = [{
                        name: 'mediafiles/mediafile',
                        id: mediafile.id,
                        numPages: mediafile.mediafile.pages,
                        page: 1,
                        scale: 'page-fit',
                        rotate: 0,
                        visible: true,
                        playing: false,
                        fullscreen: mediafile.is_pdf
                }];
                $http.post(postUrl, data);
            }
        };

        var sendMediafileCommand = function (mediafile, data) {
            var updateData = _.extend({}, mediafile);
            _.extend(updateData, data);
            var postData = {};
            postData[mediafile.uuid] = updateData;

            // Find Projector where the mediafile is projected
            $scope.projectors.forEach(function (projector) {
                if (_.find(projector.elements, function (e) {return e.uuid == mediafile.uuid;})) {
                    $http.post('/rest/core/projector/' + projector.id + '/update_elements/', postData);
                }
            });
        };

        $scope.getTitle = function (mediafile) {
            return Mediafile.get(mediafile.id).title;
        };

        $scope.getType = function (presentedMediafile) {
            var mediafile = Mediafile.get(presentedMediafile.id);
            return mediafile.is_pdf ? 'pdf' : mediafile.is_image ? 'image' : 'video';
        };

        $scope.mediafileGoToPage = function (mediafile, page) {
            if (parseInt(page) > 0) {
                sendMediafileCommand(
                    mediafile,
                    {page: parseInt(page)}
                );
            }
        };
        $scope.mediafileZoomIn = function (mediafile) {
            var scale = 1;
            if (parseFloat(mediafile.scale)) {
                scale = mediafile.scale;
            }
            sendMediafileCommand(
                mediafile,
                {scale: scale + 0.2}
            );
        };
        $scope.mediafileFit = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {scale: 'page-fit'}
            );
        };
        $scope.mediafileZoomOut = function (mediafile) {
            var scale = 1;
            if (parseFloat(mediafile.scale)) {
                scale = mediafile.scale;
            }
            sendMediafileCommand(
                mediafile,
                {scale: scale - 0.2}
            );
        };
        $scope.mediafileChangePage = function (mediafile, pageNum) {
            sendMediafileCommand(
                mediafile,
                {pageToDisplay: pageNum}
            );
        };
        $scope.mediafileRotate = function (mediafile) {
            var rotation = mediafile.rotate;
            if (rotation === 270) {
                rotation = 0;
            } else {
                rotation = rotation + 90;
            }
            sendMediafileCommand(
                mediafile,
                {rotate: rotation}
            );
        };
        $scope.mediafileToggleFullscreen = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {fullscreen: !mediafile.fullscreen}
            );
        };
        $scope.mediafileTogglePlaying = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {playing: !mediafile.playing}
            );
        };
    }
])

/*
 * Special filter only for mediafile list view.
 */
.filter('hiddenFilter', [
    '$filter',
    'operator',
    function ($filter, operator) {
        return function (array) {
            if (operator.hasPerms('mediafiles.can_see_hidden')) {
                return array;
            }
            return Array.prototype.filter.call(array, function (item) {
                return !item.hidden;
            });
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.projector', [
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for slidesProvider
])

.config([
    'slidesProvider',
    function (slidesProvider) {
        slidesProvider.registerSlide('mediafiles/mediafile', {
            template: 'static/templates/mediafiles/slide_mediafile.html'
        });
    }
])

.controller('SlideMediafileCtrl', [
    '$scope',
    'Mediafile',
    function ($scope, Mediafile) {
        // load mediafile object
        Mediafile.bindOne($scope.element.id, $scope, 'mediafile');

        // Allow the elements to render properly
        setTimeout(function() {
            if ($scope.mediafile) {
                if ($scope.mediafile.is_pdf) {
                    $scope.pdfName = $scope.mediafile.title;
                    $scope.pdfUrl = $scope.mediafile.mediafileUrl;
                } else if ($scope.mediafile.is_video) {
                    var player = angular.element.find('#video-player')[0];
                    if ($scope.element.playing) {
                        player.play();
                    } else {
                        player.pause();
                    }
                }
            }
        }, 0);
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.resources', [
    'gettext',
    'js-data',
    //TODO: Add deps for jsDataModel
])

.factory('Mediafile', [
    'DS',
    'gettext',
    'jsDataModel',
    function (DS, gettext, jsDataModel) {
        var name = 'mediafiles/mediafile';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Files'),
            verboseNamePlural: gettext('Files'),
            getAllImages: function () {
                var images = [];
                angular.forEach(this.getAll(), function(file) {
                    if (file.is_image) {
                        images.push({title: file.title, value: file.mediafileUrl});
                    }
                });
                return images;
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.title;
                },
                // return true if a specific relation matches for given searchquery
                // (here: speakers)
                hasSearchResult: function (results) {
                    var mediafile = this;
                    // search for speakers (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if ((result.getResourceName() === "users/user") &&
                                (mediafile.uploader_id === result.id)) {
                            return true;
                        }
                    });
                },
            },
            computed: {
                is_pdf: ['filetype', function (filetype) {
                    var PDF_FILE_TYPES = ['application/pdf'];
                    return _.includes(PDF_FILE_TYPES, filetype);
                }],
                is_image: ['filetype', function (filetype) {
                    var IMAGE_FILE_TYPES = ['image/png', 'image/jpeg', 'image/gif'];
                    return _.includes(IMAGE_FILE_TYPES, filetype);
                }],
                is_video: ['filetype', function (filetype) {
                    var VIDEO_FILE_TYPES = [ 'video/quicktime', 'video/mp4', 'video/webm',
                        'video/ogg', 'video/x-flv', 'application/x-mpegURL', 'video/MP2T',
                        'video/3gpp', 'video/x-msvideo', 'video/x-ms-wmv', 'video/x-matroska' ];
                    return _.includes(VIDEO_FILE_TYPES, filetype);
                }],
                is_presentable: ['is_pdf', 'is_image', 'is_video', function (is_pdf, is_image, is_video) {
                    return (is_pdf && !this.mediafile.encrypted) || is_image || is_video;
                }],
                mediafileUrl: [function () {
                    return this.media_url_prefix + this.mediafile.name;
                }],
                filename: [function () {
                    var filename = this.mediafile.name;
                    return /\/(.+?)$/.exec(filename)[1];
                }],
                filetype: [function () {
                    return this.mediafile.type;
                }],
                title_or_filename: ['title', 'mediafile', function (title) {
                    return title || this.filename;
                }]
            },
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'uploader',
                        localKey: 'uploader_id',
                    }
                }
            }
        });
    }
])

.run(['Mediafile', function (Mediafile) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.site', [
    'OpenSlidesApp.mediafiles.create',
    'OpenSlidesApp.mediafiles.list',
    'OpenSlidesApp.mediafiles.states',
    'OpenSlidesApp.mediafiles.update',
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.states', [
    'gettext',
    'ui.router',
    //TODO: Add deps for mainMenuProvider
])

.config([
    'gettext',
    'mainMenuProvider',
    function (gettext, mainMenuProvider) {
        mainMenuProvider.register({
            'ui_sref': 'mediafiles.mediafile.list',
            'img_class': 'paperclip',
            'title': gettext('Files'),
            'weight': 600,
            'perm': 'mediafiles.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Files'),
            'collectionName': 'mediafiles/mediafile',
            'urlDetailState': 'mediafiles.mediafile.detail',
            'weight': 600,
        });
    }
])

.config([
    'gettext',
    '$stateProvider',
    function (gettext, $stateProvider) {
        $stateProvider
        .state('mediafiles', {
            url: '/mediafiles',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Files'),
                basePerm: 'mediafiles.can_see',
            },
        })
        .state('mediafiles.mediafile', {
            abstract: true,
            template: "<ui-view/>",
        })
        .state('mediafiles.mediafile.list', {});
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.update', [
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for operator, User
])

.controller('MediafileUpdateCtrl', [
    '$scope',
    'operator',
    'User',
    'Mediafile',
    'mediafileId',
    'MediafileForm',
    'ErrorMessage',
    function ($scope, operator, User, Mediafile, mediafileId, MediafileForm, ErrorMessage) {
        $scope.alert = {};
        $scope.formFields = MediafileForm.getFormFields();

        // set initial values for form model by create deep copy of motion object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(Mediafile.get(mediafileId));

        // save mediafile
        $scope.save = function (mediafile) {
            // reset title and uploader_id if empty
            if (!mediafile.title) {
                mediafile.title = mediafile.filename;
            }
            if (!mediafile.uploader_id) {
                mediafile.uploader_id = operator.user.id;
            }
            // inject the changed mediafile (copy) object back into DS store
            Mediafile.inject(mediafile);
            // save change mediafile object on server
            Mediafile.save(mediafile).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    Mediafile.refresh(mediafile);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions', [
    'OpenSlidesApp.motions.motionBlock',
    'OpenSlidesApp.motions.lineNumbering',
    'OpenSlidesApp.motions.diff',
    'OpenSlidesApp.poll.majority',
    'OpenSlidesApp.users',
])

.factory('WorkflowState', [
    'DS',
    function (DS) {
        return DS.defineResource({
            name: 'motions/workflowstate',
            methods: {
                getNextStates: function () {
                    // TODO: Use filter with params with operator 'in'.
                    var states = [];
                    _.forEach(this.next_states_id, function (stateId) {
                        states.push(DS.get('motions/workflowstate', stateId));
                    });
                    return states;
                },
                getRecommendations: function () {
                    var params = {
                        where: {
                            'workflow_id': {
                                '==': this.workflow_id
                            },
                            'recommendation_label': {
                                '!=': null
                            }
                        }
                    };
                    return DS.filter('motions/workflowstate', params);
                }
            }
        });
    }
])

.factory('Workflow', [
    'DS',
    'jsDataModel',
    'WorkflowState',
    function (DS, jsDataModel, WorkflowState) {
        return DS.defineResource({
            name: 'motions/workflow',
            useClass: jsDataModel,
            relations: {
                hasMany: {
                    'motions/workflowstate': {
                        localField: 'states',
                        foreignKey: 'workflow_id',
                    }
                }
            }
        });
    }
])

.factory('MotionPoll', [
    'DS',
    'gettextCatalog',
    'Config',
    'MajorityMethods',
    function (DS, gettextCatalog, Config, MajorityMethods) {
        return DS.defineResource({
            name: 'motions/motion-poll',
            relations: {
                belongsTo: {
                    'motions/motion': {
                        localField: 'motion',
                        localKey: 'motion_id',
                    }
                }
            },
            methods: {
                // Returns percent base. Returns undefined if calculation is not possible in general.
                getPercentBase: function (config, type) {
                    var base;
                    switch (config) {
                        case 'CAST':
                            if (this.votescast <= 0 || this.votesinvalid < 0) {
                                // It would be OK to check only this.votescast < 0 because 0
                                // is checked again later but this is a little bit faster.
                                break;
                            }
                            base = this.votescast;
                            /* falls through */
                        case 'VALID':
                            if (this.votesvalid < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid') {
                                base = this.votesvalid;
                            }
                            /* falls through */
                        case 'YES_NO_ABSTAIN':
                            if (this.abstain < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                base = this.yes + this.no + this.abstain;
                            }
                            /* falls through */
                        case 'YES_NO':
                            if (this.yes < 0 || this.no < 0 || this.abstain === -1 ) {
                                // It is not allowed to set 'Abstain' to 'majority' but exclude it from calculation.
                                // Setting 'Abstain' to 'undocumented' is possible, of course.
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && (type === 'yes' || type === 'no')) {
                                base = this.yes + this.no;
                            }
                    }
                    return base;
                },

                // Returns object with value and percent for this poll.
                getVote: function (vote, type) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values
                    var value = '',
                        percentStr = '',
                        percentNumber,
                        config = Config.get('motions_poll_100_percent_base').value;

                    // Check special values
                    switch (vote) {
                        case -1:
                            value = gettextCatalog.getString('majority');
                            break;
                        case -2:
                            value = gettextCatalog.getString('undocumented');
                            break;
                        default:
                            if (vote >= 0) {
                                value = vote;
                            } else {
                                value = 0;  // Vote was not defined. Set value to 0.
                            }
                    }

                    // Calculate percent value
                    var base = this.getPercentBase(config, type);
                    if (base) {
                        percentNumber = Math.round(vote * 100 / (base) * 10) / 10;
                        percentStr = '(' + percentNumber + ' %)';
                    }
                    return {
                        'value': value,
                        'percentStr': percentStr,
                        'percentNumber': percentNumber,
                        'display': value + ' ' + percentStr
                    };
                },

                // Returns 0 or positive integer if quorum is reached or surpassed.
                // Returns negativ integer if quorum is not reached.
                // Returns undefined if we can not calculate the quorum.
                isReached: function (method) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    var isReached;
                    var config = Config.get('motions_poll_100_percent_base').value;
                    var base = this.getPercentBase(config, 'yes');
                    if (base) {
                        // Provide result only if base is not undefined and not 0.
                        isReached = MajorityMethods[method](this.yes, base);
                    }
                    return isReached;
                }
            }
        });
    }
])

.factory('Motion', [
    'DS',
    '$http',
    'MotionPoll',
    'MotionChangeRecommendation',
    'MotionComment',
    'jsDataModel',
    'gettext',
    'gettextCatalog',
    'Config',
    'lineNumberingService',
    'diffService',
    'OpenSlidesSettings',
    'Projector',
    'operator',
    function(DS, $http, MotionPoll, MotionChangeRecommendation, MotionComment, jsDataModel, gettext, gettextCatalog,
        Config, lineNumberingService, diffService, OpenSlidesSettings, Projector, operator) {
        var name = 'motions/motion';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Motion'),
            verboseNamePlural: gettext('Motions'),
            validate: function (resource, data, callback) {
                MotionComment.populateFieldsReverse(data);
                callback(null, data);
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                getVersion: function (versionId) {
                    versionId = versionId || this.active_version;
                    var index;
                    if (versionId == -1) {
                        index = this.versions.length - 1;
                    } else {
                        index = _.findIndex(this.versions, function (element) {
                            return element.id == versionId;
                        });
                    }
                    return this.versions[index] || {};
                },
                getTitle: function (versionId) {
                    return this.getVersion(versionId).title;
                },
                getText: function (versionId) {
                    return this.getVersion(versionId).text;
                },
                getTextWithLineBreaks: function (versionId, highlight, callback) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = this.getVersion(versionId).text;

                    return lineNumberingService.insertLineNumbers(html, lineLength, highlight, callback);
                },
                getTextBetweenChangeRecommendations: function (versionId, change1, change2, highlight) {
                    var line_from = (change1 ? change1.line_to : 1),
                        line_to = (change2 ? change2.line_from : null);

                    if (line_from > line_to) {
                        throw 'Invalid call of getTextBetweenChangeRecommendations: change1 needs to be before change2';
                    }
                    if (line_from == line_to) {
                        return '';
                    }

                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(this.getVersion(versionId).text, lineLength);

                    var data = diffService.extractRangeByLineNumbers(html, line_from, line_to);

                    // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                    html = diffService.addCSSClassToFirstTag(data.outerContextStart + data.innerContextStart, "merge-before") +
                        data.html + data.innerContextEnd + data.outerContextEnd;
                    html = lineNumberingService.insertLineNumbers(html, lineLength, highlight, null, line_from);

                    return html;
                },
                getTextRemainderAfterLastChangeRecommendation: function(versionId, changes, highlight) {
                    var maxLine = 0;
                    for (var i = 0; i < changes.length; i++) {
                        if (changes[i].line_to > maxLine) {
                            maxLine = changes[i].line_to;
                        }
                    }

                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(this.getVersion(versionId).text, lineLength);

                    var data = diffService.extractRangeByLineNumbers(html, maxLine, null);

                    if (data.html !== '') {
                        // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                        html = diffService.addCSSClassToFirstTag(data.outerContextStart + data.innerContextStart, "merge-before") +
                            data.html + data.innerContextEnd + data.outerContextEnd;
                        html = lineNumberingService.insertLineNumbers(html, lineLength, highlight, null, maxLine);
                    } else {
                        // Prevents empty lines at the end of the motion
                        html = '';
                    }
                    return html;
                },
                _getTextWithChangeRecommendations: function (versionId, highlight, statusCompareCb) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = this.getVersion(versionId).text,
                        changes = this.getChangeRecommendations(versionId, 'DESC');

                    for (var i = 0; i < changes.length; i++) {
                        var change = changes[i];
                        if (typeof statusCompareCb === 'undefined' || statusCompareCb(change.rejected)) {
                            html = lineNumberingService.insertLineNumbers(html, lineLength);
                            html = diffService.replaceLines(html, change.text, change.line_from, change.line_to);
                        }
                    }

                    return lineNumberingService.insertLineNumbers(html, lineLength, highlight);
                },
                getTextWithAllChangeRecommendations: function (versionId, highlight) {
                    return this._getTextWithChangeRecommendations(versionId, highlight, function() {
                        return true;
                    });
                },
                getTextWithoutRejectedChangeRecommendations: function (versionId, highlight) {
                    return this._getTextWithChangeRecommendations(versionId, highlight, function(rejected) {
                        return !rejected;
                    });
                },
                getTextByMode: function(mode, versionId, highlight) {
                    /*
                     * @param mode ['original', 'diff', 'changed', 'agreed']
                     * @param versionId [if undefined, active_version will be used]
                     * @param highlight [the line number to highlight]
                     */
                    var text;
                    switch (mode) {
                        case 'original':
                            text = this.getTextWithLineBreaks(versionId, highlight);
                            break;
                        case 'diff':
                            var changes = this.getChangeRecommendations(versionId, 'ASC');
                            text = '';
                            for (var i = 0; i < changes.length; i++) {
                                text += this.getTextBetweenChangeRecommendations(versionId, (i === 0 ? null : changes[i - 1]), changes[i], highlight);
                                text += changes[i].getDiff(this, versionId, highlight);
                            }
                            text += this.getTextRemainderAfterLastChangeRecommendation(versionId, changes);
                            break;
                        case 'changed':
                            text = this.getTextWithAllChangeRecommendations(versionId, highlight);
                            break;
                        case 'agreed':
                            text = this.getTextWithoutRejectedChangeRecommendations(versionId, highlight);
                            break;
                    }
                    return text;
                },
                setTextStrippingLineBreaks: function (text) {
                    this.text = lineNumberingService.stripLineNumbers(text);
                },
                getReason: function (versionId) {
                    return this.getVersion(versionId).reason;
                },
                // full state name - optional with custom state name extension
                // depended by state and provided by a custom comment field
                getStateName: function () {
                    var name = '';
                    var additionalName = '';
                    if (this.state) {
                        name = gettextCatalog.getString(this.state.name);
                        if (this.state.show_state_extension_field) {
                            // check motion comment fields for flag 'forState'
                            var fields = Config.get('motions_comments').value;
                            var index = _.findIndex(fields, ['forState', true]);
                            if (index > -1) {
                                additionalName = ' ' + this.comments[index];
                            }
                        }
                    }
                    return name + additionalName;
                },
                // full recommendation string - optional with custom recommendationextension
                // depended by state and provided by a custom comment field
                getRecommendationName: function () {
                    var recommendation = '';
                    var additionalName = '';
                    if (Config.get('motions_recommendations_by').value !== '' && this.recommendation) {
                        recommendation = gettextCatalog.getString(this.recommendation.recommendation_label);
                        if (this.recommendation.show_recommendation_extension_field) {
                            // check motion comment fields for flag 'forRecommendation'
                            var fields = Config.get('motions_comments').value;
                            var index = _.findIndex(fields, ['forRecommendation', true]);
                            if (index > -1) {
                                additionalName = ' ' + this.comments[index];
                            }
                        }
                    }
                    return recommendation + additionalName;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getTitle();
                },
                // return true if a specific relation matches for given searchquery
                // e.g. submitter, supporters or category
                hasSearchResult: function (results, searchquery) {
                    var motion = this;
                    // search for submitters and supporters (check if any user.id from already found users matches)
                    var foundSomething = _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(motion.submitters, {'id': result.id})) {
                                return true;
                            } else if (_.some(motion.supporters, { 'id': result.id })) {
                                return true;
                            }
                        }
                    });
                    // search for category
                    if (!foundSomething && motion.category && motion.category.name.match(new RegExp(searchquery, 'i'))) {
                        foundSomething = true;
                    }

                    // search for change recommendation
                    if (!foundSomething) {
                        var recommendations = MotionChangeRecommendation.filter({
                            where: {motion_version_id: this.active_version}
                        });
                        foundSomething = _.some(recommendations, function(recommendation) {
                            if (recommendation.text.match(new RegExp(searchquery, 'i'))) {
                                return true;
                            }
                        });
                    }
                    return foundSomething;
                },
                getChangeRecommendations: function (versionId, order) {
                    /*
                     * Returns all change recommendations for this given version, sorted by line
                     * @param versionId
                     * @param order ['DESC' or 'ASC' (default)]
                     * @returns {*}
                     */
                    versionId = versionId || this.active_version;
                    order = order || 'ASC';
                    return MotionChangeRecommendation.filter({
                        where: {
                            motion_version_id: versionId
                        },
                        orderBy: [
                            ['line_from', order]
                        ]
                    });
                },
                isAmendment: function () {
                    return this.parent_id !== null;
                },
                hasAmendments: function () {
                    return DS.filter('motions/motion', {parent_id: this.id}).length > 0;
                },
                isAllowed: function (action) {
                    /*
                     * Return true if the requested user is allowed to do the specific action.
                     * There are the following possible actions.
                     * - see
                     * - update
                     * - delete
                     * - create_poll
                     * - support
                     * - unsupport
                     * - change_state
                     * - reset_state
                     * - change_comments
                     * - change_recommendation
                     * - can_manage
                     * - can_see_amendments
                     * - can_create_amendments
                     *
                     *  NOTE: If you update this function please think about
                     *        server permissions, see motions/views.py.
                     */
                    switch (action) {
                        case 'see':
                            return (
                                operator.hasPerms('motions.can_see') &&
                                (
                                    !this.state.required_permission_to_see ||
                                    operator.hasPerms(this.state.required_permission_to_see) ||
                                    (operator.user in this.submitters)
                                )
                            );
                        case 'update':
                            return (
                                operator.hasPerms('motions.can_manage') ||
                                (
                                    (_.indexOf(this.submitters, operator.user) !== -1) &&
                                    this.state.allow_submitter_edit
                                )
                            );
                        case 'delete':
                            return operator.hasPerms('motions.can_manage');
                        case 'create_poll':
                            return (
                                operator.hasPerms('motions.can_manage') &&
                                this.state.allow_create_poll
                            );
                        case 'support':
                            return (
                                operator.hasPerms('motions.can_support') &&
                                this.state.allow_support &&
                                Config.get('motions_min_supporters').value > 0 &&
                                (_.indexOf(this.submitters, operator.user) === -1) &&
                                (_.indexOf(this.supporters, operator.user) === -1)
                            );
                        case 'unsupport':
                            return this.state.allow_support && _.indexOf(this.supporters, operator.user) !== -1;
                        case 'change_state':
                            return operator.hasPerms('motions.can_manage');
                        case 'reset_state':
                            return operator.hasPerms('motions.can_manage');
                        case 'change_comments':
                            return operator.hasPerms('motions.can_see_and_manage_comments');
                        case 'change_recommendation':
                            return operator.hasPerms('motions.can_manage');
                        case 'can_manage':
                            return operator.hasPerms('motions.can_manage');
                        case 'can_see_amendments':
                            var result;
                            if (operator.hasPerms('motions.can_create')) {
                                result = Config.get('motions_amendments_enabled').value &&
                                    (this.hasAmendments() || this.isAllowed('can_create_amendment'));
                            } else if (operator.hasPerms('motions.can_see')) {
                                result = Config.get('motions_amendments_enabled').value && this.hasAmendments();
                            }
                            return result;
                        case 'can_create_amendment':
                            return (
                                operator.hasPerms('motions.can_create') &&
                                Config.get('motions_amendments_enabled').value &&
                                ( !this.isAmendment() ||
                                  (this.isAmendment() && OpenSlidesSettings.MOTIONS_ALLOW_AMENDMENTS_OF_AMENDMENTS))
                            );
                        default:
                            return false;
                    }
                },
                /* Overrides from jsDataModel factory.
                 * Also sets the projection mode if given; If not it projects in 'original' mode. */
                project: function (projectorId, mode) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var isProjected = this.isProjectedWithMode();
                    _.forEach(isProjected, function (mapping) {
                        $http.post('/rest/core/projector/' + mapping.projectorId + '/clear_elements/');
                    });
                    // Was there a projector with the same id and mode as the given id and mode?
                    // If not, project the motion.
                    var wasProjectedBefore = _.some(isProjected, function (mapping) {
                        var value = (mapping.projectorId === projectorId);
                        if (mode) {
                            value = value && (mapping.mode === mode);
                        }
                        return value;
                    });
                    mode = mode || 'original';
                    if (!wasProjectedBefore) {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/prune_elements/',
                            [{name: name,
                              id: this.id,
                              mode: mode}]
                        );
                    }
                },
                isProjected: function (mode) {
                    var self = this;
                    var predicate = function (element) {
                        var value = element.name === name &&
                            element.id === self.id;
                        if (mode) {
                            value = value && (element.mode === mode);
                        }
                        return value;
                    };
                    var projectorIds = [];
                    _.forEach(Projector.getAll(), function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            projectorIds.push(projector.id);
                        }
                    });
                    return projectorIds;
                },
                /* returns a list of mappings between projector id and mode:
                 * [ {projectorId: 2, mode: 'original'}, ... ] */
                isProjectedWithMode: function () {
                    var self = this;
                    var mapping = [];
                    _.forEach(Projector.getAll(), function (projector) {
                        _.forEach(projector.elements, function (element) {
                            if (element.name === name && element.id === self.id) {
                                mapping.push({
                                    projectorId: projector.id,
                                    mode: element.mode || 'original',
                                });
                            }
                        });
                    });
                    return mapping;
                },
            },
            relations: {
                belongsTo: {
                    'motions/category': {
                        localField: 'category',
                        localKey: 'category_id',
                    },
                    'motions/motion-block': {
                        localField: 'motionBlock',
                        localKey: 'motion_block_id',
                    },
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'mediafiles/mediafile': {
                        localField: 'attachments',
                        localKeys: 'attachments_id',
                    },
                    'users/user': [
                        {
                            localField: 'submitters',
                            localKeys: 'submitters_id',
                        },
                        {
                            localField: 'supporters',
                            localKeys: 'supporters_id',
                        }
                    ],
                    'motions/motion-poll': {
                        localField: 'polls',
                        foreignKey: 'motion_id',
                    }
                },
                hasOne: {
                    'motions/workflowstate': [
                        {
                            localField: 'state',
                            localKey: 'state_id',
                        },
                        {
                            localField: 'recommendation',
                            localKey: 'recommendation_id',
                        }
                    ]
                }
            }
        });
    }
])

// Service for generic comment fields
.factory('MotionComment', [
    'Config',
    'operator',
    'Editor',
    function (Config, operator, Editor) {
        return {
            getFormFields: function () {
                var fields = Config.get('motions_comments').value;
                return _.map(
                    _.filter(fields, function(element) { return !element.forState && !element.forRecommendation; }),
                    function (field) {
                        return {
                            key: 'comment ' + field.name,
                            type: 'editor',
                            templateOptions: {
                                label: field.name,
                            },
                            data: {
                                ckeditorOptions: Editor.getOptions()
                            },
                            hide: !operator.hasPerms("motions.can_see_and_manage_comments")
                        };
                    }
                );
            },
            populateFields: function (motion) {
                // Populate content of motion.comments to the single comment
                // fields like motion['comment MyComment'], motion['comment MyOtherComment'], ...
                var fields = Config.get('motions_comments').value;
                if (!motion.comments) {
                    motion.comments = [];
                }
                for (var i = 0; i < fields.length; i++) {
                    motion['comment ' + fields[i].name] = motion.comments[i];
                }
            },
            populateFieldsReverse: function (motion) {
                // Reverse equivalent to populateFields.
                var fields = Config.get('motions_comments').value;
                motion.comments = [];
                for (var i = 0; i < fields.length; i++) {
                    motion.comments.push(motion['comment ' + fields[i].name] || '');
                }
            },
            getFieldNameForFlag: function (flag) {
                var fields = Config.get('motions_comments').value;
                var fieldName = '';
                var index = _.findIndex(fields, [flag, true]);
                if (index > -1) {
                    fieldName = fields[index].name;
                }
                return fieldName;
            },
        };
    }
])

.factory('Category', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'motions/category',
        });
    }
])

.factory('MotionList', [
    function () {
        return {
            getList: function (items){
                var list = [];
                _.each(items, function (item) {
                    list.push({ id: item.id,
                                item: item });
                });
                return list;
            }
        };
    }
])

.factory('MotionChangeRecommendation', [
    'DS',
    'Config',
    'jsDataModel',
    'diffService',
    'lineNumberingService',
    'gettextCatalog',
    function (DS, Config, jsDataModel, diffService, lineNumberingService, gettextCatalog) {
        return DS.defineResource({
            name: 'motions/motion-change-recommendation',
            useClass: jsDataModel,
            methods: {
                saveStatus: function() {
                    this.DSSave();
                },
                getDiff: function(motion, version, highlight) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(motion.getVersion(version).text, lineLength);

                    var data = diffService.extractRangeByLineNumbers(html, this.line_from, this.line_to),
                        oldText = data.outerContextStart + data.innerContextStart +
                            data.html + data.innerContextEnd + data.outerContextEnd;

                    oldText = lineNumberingService.insertLineNumbers(oldText, lineLength, null, null, this.line_from);
                    var diff = diffService.diff(oldText, this.text);

                    // If an insertion makes the line longer than the line length limit, we need two line breaking runs:
                    // - First, for the official line numbers, ignoring insertions (that's been done some lines before)
                    // - Second, another one to prevent the displayed including insertions to exceed the page width
                    diff = lineNumberingService.insertLineBreaksWithoutNumbers(diff, lineLength, true);

                    if (highlight > 0) {
                        diff = lineNumberingService.highlightLine(diff, highlight);
                    }

                    var origBeginning = data.outerContextStart + data.innerContextStart;
                    if (diff.toLowerCase().indexOf(origBeginning.toLowerCase()) === 0) {
                        // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                        diff = diffService.addCSSClassToFirstTag(origBeginning, "merge-before") + diff.substring(origBeginning.length);
                    }

                    return diff;
                },
                getType: function(original_full_html) {
                    return this.type;
                },
                getTitle: function(original_full_html) {
                    var title;
                    if (this.line_to > (this.line_from + 1)) {
                        title = gettextCatalog.getString('%TYPE% from line %FROM% to %TO%');
                    } else {
                        title = gettextCatalog.getString('%TYPE% in line %FROM%');
                    }
                    switch (this.getType(original_full_html)) {
                        case diffService.TYPE_INSERTION:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Insertion'));
                            break;
                        case diffService.TYPE_DELETION:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Deletion'));
                            break;
                        case diffService.TYPE_REPLACEMENT:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Replacement'));
                            break;
                    }
                    title = title.replace('%FROM%', this.line_from).replace('%TO%', (this.line_to - 1));
                    return title;
                }
            }
        });
    }
])

.run([
    'Motion',
    'Category',
    'Workflow',
    'MotionChangeRecommendation',
    function(Motion, Category, Workflow, MotionChangeRecommendation) {}
])


// Mark all motion workflow state strings for translation in JavaScript.
// (see motions/signals.py)
.config([
    'gettext',
    function (gettext) {
        // workflow 1
        gettext('Simple Workflow');
        gettext('submitted');
        gettext('accepted');
        gettext('Accept');
        gettext('Acceptance');
        gettext('rejected');
        gettext('Reject');
        gettext('Rejection');
        gettext('not decided');
        gettext('Do not decide');
        gettext('No decision');
        // workflow 2
        gettext('Complex Workflow');
        gettext('published');
        gettext('permitted');
        gettext('Permit');
        gettext('Permission');
        gettext('accepted');
        gettext('Accept');
        gettext('Acceptance');
        gettext('rejected');
        gettext('Reject');
        gettext('Rejection');
        gettext('withdrawed');
        gettext('Withdraw');
        gettext('adjourned');
        gettext('Adjourn');
        gettext('Adjournment');
        gettext('not concerned');
        gettext('Do not concern');
        gettext('No concernment');
        gettext('refered to committee');
        gettext('Refer to committee');
        gettext('Referral to committee');
        gettext('needs review');
        gettext('Needs review');
        gettext('rejected (not authorized)');
        gettext('Reject (not authorized)');
        gettext('Rejection (not authorized)');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.csv', [])

.factory('MotionCsvExport', [
    'gettextCatalog',
    'CsvDownload',
    function (gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Identifier', 'Title', 'Text', 'Reason', 'Submitter', 'Category', 'Origin'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (motions) {
                var csvRows = [
                    makeHeaderline()
                ];
                _.forEach(motions, function (motion) {
                    var row = [];
                    row.push('"' + motion.identifier !== null ? motion.identifier : '' + '"');
                    row.push('"' + motion.getTitle() + '"');
                    row.push('"' + motion.getText() + '"');
                    row.push('"' + motion.getReason() + '"');
                    var submitter = motion.submitters[0] ? motion.submitters[0].get_full_name() : '';
                    row.push('"' + submitter + '"');
                    var category = motion.category ? motion.category.name : '';
                    row.push('"' + category + '"');
                    row.push('"' + motion.origin + '"');
                    csvRows.push(row);
                });
                CsvDownload(csvRows, 'motions-export.csv');
            },
            downloadExample: function () {
                var csvRows = [makeHeaderline(),
                    // example entries
                    ['A1', 'Title 1', 'Text 1', 'Reason 1', 'Submitter A', 'Category A', 'Last Year Conference A'],
                    ['B1', 'Title 2', 'Text 2', 'Reason 2', 'Submitter B', 'Category B', ''],
                    [''  , 'Title 3', 'Text 3', '', '', '', ''],
                ];
                CsvDownload(csvRows, 'motions-example.csv');
            },
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.diff', ['OpenSlidesApp.motions.lineNumbering'])

.service('diffService', [
    'lineNumberingService',
    '$cacheFactory',
    function (lineNumberingService, $cacheFactory) {
        var ELEMENT_NODE = 1,
            TEXT_NODE = 3,
            DOCUMENT_FRAGMENT_NODE = 11;

        var diffCache = $cacheFactory('diff.service');

        this.TYPE_REPLACEMENT = 0;
        this.TYPE_INSERTION = 1;
        this.TYPE_DELETION = 2;

        this.getLineNumberNode = function(fragment, lineNumber) {
            return fragment.querySelector('os-linebreak.os-line-number.line-number-' + lineNumber);
        };

        this._getNodeContextTrace = function(node) {
            var context = [],
                currNode = node;
            while (currNode) {
                context.unshift(currNode);
                currNode = currNode.parentNode;
            }
            return context;
        };

        // Adds elements like <OS-LINEBREAK class="os-line-number line-number-23" data-line-number="23"/>
        this._insertInternalLineMarkers = function(fragment) {
            if (fragment.querySelectorAll('OS-LINEBREAK').length > 0) {
                // Prevent duplicate calls
                return;
            }
            var lineNumbers = fragment.querySelectorAll('span.os-line-number'),
                lineMarker, maxLineNumber;

            for (var i = 0; i < lineNumbers.length; i++) {
                var insertBefore = lineNumbers[i];
                while (insertBefore.parentNode.nodeType != DOCUMENT_FRAGMENT_NODE && insertBefore.parentNode.childNodes[0] == insertBefore) {
                    insertBefore = insertBefore.parentNode;
                }
                lineMarker = document.createElement('OS-LINEBREAK');
                lineMarker.setAttribute('data-line-number', lineNumbers[i].getAttribute('data-line-number'));
                lineMarker.setAttribute('class', lineNumbers[i].getAttribute('class'));
                insertBefore.parentNode.insertBefore(lineMarker, insertBefore);
                maxLineNumber = lineNumbers[i].getAttribute('data-line-number');
            }

            // Add one more "fake" line number at the end and beginning, so we can select the last line as well
            lineMarker = document.createElement('OS-LINEBREAK');
            lineMarker.setAttribute('data-line-number', (parseInt(maxLineNumber) + 1));
            lineMarker.setAttribute('class', 'os-line-number line-number-' + (parseInt(maxLineNumber) + 1));
            fragment.appendChild(lineMarker);

            lineMarker = document.createElement('OS-LINEBREAK');
            lineMarker.setAttribute('data-line-number', '0');
            lineMarker.setAttribute('class', 'os-line-number line-number-0');
            fragment.insertBefore(lineMarker, fragment.firstChild);
        };

        // @TODO Check if this is actually necessary
        this._insertInternalLiNumbers = function(fragment) {
            if (fragment.querySelectorAll('LI[os-li-number]').length > 0) {
                // Prevent duplicate calls
                return;
            }
            var ols = fragment.querySelectorAll('OL');
            for (var i = 0; i < ols.length; i++) {
                var ol = ols[i],
                    liNo = 0;
                for (var j = 0; j < ol.childNodes.length; j++) {
                    if (ol.childNodes[j].nodeName == 'LI') {
                        liNo++;
                        ol.childNodes[j].setAttribute('os-li-number', liNo);
                    }
                }
            }
        };

        this._addStartToOlIfNecessary = function(node) {
            var firstLiNo = null;
            for (var i = 0; i < node.childNodes.length && firstLiNo === null; i++) {
                if (node.childNode[i].nodeName == 'LI') {
                    var lineNo = node.childNode[i].getAttribute('ol-li-number');
                    if (lineNo) {
                        firstLiNo = parseInt(lineNo);
                    }
                }
            }
            if (firstLiNo > 1) {
                node.setAttribute('start', firstLiNo);
            }
        };

        this._isWithinNthLIOfOL = function(olNode, descendantNode) {
            var nthLIOfOL = null;
            while (descendantNode.parentNode) {
                if (descendantNode.parentNode == olNode) {
                    var lisBeforeOl = 0,
                        foundMe = false;
                    for (var i = 0; i < olNode.childNodes.length && !foundMe; i++) {
                        if (olNode.childNodes[i] == descendantNode) {
                            foundMe = true;
                        } else if (olNode.childNodes[i].nodeName == 'LI') {
                            lisBeforeOl++;
                        }
                    }
                    nthLIOfOL = lisBeforeOl + 1;
                }
                descendantNode = descendantNode.parentNode;
            }
            return nthLIOfOL;
        };

       /*
        * Returns an array with the following values:
        * 0: the most specific DOM-node that contains both line numbers
        * 1: the context of node1 (an array of dom-elements; 0 is the document fragment)
        * 2: the context of node2 (an array of dom-elements; 0 is the document fragment)
        * 3: the index of [0] in the two arrays
        */
        this._getCommonAncestor = function(node1, node2) {
            var trace1 = this._getNodeContextTrace(node1),
                trace2 = this._getNodeContextTrace(node2),
                commonAncestor = null,
                commonIndex = null,
                childTrace1 = [],
                childTrace2 = [];

            for (var i = 0; i < trace1.length && i < trace2.length; i++) {
                if (trace1[i] == trace2[i]) {
                    commonAncestor = trace1[i];
                    commonIndex = i;
                }
            }
            for (i = commonIndex + 1; i < trace1.length; i++) {
                childTrace1.push(trace1[i]);
            }
            for (i = commonIndex + 1; i < trace2.length; i++) {
                childTrace2.push(trace2[i]);
            }
            return {
                'commonAncestor': commonAncestor,
                'trace1' : childTrace1,
                'trace2' : childTrace2,
                'index': commonIndex
            };
        };

        this._serializeTag = function(node) {
            if (node.nodeType == DOCUMENT_FRAGMENT_NODE) {
                // Fragments are only placeholders and do not have an HTML representation
                return '';
            }
            var html = '<' + node.nodeName;
            for (var i = 0; i < node.attributes.length; i++) {
                var attr = node.attributes[i];
                if (attr.name != 'os-li-number') {
                    html += ' ' + attr.name + '="' + attr.value + '"';
                }
            }
            html += '>';
            return html;
        };

        this._serializeDom = function(node, stripLineNumbers) {
            if (node.nodeType == TEXT_NODE) {
                return node.nodeValue.replace(/</g, "&lt;").replace(/>/g, "&gt;");
            }
            if (stripLineNumbers && (
                lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node))) {
                return '';
            }
            if (node.nodeName == 'OS-LINEBREAK') {
                return '';
            }
            if (node.nodeName == 'BR') {
                var br = '<BR';
                for (i = 0; i < node.attributes.length; i++) {
                    var attr = node.attributes[i];
                    br += " " + attr.name + "=\"" + attr.value + "\"";
                }
                return br + '>';
            }

            var html = this._serializeTag(node);
            for (var i = 0; i < node.childNodes.length; i++) {
                if (node.childNodes[i].nodeType == TEXT_NODE) {
                    html += node.childNodes[i].nodeValue.replace(/</g, "&lt;").replace(/>/g, "&gt;");
                } else if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) && !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                    html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                }
            }
            if (node.nodeType != DOCUMENT_FRAGMENT_NODE) {
                html += '</' + node.nodeName + '>';
            }

            return html;
        };

        /**
         * Implementation hint: the first element of "toChildTrace" array needs to be a child element of "node"
         */
        this._serializePartialDomToChild = function(node, toChildTrace, stripLineNumbers) {
            if (lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node)) {
                return '';
            }
            if (node.nodeName == 'OS-LINEBREAK') {
                return '';
            }

            var html = this._serializeTag(node);

            for (var i = 0, found = false; i < node.childNodes.length && !found; i++) {
                if (node.childNodes[i] == toChildTrace[0]) {
                    found = true;
                    var remainingTrace = toChildTrace;
                    remainingTrace.shift();
                    if (!lineNumberingService._isOsLineNumberNode(node.childNodes[i])) {
                        html += this._serializePartialDomToChild(node.childNodes[i], remainingTrace, stripLineNumbers);
                    }
                } else if (node.childNodes[i].nodeType == TEXT_NODE) {
                    html += node.childNodes[i].nodeValue;
                } else {
                    if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) &&
                      !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                        html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                    }
                }
            }
            if (!found) {
                console.trace();
                throw "Inconsistency or invalid call of this function detected (to)";
            }
            return html;
        };

        /**
         * Implementation hint: the first element of "toChildTrace" array needs to be a child element of "node"
         */
        this._serializePartialDomFromChild = function(node, fromChildTrace, stripLineNumbers) {
            if (lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node)) {
                return '';
            }
            if (node.nodeName == 'OS-LINEBREAK') {
                return '';
            }

            var html = '';
            for (var i = 0, found = false; i < node.childNodes.length; i++) {
                if (node.childNodes[i] == fromChildTrace[0]) {
                    found = true;
                    var remainingTrace = fromChildTrace;
                    remainingTrace.shift();
                    if (!lineNumberingService._isOsLineNumberNode(node.childNodes[i])) {
                        html += this._serializePartialDomFromChild(node.childNodes[i], remainingTrace, stripLineNumbers);
                    }
                } else if (found) {
                    if (node.childNodes[i].nodeType == TEXT_NODE) {
                        html += node.childNodes[i].nodeValue;
                    } else {
                        if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) &&
                          !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                            html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                        }
                    }
                }
            }
            if (!found) {
                console.trace();
                throw "Inconsistency or invalid call of this function detected (from)";
            }
            if (node.nodeType != DOCUMENT_FRAGMENT_NODE) {
                html += '</' + node.nodeName + '>';
            }
            return html;
        };

        this.htmlToFragment = function(html) {
            var fragment = document.createDocumentFragment(),
                div = document.createElement('DIV');
            div.innerHTML = html;
            while (div.childElementCount) {
                var child = div.childNodes[0];
                div.removeChild(child);
                fragment.appendChild(child);
            }
            return fragment;
        };

        /**
         * Returns the HTML snippet between two given line numbers.
         *
         * Hint:
         * - The last line (toLine) is not included anymore, as the number refers to the line breaking element
         * - if toLine === null, then everything from fromLine to the end of the fragment is returned
         *
         * In addition to the HTML snippet, additional information is provided regarding the most specific DOM element
         * that contains the whole section specified by the line numbers (like a P-element if only one paragraph is selected
         * or the most outer DIV, if multiple sections selected).
         *
         * This additional information is meant to render the snippet correctly without producing broken HTML
         *
         * The return object has the following fields:
         * - html: The HTML between the two line numbers.
         *         Line numbers and automatically set line breaks are stripped.
         *         All HTML tags are converted to uppercase
         *         (e.g. Line 2</LI><LI>Line3</LI><LI>Line 4 <br>)
         * - ancestor: the most specific DOM element that contains the HTML snippet (e.g. a UL, if several LIs are selected)
         * - outerContextStart: An HTML string that opens all necessary tags to get the browser into the rendering mode
         *                      of the ancestor element (e.g. <DIV><UL> in the case of the multiple LIs)
         * - outerContectEnd:   An HTML string that closes all necessary tags from the ancestor element (e.g. </UL></DIV>
         * - innerContextStart: A string that opens all necessary tags between the ancestor
         *                      and the beginning of the selection (e.g. <LI>)
         * - innerContextEnd:   A string that closes all tags after the end of the selection to the ancestor (e.g. </LI>)
         * - previousHtml:      The HTML before the selected area begins (including line numbers)
         * - previousHtmlEndSnippet: A HTML snippet that closes all open tags from previousHtml
         * - followingHtml:     The HTML after the selected area
         * - followingHtmlStartSnippet: A HTML snippet that opens all HTML tags necessary to render "followingHtml"
         *
         */
        this.extractRangeByLineNumbers = function(htmlIn, fromLine, toLine, debug) {
            if (typeof(htmlIn) !== 'string') {
                throw 'Invalid call - extractRangeByLineNumbers expects a string as first argument';
            }

            var cacheKey = fromLine + "-" + toLine + "-" + lineNumberingService.djb2hash(htmlIn),
                cached = diffCache.get(cacheKey);

            if (!angular.isUndefined(cached)) {
                return cached;
            }

            var fragment = this.htmlToFragment(htmlIn);

            this._insertInternalLineMarkers(fragment);
            this._insertInternalLiNumbers(fragment);
            if (toLine === null) {
                var internalLineMarkers = fragment.querySelectorAll('OS-LINEBREAK');
                toLine = parseInt(internalLineMarkers[internalLineMarkers.length - 1].getAttribute("data-line-number"));
            }

            var fromLineNode = this.getLineNumberNode(fragment, fromLine),
                toLineNode = (toLine ? this.getLineNumberNode(fragment, toLine) : null),
                ancestorData = this._getCommonAncestor(fromLineNode, toLineNode);

            var fromChildTraceRel = ancestorData.trace1,
                fromChildTraceAbs = this._getNodeContextTrace(fromLineNode),
                toChildTraceRel = ancestorData.trace2,
                toChildTraceAbs = this._getNodeContextTrace(toLineNode),
                ancestor = ancestorData.commonAncestor,
                htmlOut = '',
                outerContextStart = '',
                outerContextEnd = '',
                innerContextStart = '',
                innerContextEnd = '',
                previousHtmlEndSnippet = '',
                followingHtmlStartSnippet = '',
                fakeOl;


            fromChildTraceAbs.shift();
            var previousHtml = this._serializePartialDomToChild(fragment, fromChildTraceAbs, false);
            toChildTraceAbs.shift();
            var followingHtml = this._serializePartialDomFromChild(fragment, toChildTraceAbs, false);

            var currNode = fromLineNode.parentNode;
            while (currNode.parentNode) {
                previousHtmlEndSnippet += '</' + currNode.nodeName + '>';
                currNode = currNode.parentNode;
            }
            currNode = toLineNode.parentNode;
            while (currNode.parentNode) {
                followingHtmlStartSnippet = this._serializeTag(currNode) + followingHtmlStartSnippet;
                currNode = currNode.parentNode;
            }

            var found = false;
            for (var i = 0; i < fromChildTraceRel.length && !found; i++) {
                if (fromChildTraceRel[i].nodeName == 'OS-LINEBREAK') {
                    found = true;
                } else {
                    if (fromChildTraceRel[i].nodeName == 'OL') {
                        fakeOl = fromChildTraceRel[i].cloneNode(false);
                        fakeOl.setAttribute('start', this._isWithinNthLIOfOL(fromChildTraceRel[i], fromLineNode));
                        innerContextStart += this._serializeTag(fakeOl);
                    } else {
                        innerContextStart += this._serializeTag(fromChildTraceRel[i]);
                    }
                }
            }
            found = false;
            for (i = 0; i < toChildTraceRel.length && !found; i++) {
                if (toChildTraceRel[i].nodeName == 'OS-LINEBREAK') {
                    found = true;
                } else {
                    innerContextEnd = '</' + toChildTraceRel[i].nodeName + '>' + innerContextEnd;
                }
            }

            found = false;
            for (i = 0; i < ancestor.childNodes.length; i++) {
                if (ancestor.childNodes[i] == fromChildTraceRel[0]) {
                    found = true;
                    fromChildTraceRel.shift();
                    htmlOut += this._serializePartialDomFromChild(ancestor.childNodes[i], fromChildTraceRel, true);
                } else if (ancestor.childNodes[i] == toChildTraceRel[0]) {
                    found = false;
                    toChildTraceRel.shift();
                    htmlOut += this._serializePartialDomToChild(ancestor.childNodes[i], toChildTraceRel, true);
                } else if (found === true) {
                    htmlOut += this._serializeDom(ancestor.childNodes[i], true);
                }
            }

            currNode = ancestor;
            while (currNode.parentNode) {
                if (currNode.nodeName == 'OL') {
                    fakeOl = currNode.cloneNode(false);
                    fakeOl.setAttribute('start', this._isWithinNthLIOfOL(currNode, fromLineNode));
                    outerContextStart = this._serializeTag(fakeOl) + outerContextStart;
                } else {
                    outerContextStart = this._serializeTag(currNode) + outerContextStart;
                }
                outerContextEnd += '</' + currNode.nodeName + '>';
                currNode = currNode.parentNode;
            }

            var ret = {
                'html': htmlOut,
                'ancestor': ancestor,
                'outerContextStart': outerContextStart,
                'outerContextEnd': outerContextEnd,
                'innerContextStart': innerContextStart,
                'innerContextEnd': innerContextEnd,
                'previousHtml': previousHtml,
                'previousHtmlEndSnippet': previousHtmlEndSnippet,
                'followingHtml': followingHtml,
                'followingHtmlStartSnippet': followingHtmlStartSnippet
            };

            diffCache.put(cacheKey, ret);
            return ret;
        };

        /*
         * This is a workardoun to prevent the last word of the inserted text from accidently being merged with the
         * first word of the following line.
         *
         * This happens as trailing spaces in the change recommendation's text are frequently stripped,
         * which is pretty nasty if the original text goes on after the affected line. So we insert a space
         * if the original line ends with one.
         */
        this._insertDanglingSpace = function(element) {
            if (element.childNodes.length > 0) {
                var lastChild = element.childNodes[element.childNodes.length - 1];
                if (lastChild.nodeType == TEXT_NODE && !lastChild.nodeValue.match(/[\S]/) && element.childNodes.length > 1) {
                    // If the text node only contains whitespaces, chances are high it's just space between block elmeents,
                    // like a line break between </LI> and </UL>
                    lastChild = element.childNodes[element.childNodes.length - 2];
                }
                if (lastChild.nodeType == TEXT_NODE) {
                    if (lastChild.nodeValue === '' || lastChild.nodeValue.substr(-1) != ' ') {
                        lastChild.nodeValue += ' ';
                    }
                } else {
                    this._insertDanglingSpace(lastChild);
                }
            }
        };

        /*
         * This functions merges to arrays of nodes. The last element of nodes1 and the first element of nodes2
         * are merged, if they are of the same type.
         *
         * This is done recursively until a TEMPLATE-Tag is is found, which was inserted in this.replaceLines.
         * Using a TEMPLATE-Tag is a rather dirty hack, as it is allowed inside of any other element, including <ul>.
         *
         */
        this._replaceLinesMergeNodeArrays = function(nodes1, nodes2) {
            if (nodes1.length === 0) {
                return nodes2;
            }
            if (nodes2.length === 0) {
                return nodes1;
            }

            var out = [];
            for (var i = 0; i < nodes1.length - 1; i++) {
                out.push(nodes1[i]);
            }

            var lastNode = nodes1[nodes1.length - 1],
                firstNode = nodes2[0];
            if (lastNode.nodeType == TEXT_NODE && firstNode.nodeType == TEXT_NODE) {
                var newTextNode = lastNode.ownerDocument.createTextNode(lastNode.nodeValue + firstNode.nodeValue);
                out.push(newTextNode);
            } else if (lastNode.nodeName == firstNode.nodeName) {
                var newNode = lastNode.ownerDocument.createElement(lastNode.nodeName);
                for (i = 0; i < lastNode.attributes.length; i++) {
                    var attr = lastNode.attributes[i];
                    newNode.setAttribute(attr.name, attr.value);
                }

                // Remove #text nodes inside of List elements, as they are confusing
                var lastChildren, firstChildren;
                if (lastNode.nodeName == 'OL' || lastNode.nodeName == 'UL') {
                    lastChildren = [];
                    firstChildren = [];
                    for (i = 0; i < firstNode.childNodes.length; i++) {
                        if (firstNode.childNodes[i].nodeType == ELEMENT_NODE) {
                            firstChildren.push(firstNode.childNodes[i]);
                        }
                    }
                    for (i = 0; i < lastNode.childNodes.length; i++) {
                        if (lastNode.childNodes[i].nodeType == ELEMENT_NODE) {
                            lastChildren.push(lastNode.childNodes[i]);
                        }
                    }
                } else {
                    lastChildren = lastNode.childNodes;
                    firstChildren = firstNode.childNodes;
                }

                var children = this._replaceLinesMergeNodeArrays(lastChildren, firstChildren);
                for (i = 0; i < children.length; i++) {
                    newNode.appendChild(children[i]);
                }
                out.push(newNode);
            } else {
                if (lastNode.nodeName != 'TEMPLATE') {
                    out.push(lastNode);
                }
                if (firstNode.nodeName != 'TEMPLATE') {
                    out.push(firstNode);
                }
            }

            for (i = 1; i < nodes2.length; i++) {
                out.push(nodes2[i]);
            }

            return out;
        };

        /**
         *
         * @param {string} html
         * @returns {string}
         * @private
         */
        this._normalizeHtmlForDiff = function (html) {
            // Convert all HTML tags to uppercase, strip trailing whitespaces
            html = html.replace(/<[^>]+>/g, function (tag) {
                return tag.toUpperCase();
            });

            var entities = {
                '&nbsp;': ' ',
                '&ndash;': '-',
                '&auml;': 'ä',
                '&ouml;': 'ö',
                '&uuml;': 'ü',
                '&Auml;': 'Ä',
                '&Ouml;': 'Ö',
                '&Uuml;': 'Ü',
                '&szlig;': 'ß'
            };

            html = html.replace(/\s+<\/P>/gi, '</P>').replace(/\s+<\/DIV>/gi, '</DIV>').replace(/\s+<\/LI>/gi, '</LI>');
            html = html.replace(/\s+<LI>/gi, '<LI>').replace(/<\/LI>\s+/gi, '</LI>');
            html = html.replace(/\u00A0/g, ' ');
            html = html.replace(/\u2013/g, '-');
            for (var ent in entities) {
                html = html.replace(new RegExp(ent, 'g'), entities[ent]);
            }

            return html;
        };

        /**
         * @param {string} htmlOld
         * @param {string} htmlNew
         * @returns {number}
         */
        this.detectReplacementType = function (htmlOld, htmlNew) {
            htmlOld = this._normalizeHtmlForDiff(htmlOld);
            htmlNew = this._normalizeHtmlForDiff(htmlNew);

            if (htmlOld == htmlNew) {
                return this.TYPE_REPLACEMENT;
            }

            var i, foundDiff;
            for (i = 0, foundDiff = false; i < htmlOld.length && i < htmlNew.length && foundDiff === false; i++) {
                if (htmlOld[i] != htmlNew[i]) {
                    foundDiff = true;
                }
            }

            var remainderOld = htmlOld.substr(i - 1),
                remainderNew = htmlNew.substr(i - 1),
                type = this.TYPE_REPLACEMENT;

            if (remainderOld.length > remainderNew.length) {
                if (remainderOld.substr(remainderOld.length - remainderNew.length) == remainderNew) {
                    type = this.TYPE_DELETION;
                }
            } else if (remainderOld.length < remainderNew.length) {
                if (remainderNew.substr(remainderNew.length - remainderOld.length) == remainderOld) {
                    type = this.TYPE_INSERTION;
                }
            }

            return type;
        };

        /**
         * @param {string} oldHtml
         * @param {string} newHTML
         * @param {number} fromLine
         * @param {number} toLine
         */
        this.replaceLines = function (oldHtml, newHTML, fromLine, toLine) {
            var data = this.extractRangeByLineNumbers(oldHtml, fromLine, toLine),
                previousHtml = data.previousHtml + '<TEMPLATE></TEMPLATE>' + data.previousHtmlEndSnippet,
                previousFragment = this.htmlToFragment(previousHtml),
                followingHtml = data.followingHtmlStartSnippet + '<TEMPLATE></TEMPLATE>' + data.followingHtml,
                followingFragment = this.htmlToFragment(followingHtml),
                newFragment = this.htmlToFragment(newHTML);

            if (data.html.length > 0 && data.html.substr(-1) == ' ') {
                this._insertDanglingSpace(newFragment);
            }

            var merged = this._replaceLinesMergeNodeArrays(previousFragment.childNodes, newFragment.childNodes);
            merged = this._replaceLinesMergeNodeArrays(merged, followingFragment.childNodes);

            var mergedFragment = document.createDocumentFragment();
            for (var i = 0; i < merged.length; i++) {
                mergedFragment.appendChild(merged[i]);
            }

            var forgottenTemplates = mergedFragment.querySelectorAll("TEMPLATE");
            for (i = 0; i < forgottenTemplates.length; i++) {
                var el = forgottenTemplates[i];
                el.parentNode.removeChild(el);
            }

            return this._serializeDom(mergedFragment, true);
        };

        this.addCSSClass = function (node, className) {
            if (node.nodeType != ELEMENT_NODE) {
                return;
            }
            var classes = node.getAttribute('class');
            classes = (classes ? classes.split(' ') : []);
            if (classes.indexOf(className) == -1) {
                classes.push(className);
            }
            node.setAttribute('class', classes.join(' '));
        };

        this.addDiffMarkup = function (originalHTML, newHTML, fromLine, toLine, diffFormatterCb) {
            var data = this.extractRangeByLineNumbers(originalHTML, fromLine, toLine),
                previousHtml = data.previousHtml + '<TEMPLATE></TEMPLATE>' + data.previousHtmlEndSnippet,
                previousFragment = this.htmlToFragment(previousHtml),
                followingHtml = data.followingHtmlStartSnippet + '<TEMPLATE></TEMPLATE>' + data.followingHtml,
                followingFragment = this.htmlToFragment(followingHtml),
                newFragment = this.htmlToFragment(newHTML),
                oldHTML = data.outerContextStart + data.innerContextStart + data.html +
                    data.innerContextEnd + data.outerContextEnd,
                oldFragment = this.htmlToFragment(oldHTML),
                el;

            var diffFragment = diffFormatterCb(oldFragment, newFragment);

            var mergedFragment = document.createDocumentFragment();
            while (previousFragment.firstChild) {
                el = previousFragment.firstChild;
                previousFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (diffFragment.firstChild) {
                el = diffFragment.firstChild;
                diffFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (followingFragment.firstChild) {
                el = followingFragment.firstChild;
                followingFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }

            var forgottenTemplates = mergedFragment.querySelectorAll("TEMPLATE");
            for (var i = 0; i < forgottenTemplates.length; i++) {
                el = forgottenTemplates[i];
                el.parentNode.removeChild(el);
            }

            return this._serializeDom(mergedFragment, true);
        };

        /**
         * Adapted from http://ejohn.org/projects/javascript-diff-algorithm/
         * by John Resig, MIT License
         * @param {array} oldArr
         * @param {array} newArr
         * @returns {object}
         */
        this._diff = function (oldArr, newArr) {
            var ns = {},
                os = {},
                i;

            for (i = 0; i < newArr.length; i++) {
                if (ns[newArr[i]] === undefined)
                    ns[newArr[i]] = {rows: [], o: null};
                ns[newArr[i]].rows.push(i);
            }

            for (i = 0; i < oldArr.length; i++) {
                if (os[oldArr[i]] === undefined)
                    os[oldArr[i]] = {rows: [], n: null};
                os[oldArr[i]].rows.push(i);
            }

            for (i in ns) {
                if (ns[i].rows.length == 1 && typeof(os[i]) != "undefined" && os[i].rows.length == 1) {
                    newArr[ns[i].rows[0]] = {text: newArr[ns[i].rows[0]], row: os[i].rows[0]};
                    oldArr[os[i].rows[0]] = {text: oldArr[os[i].rows[0]], row: ns[i].rows[0]};
                }
            }

            for (i = 0; i < newArr.length - 1; i++) {
                if (newArr[i].text !== null && newArr[i + 1].text === undefined && newArr[i].row + 1 < oldArr.length &&
                    oldArr[newArr[i].row + 1].text === undefined && newArr[i + 1] == oldArr[newArr[i].row + 1]) {
                    newArr[i + 1] = {text: newArr[i + 1], row: newArr[i].row + 1};
                    oldArr[newArr[i].row + 1] = {text: oldArr[newArr[i].row + 1], row: i + 1};
                }
            }

            for (i = newArr.length - 1; i > 0; i--) {
                if (newArr[i].text !== null && newArr[i - 1].text === undefined && newArr[i].row > 0 &&
                    oldArr[newArr[i].row - 1].text === undefined && newArr[i - 1] == oldArr[newArr[i].row - 1]) {
                    newArr[i - 1] = {text: newArr[i - 1], row: newArr[i].row - 1};
                    oldArr[newArr[i].row - 1] = {text: oldArr[newArr[i].row - 1], row: i - 1};
                }
            }

            return {o: oldArr, n: newArr};
        };

        this._tokenizeHtml = function (str) {
            var splitArrayEntriesEmbedSeparator = function (arr, by, prepend) {
                var newArr = [];
                for (var i = 0; i < arr.length; i++) {
                    if (arr[i][0] === '<' && (by === " " || by === "\n")) {
                        // Don't split HTML tags
                        newArr.push(arr[i]);
                        continue;
                    }

                    var parts = arr[i].split(by);
                    if (parts.length === 1) {
                        newArr.push(arr[i]);
                    } else {
                        var j;
                        if (prepend) {
                            if (parts[0] !== '') {
                                newArr.push(parts[0]);
                            }
                            for (j = 1; j < parts.length; j++) {
                                newArr.push(by + parts[j]);
                            }
                        } else {
                            for (j = 0; j < parts.length - 1; j++) {
                                newArr.push(parts[j] + by);
                            }
                            if (parts[parts.length - 1] !== '') {
                                newArr.push(parts[parts.length - 1]);
                            }
                        }
                    }
                }
                return newArr;
            };
            var splitArrayEntriesSplitSeparator = function (arr, by) {
                var newArr = [];
                for (var i = 0; i < arr.length; i++) {
                    if (arr[i][0] === '<') {
                        newArr.push(arr[i]);
                        continue;
                    }
                    var parts = arr[i].split(by);
                    for (var j = 0; j < parts.length; j++) {
                        if (j > 0) {
                            newArr.push(by);
                        }
                        newArr.push(parts[j]);
                    }
                }
                return newArr;
            };
            var arr = splitArrayEntriesEmbedSeparator([str], '<', true);
            arr = splitArrayEntriesEmbedSeparator(arr, '>', false);
            arr = splitArrayEntriesSplitSeparator(arr, " ");
            arr = splitArrayEntriesSplitSeparator(arr, ".");
            arr = splitArrayEntriesEmbedSeparator(arr, "\n", false);

            var arrWithoutEmptes = [];
            for (var i = 0; i < arr.length; i++) {
                if (arr[i] !== '') {
                    arrWithoutEmptes.push(arr[i]);
                }
            }

            return arrWithoutEmptes;
        };

        /**
         * @param {string} oldStr
         * @param {string} newStr
         * @returns {string}
         */
        this._diffString = function (oldStr, newStr) {
            oldStr = this._normalizeHtmlForDiff(oldStr.replace(/\s+$/, '').replace(/^\s+/, ''));
            newStr = this._normalizeHtmlForDiff(newStr.replace(/\s+$/, '').replace(/^\s+/, ''));

            var out = this._diff(this._tokenizeHtml(oldStr), this._tokenizeHtml(newStr));
            var str = "";
            var i;

            if (out.n.length === 0) {
                for (i = 0; i < out.o.length; i++) {
                    str += '<del>' + out.o[i] + "</del>";
                }
            } else {
                if (out.n[0].text === undefined) {
                    for (var k = 0; k < out.o.length && out.o[k].text === undefined; k++) {
                        str += '<del>' + out.o[k] + "</del>";
                    }
                }

                for (i = 0; i < out.n.length; i++) {
                    if (out.n[i].text === undefined) {
                        if (out.n[i] !== "") {
                            str += '<ins>' + out.n[i] + "</ins>";
                        }
                    } else {
                        var pre = "";

                        for (var j = out.n[i].row + 1; j < out.o.length && out.o[j].text === undefined; j++) {
                            pre += '<del>' + out.o[j] + "</del>";
                        }
                        str += out.n[i].text + pre;
                    }
                }
            }

            return str.replace(/^\s+/g, '').replace(/\s+$/g, '').replace(/ {2,}/g, ' ');
        };

        /**
         * Calculates the ratio of the text affected by inline diff
         * From 0 (no changes at all) to 1 (everything has changed)
         *
         * @param html
         * @returns {number}
         * @private
         */
        this._calcChangeRatio = function(html) {
            var lengthChanged = 0;

            html = html.replace(/<del>(.*?)<\/del>/gi, function() { lengthChanged += arguments[1].length; return ""; });
            html = html.replace(/<ins>(.*?)<\/ins>/gi, function() { lengthChanged += arguments[1].length; return ""; });
            html = html.replace(/<.*?>/, "").trim();

            var lengthRemaining = html.length;
            if (lengthRemaining === 0 && lengthChanged === 0) {
                return 0;
            } else {
                return (lengthChanged / (lengthChanged + lengthRemaining));
            }
        };

        /**
         *
         * @param {string} html
         * @returns {boolean}
         * @private
         */
        this._diffDetectBrokenDiffHtml = function(html) {
            // If a regular HTML tag is enclosed by INS/DEL, the HTML is broken
            var match = html.match(/<(ins|del)><[^>]*><\/(ins|del)>/gi);
            if (match !== null && match.length > 0) {
                return true;
            }

            // If other HTML tags are contained within INS/DEL (e.g. "<ins>Test</p></ins>"), let's better be cautious
            // The "!!(found=...)"-construction is only used to make jshint happy :)
            var findDel = /<del>(.*?)<\/del>/gi,
                findIns = /<ins>(.*?)<\/ins>/gi,
                found;
            while (!!(found = findDel.exec(html))) {
                if (found[1].match(/<[^>]*>/)) {
                    return true;
                }
            }
            while (!!(found = findIns.exec(html))) {
                if (found[1].match(/<[^>]*>/)) {
                    return true;
                }
            }

            // If too much of the text is changed, it's better to separate the old from new new version,
            // otherwise the result looks strange
            if (this._calcChangeRatio(html) > 0.66) {
                return true;
            }

            // If non of the conditions up to now is met, we consider the diff as being sane
            return false;
        };

        this._diffParagraphs = function(oldText, newText, lineLength, firstLineNumber) {
            var oldTextWithBreaks, newTextWithBreaks, currChild;

            if (lineLength !== undefined) {
                oldTextWithBreaks = lineNumberingService.insertLineNumbersNode(oldText, lineLength, null, firstLineNumber);
                newTextWithBreaks = lineNumberingService.insertLineNumbersNode(newText, lineLength, null, firstLineNumber);
            } else {
                oldTextWithBreaks = document.createElement('div');
                oldTextWithBreaks.innerHTML = oldText;
                newTextWithBreaks = document.createElement('div');
                newTextWithBreaks.innerHTML = newText;
            }

            for (var i = 0; i < oldTextWithBreaks.childNodes.length; i++) {
                currChild = oldTextWithBreaks.childNodes[i];
                if (currChild.nodeType === TEXT_NODE) {
                    var wrapDel = document.createElement('del');
                    oldTextWithBreaks.insertBefore(wrapDel, currChild);
                    oldTextWithBreaks.removeChild(currChild);
                    wrapDel.appendChild(currChild);
                } else {
                    this.addCSSClass(currChild, 'delete');
                }
            }
            for (i = 0; i < newTextWithBreaks.childNodes.length; i++) {
                currChild = newTextWithBreaks.childNodes[i];
                if (currChild.nodeType === TEXT_NODE) {
                    var wrapIns = document.createElement('ins');
                    newTextWithBreaks.insertBefore(wrapIns, currChild);
                    newTextWithBreaks.removeChild(currChild);
                    wrapIns.appendChild(currChild);
                } else {
                    this.addCSSClass(currChild, 'insert');
                }
            }

            var mergedFragment = document.createDocumentFragment(),
                el;
            while (oldTextWithBreaks.firstChild) {
                el = oldTextWithBreaks.firstChild;
                oldTextWithBreaks.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (newTextWithBreaks.firstChild) {
                el = newTextWithBreaks.firstChild;
                newTextWithBreaks.removeChild(el);
                mergedFragment.appendChild(el);
            }

            return this._serializeDom(mergedFragment);
        };

        this.addCSSClassToFirstTag = function (html, className) {
            return html.replace(/<[a-z][^>]*>/i, function (match) {
                if (match.match(/class=["'][a-z0-9 _-]*["']/i)) {
                    return match.replace(/class=["']([a-z0-9 _-]*)["']/i, function (match2, previousClasses) {
                        return "class=\"" + previousClasses + " " + className + "\"";
                    });
                } else {
                    return match.substring(0, match.length - 1) + " class=\"" + className + "\">";
                }
            });
        };

        /**
         * This function calculates the diff between two strings and tries to fix problems with the resulting HTML.
         * If lineLength and firstLineNumber is given, line numbers will be returned es well
         *
         * @param {number} lineLength
         * @param {number} firstLineNumber
         * @param {string} htmlOld
         * @param {string} htmlNew
         * @returns {string}
         */
        this.diff = function (htmlOld, htmlNew, lineLength, firstLineNumber) {
            var cacheKey = lineLength + ' ' + firstLineNumber + ' ' +
                    lineNumberingService.djb2hash(htmlOld) + lineNumberingService.djb2hash(htmlNew),
                cached = diffCache.get(cacheKey);
            if (!angular.isUndefined(cached)) {
                return cached;
            }

            // This fixes a really strange artefact with the diff that occures under the following conditions:
            // - The first tag of the two texts is identical, e.g. <p>
            // - A change happens in the next tag, e.g. inserted text
            // - The first tag occures a second time in the text, e.g. another <p>
            // In this condition, the first tag is deleted first and inserted afterwards again
            // Test case: "does not break when an insertion followes a beginning tag occuring twice"
            // The work around inserts to tags at the beginning and removes them afterwards again,
            // to make sure this situation does not happen (and uses invisible pseudo-tags in case something goes wrong)
            var workaroundPrepend = "<DUMMY><PREPEND>";

            var str = this._diffString(workaroundPrepend + htmlOld, workaroundPrepend + htmlNew),
                diffUnnormalized = str.replace(/^\s+/g, '').replace(/\s+$/g, '').replace(/ {2,}/g, ' ')
                .replace(/<\/ins><ins>/gi, '').replace(/<\/del><del>/gi, '');

            diffUnnormalized = diffUnnormalized.replace(
                /<ins>(\s*)(<p( [^>]*)?>[\s\S]*?<\/p>)(\s*)<\/ins>/gim,
                function(match, whiteBefore, inner, tagInner, whiteAfter) {
                    return whiteBefore +
                        inner
                        .replace(/<p( [^>]*)?>/gi, function(match) {
                            return match + "<ins>";
                        })
                        .replace(/<\/p>/gi, "</ins></p>") +
                        whiteAfter;
                }
            );

            diffUnnormalized = diffUnnormalized.replace(/<del>([a-z0-9,_-]* ?)<\/del><ins>([a-z0-9,_-]* ?)<\/ins>/gi, function (found, oldText, newText) {
                var foundDiff = false, commonStart = '', commonEnd = '',
                    remainderOld = oldText, remainderNew = newText;

                while (remainderOld.length > 0 && remainderNew.length > 0 && !foundDiff) {
                    if (remainderOld[0] == remainderNew[0]) {
                        commonStart += remainderOld[0];
                        remainderOld = remainderOld.substr(1);
                        remainderNew = remainderNew.substr(1);
                    } else {
                        foundDiff = true;
                    }
                }

                foundDiff = false;
                while (remainderOld.length > 0 && remainderNew.length > 0 && !foundDiff) {
                    if (remainderOld[remainderOld.length - 1] == remainderNew[remainderNew.length - 1]) {
                        commonEnd = remainderOld[remainderOld.length - 1] + commonEnd;
                        remainderNew = remainderNew.substr(0, remainderNew.length - 1);
                        remainderOld = remainderOld.substr(0, remainderOld.length - 1);
                    } else {
                        foundDiff = true;
                    }
                }

                var out = commonStart;
                if (remainderOld !== '') {
                    out += '<del>' + remainderOld + '</del>';
                }
                if (remainderNew !== '') {
                    out += '<ins>' + remainderNew + '</ins>';
                }
                out += commonEnd;

                return out;
            });

            diffUnnormalized = diffUnnormalized.replace(
                /<del>((<BR CLASS="OS-LINE-BREAK">)?<span[^>]+OS-LINE-NUMBER[^>]+?>\s*<\/span>)<\/del>/gi,
                function(found,tag) {
                    return tag.toLowerCase().replace(/> <\/span/gi, ">&nbsp;</span");
                }
            );

            if (diffUnnormalized.substr(0, workaroundPrepend.length) === workaroundPrepend) {
                diffUnnormalized = diffUnnormalized.substring(workaroundPrepend.length);
            }

            var diff;
            if (this._diffDetectBrokenDiffHtml(diffUnnormalized)) {
                diff = this._diffParagraphs(htmlOld, htmlNew, lineLength, firstLineNumber);
            } else {
                diffUnnormalized = diffUnnormalized.replace(/<ins>.*?(\n.*?)*<\/ins>/gi, function (found) {
                    found = found.replace(/<(div|p|li)[^>]*>/gi, function(match) { return match + '<ins>'; });
                    found = found.replace(/<\/(div|p|li)[^>]*>/gi, function(match) { return '</ins>' + match; });
                    return found;
                });
                diffUnnormalized = diffUnnormalized.replace(/<del>.*?(\n.*?)*<\/del>/gi, function (found) {
                    found = found.replace(/<(div|p|li)[^>]*>/gi, function(match) { return match + '<del>'; });
                    found = found.replace(/<\/(div|p|li)[^>]*>/gi, function(match) { return '</del>' + match; });
                    return found;
                });
                diffUnnormalized = diffUnnormalized.replace(/^<del><p>(.*)<\/p><\/del>$/gi, function(match, inner) { return "<p>" + inner + "</p>"; });

                var node = document.createElement('div');
                node.innerHTML = diffUnnormalized;
                diff = node.innerHTML;

                if (lineLength !== undefined && firstLineNumber !== undefined) {
                    node = lineNumberingService.insertLineNumbersNode(diff, lineLength, null, firstLineNumber);
                    diff = node.innerHTML;
                }
            }

            diffCache.put(cacheKey, diff);
            return diff;
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.docx', [])

.factory('MotionDocxExport', [
    '$http',
    '$q',
    'Config',
    'gettextCatalog',
    'FileSaver',
    function ($http, $q, Config, gettextCatalog, FileSaver) {

        var PAGEBREAK = '<w:p><w:r><w:br w:type="page" /></w:r></w:p>';
        var TAGS_NO_PARAM = ['b', 'strong', 'em', 'i'];

        var images;
        var relationships;
        var contentTypes;

        // $scope.motionsFiltered, $scope.categories

        var getData = function (motions, categories) {
            var data = {};
            // header
            var headerline1 = [
                Config.translate(Config.get('general_event_name').value),
                Config.translate(Config.get('general_event_description').value)
            ].filter(Boolean).join(' – ');
            var headerline2 = [
                Config.get('general_event_location').value,
                Config.get('general_event_date').value
            ].filter(Boolean).join(', ');
            data.header = [headerline1, headerline2].join('\n');

            // motion catalog title/preamble
            data.title = Config.translate(Config.get('motions_export_title').value);
            data.preamble = Config.get('motions_export_preamble').value;

            // categories
            data.has_categories = categories.length === 0 ? false : true;
            data.categories_translation = gettextCatalog.getString('Categories');
            data.categories = getCategoriesData(categories);
            data.no_categories = gettextCatalog.getString('No categories available.');
            data.pagebreak_main = categories.length === 0 ? '' : PAGEBREAK;

            // motions
            data.tableofcontents_translation = gettextCatalog.getString('Table of contents');
            data.motions = getMotionFullData(motions);
            data.motions_list = getMotionShortData(motions);
            data.no_motions = gettextCatalog.getString('No motions available.');

            return data;
        };

        var getCategoriesData = function (categories) {
            return _.map(categories, function (category) {
                return {
                    prefix: category.prefix,
                    name: category.name,
                };
            });
        };

        var getMotionShortData = function (motions) {
            return _.map(motions, function (motion) {
                return {
                    identifier: motion.identifier,
                    title: motion.getTitle(),
                };
            });
        };

        var getMotionFullData = function (motions) {
            var translation = gettextCatalog.getString('Motion'),
                sequential_translation = gettextCatalog.getString('Sequential number'),
                submitters_translation = gettextCatalog.getString('Submitters'),
                status_translation = gettextCatalog.getString('Status'),
                reason_translation = gettextCatalog.getString('Reason'),
                data = _.map(motions, function (motion) {
                    return {
                        motion_translation: translation,
                        sequential_translation: sequential_translation,
                        id: motion.id,
                        identifier: motion.identifier,
                        title: motion.getTitle(),
                        submitters_translation: submitters_translation,
                        submitters: _.map(motion.submitters, function (submitter) {
                                        return submitter.get_full_name();
                                    }).join(', '),
                        status_translation: status_translation,
                        status: motion.getStateName(),
                        preamble: gettextCatalog.getString(Config.get('motions_preamble').value),
                        text: html2docx(motion.getText()),
                        reason_translation: motion.getReason().length === 0 ? '' : reason_translation,
                        reason: html2docx(motion.getReason()),
                        pagebreak: PAGEBREAK,
                    };
                });
            if (data.length) {
                // clear pagebreak on last element
                data[data.length - 1].pagebreak = '';
            }
            return data;
        };

        var html2docx = function (html) {
            var docx = '';
            var stack = [];

            var isTag = false; // Even if html starts with '<p....' it is split to '', '<', ..., so always no tag at the beginning
            var hasParagraph = true;
            var skipFirstParagraphClosing = true;
            if (html.substring(0,3) != '<p>') {
                docx += '<w:p>';
                skipFirstParagraphClosing = false;
            }
            html = html.split(/(<|>)/g);

            html.forEach(function (part) {
                if (part !== '' && part != '\n' && part != '<' && part != '>') {
                    if (isTag) {
                        if (part.startsWith('p')) { /** p **/
                            // Special: begin new paragraph (only if its the first):
                            if (hasParagraph && !skipFirstParagraphClosing) {
                                // End, if there is one
                                docx += '</w:p>';
                            }
                            skipFirstParagraphClosing = false;
                            docx += '<w:p>';
                            hasParagraph = true;
                        } else if (part.startsWith('/p')) {
                            // Special: end paragraph:
                            docx += '</w:p>';
                            hasParagraph = false;

                        } else if (part.charAt(0) == "/") {
                            // remove from stack
                            stack.pop();
                        } else { // now all other tags
                            var tag = {};
                            if (_.indexOf(TAGS_NO_PARAM, part) > -1) { /** b, strong, em, i **/
                                stack.push({tag: part});
                            } else if (part.startsWith('span')) { /** span **/
                                tag = {tag: 'span', attrs: {}};
                                var rStyle = /(?:\"|\;\s?)([a-zA-z\-]+)\:\s?([a-zA-Z0-9\-\#]+)/g, matchSpan;
                                while ((matchSpan = rStyle.exec(part)) !== null) {
                                    switch (matchSpan[1]) {
                                        case 'color':
                                                tag.attrs.color = matchSpan[2].slice(1); // cut off the #
                                            break;
                                        case 'background-color':
                                                tag.attrs.backgroundColor = matchSpan[2].slice(1); // cut off the #
                                            break;
                                        case 'text-decoration':
                                            if (matchSpan[2] === 'underline') {
                                                tag.attrs.underline = true;
                                            } else if (matchSpan[2] === 'line-through') {
                                                tag.attrs.strike = true;
                                            }
                                            break;
                                    }
                                }
                                stack.push(tag);
                            } else if (part.startsWith('a')) { /** a **/
                                var rHref = /href="([^"]+)"/g;
                                var href = rHref.exec(part)[1];
                                tag = {tag: 'a', href: href};
                                stack.push(tag);
                            } else if (part.startsWith('img')) {
                                // images has to be placed instantly, so there is no use of 'tag'.
                                var img = {}, rImg = /(\w+)=\"([^\"]*)\"/g, matchImg;
                                while ((matchImg = rImg.exec(part)) !== null) {
                                    img[matchImg[1]] = matchImg[2];
                                }

                                // With and height and source have to be given!
                                if (img.width && img.height && img.src) {
                                    var rrId = relationships.length + 1;
                                    var imgId = images.length + 1;

                                    // set name ('pic.jpg'), title, ext ('jpg'), mime ('image/jpeg')
                                    img.name = img.src.split('/');
                                    img.name = _.last(img.name);

                                    var tmp = img.name.split('.');
                                    // set name without extension as title if there isn't a title
                                    if (!img.title) {
                                        img.title = tmp[0];
                                    }
                                    img.ext = tmp[1];

                                    img.mime = 'image/' + img.ext;
                                    if (img.ext == 'jpe' || img.ext == 'jpg') {
                                        img.mime = 'image/jpeg';
                                    }

                                    // x and y for the container and picture size in EMU (assuming 96dpi)!
                                    var x = img.width * 914400 / 96;
                                    var y = img.height * 914400 / 96;

                                    // Own paragraph for the image
                                    if (hasParagraph) {
                                        docx += '</w:p>';
                                    }
                                    docx += '<w:p><w:r><w:drawing><wp:inline distT="0" distB="0" distL="0" distR="0"><wp:extend cx="' + x +'" cy="' + y + '"/><wp:effectExtent l="0" t="0" r="0" b="0"/>' +
                                        '<wp:docPr id="' + imgId + '" name="' + img.name + '" title="' + img.title + '" descr="' + img.title + '"/><wp:cNvGraphicFramePr>' +
                                        '<a:graphicFrameLocks xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main" noChangeAspect="1"/></wp:cNvGraphicFramePr>' +
                                        '<a:graphic xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main"><a:graphicData uri="http://schemas.openxmlformats.org/drawingml/2006/picture">' +
                                        '<pic:pic xmlns:pic="http://schemas.openxmlformats.org/drawingml/2006/picture"><pic:nvPicPr><pic:cNvPr id="' + imgId + '" name="' +
                                        img.name + '" title="' + img.title + '" descr="' + img.title + '"/><pic:cNvPicPr/></pic:nvPicPr><pic:blipFill><a:blip r:embed="rrId' + rrId + '"/><a:stretch>' +
                                        '<a:fillRect/></a:stretch></pic:blipFill><pic:spPr bwMode="auto"><a:xfrm><a:off x="0" y="0"/><a:ext cx="' + x + '" cy="' + y + '"/></a:xfrm>' +
                                        '<a:prstGeom prst="rect"><a:avLst/></a:prstGeom></pic:spPr></pic:pic></a:graphicData></a:graphic></wp:inline></w:drawing></w:r></w:p>';

                                    // hasParagraph stays untouched, the documents paragraph state is restored here
                                    if (hasParagraph) {
                                        docx += '<w:p>';
                                    }

                                    // entries in images, relationships and contentTypes
                                    images.push({
                                        url: img.src,
                                        zipPath: 'word/media/' + img.name
                                    });
                                    relationships.push({
                                        Id: 'rrId' + rrId,
                                        Type: 'http://schemas.openxmlformats.org/officeDocument/2006/relationships/image',
                                        Target: 'media/' + img.name
                                    });
                                    contentTypes.push({
                                        PartName: '/word/media/' + img.name,
                                        ContentType: img.mime
                                    });
                                }
                            }
                        }
                    } else { /** No tag **/
                        if (!hasParagraph) {
                            docx += '<w:p>';
                            hasParagraph = true;
                        }
                        var docx_part = '<w:r><w:rPr>';
                        var hyperlink = false;
                        stack.forEach(function (tag) {
                            switch (tag.tag) {
                                case 'b': case 'strong':
                                    docx_part += '<w:b/><w:bCs/>';
                                    break;
                                case 'em': case 'i':
                                        docx_part += '<w:i/><w:iCs/>';
                                    break;
                                case 'span':
                                    for (var key in tag.attrs) {
                                        switch (key) {
                                            case 'color':
                                                docx_part += '<w:color w:val="' + tag.attrs[key] + '"/>';
                                                break;
                                            case 'backgroundColor':
                                                docx_part += '<w:shd w:fill="' + tag.attrs[key] + '"/>';
                                                break;
                                            case 'underline':
                                                docx_part += '<w:u w:val="single"/>';
                                                break;
                                            case 'strike':
                                                docx_part += '<w:strike/>';
                                                break;
                                        }
                                    }
                                    break;
                                case 'a':
                                    var id = relationships.length + 1;
                                    docx_part = '<w:hyperlink r:id="rrId' + id + '">' + docx_part;
                                    docx_part += '<w:rStyle w:val="Internetlink"/>'; // necessary?
                                    relationships.push({
                                        Id: 'rrId' + id,
                                        Type: 'http://schemas.openxmlformats.org/officeDocument/2006/relationships/hyperlink',
                                        Target: tag.href,
                                        TargetMode: 'External'
                                    });
                                    hyperlink = true;
                                    break;
                            }
                        });
                        docx_part += '</w:rPr><w:t>' + part + '</w:t></w:r>';
                        if (hyperlink) {
                            docx_part += '</w:hyperlink>';
                        }

                        // append to docx
                        docx += docx_part;
                    }
                    isTag = !isTag;
                }
                if (part === '' || part == '\n') {
                    // just if two tags following eachother: <b><span> --> ...,'>', '', '<',...
                    // or there is a line break between: <b>\n<span> --> ...,'>', '\n', '<',...
                    isTag = !isTag;
                }
            });

            // for finishing close the last paragraph (if open)
            if (hasParagraph) {
                docx += '</w:p>';
            }

            // replacing of special symbols:
            docx = docx.replace(new RegExp('\&auml\;', 'g'), 'ä');
            docx = docx.replace(new RegExp('\&uuml\;', 'g'), 'ü');
            docx = docx.replace(new RegExp('\&ouml\;', 'g'), 'ö');
            docx = docx.replace(new RegExp('\&Auml\;', 'g'), 'Ä');
            docx = docx.replace(new RegExp('\&Uuml\;', 'g'), 'Ü');
            docx = docx.replace(new RegExp('\&Ouml\;', 'g'), 'Ö');
            docx = docx.replace(new RegExp('\&szlig\;', 'g'), 'ß');
            docx = docx.replace(new RegExp('\&nbsp\;', 'g'), ' ');
            docx = docx.replace(new RegExp('\&sect\;', 'g'), '§');

            // remove all entities except gt, lt and amp
            var rEntity = /\&(?!gt|lt|amp)\w+\;/g, matchEntry, indexes = [];
            while ((matchEntry = rEntity.exec(docx)) !== null) {
                indexes.push({
                    startId: matchEntry.index,
                    stopId: matchEntry.index + matchEntry[0].length
                });
            }
            for (var i = indexes.length - 1; i>=0; i--) {
                docx = docx.substring(0, indexes[i].startId) + docx.substring(indexes[i].stopId, docx.length);
            }

            return docx;
        };
        var updateRelationships = function (oldContent) {
            var content = oldContent.split('\n');
            relationships.forEach(function (rel) {
                content[1] += '<Relationship';
                for (var key in rel) {
                    content[1] += ' ' + key + '="' + rel[key] + '"';
                }
                content[1] += '/>';
            });
            return content.join('\n');
        };
        var updateContentTypes = function (oldContent) {
            var content = oldContent.split('\n');
            contentTypes.forEach(function (type) {
                content[1] += '<Override';
                for (var key in type) {
                    content[1] += ' ' + key + '="' + type[key] + '"';
                }
                content[1] += '/>';
            });
            return content.join('\n');
        };

        return {
            export: function (motions, categories) {
                images = [];
                relationships = [];
                contentTypes = [];
                $http.get('/motions/docxtemplate/').then(function (success) {
                    var content = window.atob(success.data);
                    var doc = new Docxgen(content);

                    doc.setData(getData(motions, categories));
                    doc.render();
                    var zip = doc.getZip();

                    // update relationships from 'relationships'
                    var rels = updateRelationships(zip.file('word/_rels/document.xml.rels').asText());
                    zip.file('word/_rels/document.xml.rels', rels);

                    // update content type from 'contentTypes'
                    var contentTypes = updateContentTypes(zip.file('[Content_Types].xml').asText());
                    zip.file('[Content_Types].xml', contentTypes);

                    var imgPromises = [];
                    images.forEach(function (img) {
                        imgPromises.push(
                            $http.get(img.url, {responseType: 'arraybuffer'}).then(function (resolvedImage) {
                                zip.file(img.zipPath, resolvedImage.data);
                            })
                        );
                    });
                    // wait for all images to be resolved
                    $q.all(imgPromises).then(function () {
                        var out = zip.generate({type: 'blob'});
                        FileSaver.saveAs(out, 'motions-export.docx');
                    });
                });
            },
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.lineNumbering', [])

/**
 * Current limitations of this implementation:
 *
 * Only the following inline elements are supported:
 * - 'SPAN', 'A', 'EM', 'S', 'B', 'I', 'STRONG', 'U', 'BIG', 'SMALL', 'SUB', 'SUP', 'TT'
 * - 'INS' and 'DEL' are supported, but line numbering does not affect the content of 'INS'-elements
 *
 * Only other inline elements are allowed within inline elements.
 * No constructs like <a...><div></div></a> are allowed. CSS-attributes like 'display: block' are ignored.
 */

.service('lineNumberingService', [
    '$cacheFactory',
    function ($cacheFactory) {
        var ELEMENT_NODE = 1,
            TEXT_NODE = 3;

        this._currentInlineOffset = null;
        this._currentLineNumber = null;
        this._prependLineNumberToFirstText = false;
        this._ignoreNextRegularLineNumber = false;
        this._ignoreInsertedText = false;

        var lineNumberCache = $cacheFactory('linenumbering.service');

        this.djb2hash = function(str) {
            var hash = 5381, char;
            for (var i = 0; i < str.length; i++) {
                char = str.charCodeAt(i);
                hash = ((hash << 5) + hash) + char;
            }
            return hash.toString();
        };

        this._isInlineElement = function (node) {
            var inlineElements = [
                'SPAN', 'A', 'EM', 'S', 'B', 'I', 'STRONG', 'U', 'BIG', 'SMALL', 'SUB', 'SUP', 'TT', 'INS', 'DEL',
                'STRIKE'
            ];
            return (inlineElements.indexOf(node.nodeName) > -1);
        };

        this._isIgnoredByLineNumbering = function (node) {
            if (node.nodeName === 'INS') {
                return this._ignoreInsertedText;
            } else if (this._isOsLineNumberNode(node)) {
                return true;
            } else {
                return false;
            }
        };

        this._isOsLineBreakNode = function (node) {
            var isLineBreak = false;
            if (node && node.nodeType === ELEMENT_NODE && node.nodeName == 'BR' && node.hasAttribute('class')) {
                var classes = node.getAttribute('class').split(' ');
                if (classes.indexOf('os-line-break') > -1) {
                    isLineBreak = true;
                }
            }
            return isLineBreak;
        };

        this._isOsLineNumberNode = function (node) {
            var isLineNumber = false;
            if (node && node.nodeType === ELEMENT_NODE && node.nodeName == 'SPAN' && node.hasAttribute('class')) {
                var classes = node.getAttribute('class').split(' ');
                if (classes.indexOf('os-line-number') > -1) {
                    isLineNumber = true;
                }
            }
            return isLineNumber;
        };

        this._getLineNumberNode = function(fragment, lineNumber) {
            return fragment.querySelector('.os-line-number.line-number-' + lineNumber);
        };

        this._htmlToFragment = function(html) {
            var fragment = document.createDocumentFragment(),
                div = document.createElement('DIV');
            div.innerHTML = html;
            while (div.childElementCount) {
                var child = div.childNodes[0];
                div.removeChild(child);
                fragment.appendChild(child);
            }
            return fragment;
        };

        this._fragmentToHtml = function(fragment) {
            var div = document.createElement('DIV');
            while (fragment.firstChild) {
                var child = fragment.firstChild;
                fragment.removeChild(child);
                div.appendChild(child);
            }
            return div.innerHTML;
        };

        this._createLineBreak = function () {
            var br = document.createElement('br');
            br.setAttribute('class', 'os-line-break');
            return br;
        };

        this._createLineNumber = function () {
            if (this._ignoreNextRegularLineNumber) {
                this._ignoreNextRegularLineNumber = false;
                return;
            }
            var node = document.createElement('span');
            var lineNumber = this._currentLineNumber;
            this._currentLineNumber++;
            node.setAttribute('class', 'os-line-number line-number-' + lineNumber);
            node.setAttribute('data-line-number', lineNumber + '');
            node.setAttribute('contenteditable', 'false');
            node.innerHTML = '&nbsp;'; // Prevent ckeditor from stripping out empty span's
            return node;
        };

        /**
         * Splits a TEXT_NODE into an array of TEXT_NODEs and BR-Elements separating them into lines.
         * Each line has a maximum length of 'length', with one exception: spaces are accepted to exceed the length.
         * Otherwise the string is split by the last space or dash in the line.
         *
         * @param node
         * @param length
         * @param highlight
         * @returns Array
         * @private
         */
        this._textNodeToLines = function (node, length, highlight) {
            var out = [],
                currLineStart = 0,
                i = 0,
                firstTextNode = true,
                lastBreakableIndex = null,
                service = this;
            var addLine = function (text, highlight) {
                var node;
                if (typeof highlight === 'undefined') {
                    highlight = -1;
                }
                if (firstTextNode) {
                    if (highlight === service._currentLineNumber - 1) {
                        node = document.createElement('span');
                        node.setAttribute('class', 'highlight');
                        node.innerHTML = text;
                    } else {
                        node = document.createTextNode(text);
                    }
                    firstTextNode = false;
                } else {
                    if (service._currentLineNumber === highlight && highlight !== null) {
                        node = document.createElement('span');
                        node.setAttribute('class', 'highlight');
                        node.innerHTML = text;
                    } else {
                        node = document.createTextNode(text);
                    }
                    out.push(service._createLineBreak());
                    if (service._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                }
                out.push(node);
            };

            if (node.nodeValue == "\n") {
                out.push(node);
            } else {

                // This happens if a previous inline element exactly stretches to the end of the line
                if (this._currentInlineOffset >= length) {
                    out.push(service._createLineBreak());
                    if (this._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                    this._currentInlineOffset = 0;
                } else if (this._prependLineNumberToFirstText) {
                    if (this._ignoreNextRegularLineNumber) {
                        this._ignoreNextRegularLineNumber = false;
                    } else if (service._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                }
                this._prependLineNumberToFirstText = false;

                while (i < node.nodeValue.length) {
                    var lineBreakAt = null;
                    if (this._currentInlineOffset >= length) {
                        if (lastBreakableIndex !== null) {
                            lineBreakAt = lastBreakableIndex;
                        } else {
                            lineBreakAt = i - 1;
                        }
                    }
                    if (lineBreakAt !== null && node.nodeValue[i] != ' ') {
                        var currLine = node.nodeValue.substring(currLineStart, lineBreakAt + 1);
                        addLine(currLine, highlight);

                        currLineStart = lineBreakAt + 1;
                        this._currentInlineOffset = i - lineBreakAt - 1;
                        lastBreakableIndex = null;
                    }

                    if (node.nodeValue[i] == ' ' || node.nodeValue[i] == '-') {
                        lastBreakableIndex = i;
                    }

                    this._currentInlineOffset++;
                    i++;

                }
                addLine(node.nodeValue.substring(currLineStart), highlight);
            }
            return out;
        };


        /**
         * Moves line breaking and line numbering markup before inline elements
         *
         * @param innerNode
         * @param outerNode
         * @private
         */
        this._moveLeadingLineBreaksToOuterNode = function (innerNode, outerNode) {
            if (this._isInlineElement(innerNode)) {
                if (this._isOsLineBreakNode(innerNode.firstChild)) {
                    var br = innerNode.firstChild;
                    innerNode.removeChild(br);
                    outerNode.appendChild(br);
                }
                if (this._isOsLineNumberNode(innerNode.firstChild)) {
                    var span = innerNode.firstChild;
                    innerNode.removeChild(span);
                    outerNode.appendChild(span);
                }
            }
        };

        this._lengthOfFirstInlineWord = function (node) {
            if (!node.firstChild) {
                return 0;
            }
            if (node.firstChild.nodeType == TEXT_NODE) {
                var parts = node.firstChild.nodeValue.split(' ');
                return parts[0].length;
            } else {
                return this._lengthOfFirstInlineWord(node.firstChild);
            }
        };

        this._insertLineNumbersToInlineNode = function (node, length, highlight) {
            var oldChildren = [], i;
            for (i = 0; i < node.childNodes.length; i++) {
                oldChildren.push(node.childNodes[i]);
            }

            while (node.firstChild) {
                node.removeChild(node.firstChild);
            }

            for (i = 0; i < oldChildren.length; i++) {
                if (oldChildren[i].nodeType == TEXT_NODE) {
                    var ret = this._textNodeToLines(oldChildren[i], length, highlight);
                    for (var j = 0; j < ret.length; j++) {
                        node.appendChild(ret[j]);
                    }
                } else if (oldChildren[i].nodeType == ELEMENT_NODE) {
                    var firstword = this._lengthOfFirstInlineWord(oldChildren[i]),
                        overlength = ((this._currentInlineOffset + firstword) > length && this._currentInlineOffset > 0);
                    if (overlength && this._isInlineElement(oldChildren[i])) {
                        this._currentInlineOffset = 0;
                        node.appendChild(this._createLineBreak());
                        if (this._currentLineNumber !== null) {
                            node.appendChild(this._createLineNumber());
                        }
                    }
                    var changedNode = this._insertLineNumbersToNode(oldChildren[i], length, highlight);
                    this._moveLeadingLineBreaksToOuterNode(changedNode, node);
                    node.appendChild(changedNode);
                } else {
                    throw 'Unknown nodeType: ' + i + ': ' + oldChildren[i];
                }
            }

            return node;
        };

        this._calcBlockNodeLength = function (node, oldLength) {
            var newLength = oldLength;
            switch (node.nodeName) {
                case 'LI':
                    newLength -= 5;
                    break;
                case 'BLOCKQUOTE':
                    newLength -= 20;
                    break;
                case 'DIV':
                case 'P':
                    var styles = node.getAttribute("style"),
                        padding = 0;
                    if (styles) {
                        var leftpad = styles.split("padding-left:");
                        if (leftpad.length > 1) {
                            leftpad = parseInt(leftpad[1]);
                            padding += leftpad;
                        }
                        var rightpad = styles.split("padding-right:");
                        if (rightpad.length > 1) {
                            rightpad = parseInt(rightpad[1]);
                            padding += rightpad;
                        }
                        newLength -= (padding / 5);
                    }
                    break;
                case 'H1':
                    newLength *= 0.5;
                    break;
                case 'H2':
                    newLength *= 0.66;
                    break;
                case 'H3':
                    newLength *= 0.66;
                    break;
            }
            return Math.ceil(newLength);
        };

        this._insertLineNumbersToBlockNode = function (node, length, highlight) {
            this._currentInlineOffset = 0;
            this._prependLineNumberToFirstText = true;

            var oldChildren = [], i;
            for (i = 0; i < node.childNodes.length; i++) {
                oldChildren.push(node.childNodes[i]);
            }

            while (node.firstChild) {
                node.removeChild(node.firstChild);
            }

            for (i = 0; i < oldChildren.length; i++) {
                if (oldChildren[i].nodeType == TEXT_NODE) {
                    if (!oldChildren[i].nodeValue.match(/\S/)) {
                        // White space nodes between block elements should be ignored
                        var prevIsBlock = (i > 0 && !this._isInlineElement(oldChildren[i - 1]));
                        var nextIsBlock = (i < oldChildren.length - 1 && !this._isInlineElement(oldChildren[i + 1]));
                        if ((prevIsBlock && nextIsBlock) || (i === 0 && nextIsBlock) || (i === oldChildren.length - 1 && prevIsBlock)) {
                            node.appendChild(oldChildren[i]);
                            continue;
                        }
                    }
                    var ret = this._textNodeToLines(oldChildren[i], length, highlight);
                    for (var j = 0; j < ret.length; j++) {
                        node.appendChild(ret[j]);
                    }
                } else if (oldChildren[i].nodeType == ELEMENT_NODE) {
                    var firstword = this._lengthOfFirstInlineWord(oldChildren[i]),
                        overlength = ((this._currentInlineOffset + firstword) > length && this._currentInlineOffset > 0);
                    if (overlength && this._isInlineElement(oldChildren[i]) && !this._isIgnoredByLineNumbering(oldChildren[i])) {
                        this._currentInlineOffset = 0;
                        node.appendChild(this._createLineBreak());
                        if (this._currentLineNumber !== null) {
                            node.appendChild(this._createLineNumber());
                        }
                    }
                    var changedNode = this._insertLineNumbersToNode(oldChildren[i], length, highlight);
                    this._moveLeadingLineBreaksToOuterNode(changedNode, node);
                    node.appendChild(changedNode);
                } else {
                    throw 'Unknown nodeType: ' + i + ': ' + oldChildren[i];
                }
            }

            this._currentInlineOffset = 0;
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;

            return node;
        };

        this._insertLineNumbersToNode = function (node, length, highlight) {
            if (node.nodeType !== ELEMENT_NODE) {
                throw 'This method may only be called for ELEMENT-nodes: ' + node.nodeValue;
            }
            if (this._isIgnoredByLineNumbering(node)) {
                if (this._currentInlineOffset === 0 && this._currentLineNumber !== null) {
                    var lineNumberNode = this._createLineNumber();
                    if (lineNumberNode) {
                        node.insertBefore(lineNumberNode, node.firstChild);
                        this._ignoreNextRegularLineNumber = true;
                    }
                }
                return node;
            } else if (this._isInlineElement(node)) {
                return this._insertLineNumbersToInlineNode(node, length, highlight);
            } else {
                var newLength = this._calcBlockNodeLength(node, length);
                return this._insertLineNumbersToBlockNode(node, newLength, highlight);
            }
        };

        this._stripLineNumbers = function (node) {
            for (var i = 0; i < node.childNodes.length; i++) {
                if (this._isOsLineBreakNode(node.childNodes[i]) || this._isOsLineNumberNode(node.childNodes[i])) {
                    node.removeChild(node.childNodes[i]);
                    i--;
                } else {
                    this._stripLineNumbers(node.childNodes[i]);
                }
            }
        };

        this._nodesToHtml = function (nodes) {
            var root = document.createElement('div');
            for (var i in nodes) {
                if (nodes.hasOwnProperty(i)) {
                    root.appendChild(nodes[i]);
                }
            }
            return root.innerHTML;
        };

        /**
         *
         * @param {string} html
         * @param {number} lineLength
         * @param {number|null} highlight - optional
         * @param {number|null} firstLine
         */
        this.insertLineNumbersNode = function (html, lineLength, highlight, firstLine) {
            var root = document.createElement('div');
            root.innerHTML = html;

            this._currentInlineOffset = 0;
            if (firstLine) {
                this._currentLineNumber = parseInt(firstLine);
            } else {
                this._currentLineNumber = 1;
            }
            if (highlight !== null) {
                highlight = parseInt(highlight);
            }
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;
            this._ignoreInsertedText = true;

            return this._insertLineNumbersToNode(root, lineLength, highlight);
        };

        /**
         *
         * @param {string} html
         * @param {number} lineLength
         * @param {number|null} highlight - optional
         * @param {function} callback
         * @param {number} firstLine
         * @returns {string}
         */
        this.insertLineNumbers = function (html, lineLength, highlight, callback, firstLine) {
            var newHtml, newRoot;

            if (highlight > 0) {
                // Caching versions with highlighted line numbers is probably not worth it
                newRoot = this.insertLineNumbersNode(html, lineLength, highlight, firstLine);
                newHtml = newRoot.innerHTML;
            } else {
                var cacheKey = this.djb2hash(html);
                newHtml = lineNumberCache.get(cacheKey);

                if (angular.isUndefined(newHtml)) {
                    newRoot = this.insertLineNumbersNode(html, lineLength, null, firstLine);
                    newHtml = newRoot.innerHTML;
                    lineNumberCache.put(cacheKey, newHtml);
                }
            }

            if (callback) {
                callback();
            }

            return newHtml;
        };

        /**
         * @param {string} html
         * @param {number} lineLength
         * @param {boolean} countInserted
         */
        this.insertLineBreaksWithoutNumbers = function (html, lineLength, countInserted) {
            var root = document.createElement('div');
            root.innerHTML = html;

            this._currentInlineOffset = 0;
            this._currentLineNumber = null;
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;
            this._ignoreInsertedText = !countInserted;

            var newRoot = this._insertLineNumbersToNode(root, lineLength, null);

            return newRoot.innerHTML;
        };

        /**
         * @param {string} html
         * @returns {string}
         */
        this.stripLineNumbers = function (html) {
            var root = document.createElement('div');
            root.innerHTML = html;
            this._stripLineNumbers(root);
            return root.innerHTML;
        };

        /**
         * Traverses up the DOM tree until it finds a node with a nextSibling, then returns that sibling
         *
         * @param node
         * @private
         */
        this._findNextAuntNode = function(node) {
            if (node.nextSibling) {
                return node.nextSibling;
            } else if (node.parentNode) {
                return this._findNextAuntNode(node.parentNode);
            } else {
                return null;
            }
        };

        this._highlightUntilNextLine = function(lineNumberNode) {
            var currentNode = lineNumberNode,
                foundNextLineNumber = false;

            do {
                var wasHighlighted = false;
                if (currentNode.nodeType === TEXT_NODE) {
                    var node = document.createElement('span');
                    node.setAttribute('class', 'highlight');
                    node.innerHTML = currentNode.nodeValue;
                    currentNode.parentNode.insertBefore(node, currentNode);
                    currentNode.parentNode.removeChild(currentNode);
                    currentNode = node;
                    wasHighlighted = true;
                } else {
                    wasHighlighted = false;
                }

                if (currentNode.childNodes.length > 0 && !this._isOsLineNumberNode(currentNode) && !wasHighlighted) {
                    currentNode = currentNode.childNodes[0];
                } else if (currentNode.nextSibling) {
                    currentNode = currentNode.nextSibling;
                } else {
                    currentNode = this._findNextAuntNode(currentNode);
                }

                if (this._isOsLineNumberNode(currentNode)) {
                    foundNextLineNumber = true;
                }
            } while (!foundNextLineNumber && currentNode !== null);
        };

        /**
         * @param {string} html
         * @param {number} lineNumber
         * @return {string}
         */
        this.highlightLine = function (html, lineNumber) {
            lineNumber = parseInt(lineNumber);
            var fragment = this._htmlToFragment(html),
                lineNumberNode = this._getLineNumberNode(fragment, lineNumber);

            if (lineNumberNode) {
                this._highlightUntilNextLine(lineNumberNode);
                html = this._fragmentToHtml(fragment);
            }

            return html;
        };
    }
]);


}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.motionBlockProjector', [])


// MotionBlock projector elements

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('motions/motion-block', {
            template: 'static/templates/motions/slide_motion_block.html',
        });
    }
])

.controller('SlideMotionBlockCtrl', [
    '$scope',
    'Motion',
    'MotionBlock',
    function($scope, Motion, MotionBlock) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        MotionBlock.bindOne(id, $scope, 'motionBlock');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.motionBlock', [])


// MotionBlock model

.factory('MotionBlock', [
    'DS',
    'jsDataModel',
    'gettext',
    function(DS, jsDataModel, gettext) {
        var name = 'motions/motion-block';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Motion block'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'motions/motion': {
                        localField: 'motions',
                        foreignKey: 'motion_block_id',
                        osProtectedRelation: true,
                    }
                },
            }
        });
    }
])

.run(['MotionBlock', function(MotionBlock) {}])

// MotionBlock views (list view, create dialog, update dialog)
.factory('MotionBlockForm', [
    '$http',
    'operator',
    'gettextCatalog',
    'Agenda',
    'AgendaTree',
    function ($http, operator, gettextCatalog, Agenda, AgendaTree) {
        return {
            // Get ngDialog configuration.
            getDialog: function (motionBlock) {
                return {
                    template: 'static/templates/motions/motion-block-form.html',
                    controller: (motionBlock) ? 'MotionBlockUpdateCtrl' : 'MotionBlockCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motionBlockId: function () {return motionBlock ? motionBlock.id : void 0;}
                    }
                };
            },
            // Get angular-formly fields.
            getFormFields: function () {
                return [
                    {
                        key: 'title',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Title')
                        }
                    },
                    {
                        key: 'showAsAgendaItem',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show as agenda item'),
                            description: gettextCatalog.getString('If deactivated it appears as internal item on agenda.')
                        },
                        hide: !(operator.hasPerms('motions.can_manage') && operator.hasPerms('agenda.can_manage'))
                    },
                    {
                        key: 'agenda_parent_item_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        }
                    }
                ];
            }
        };
    }
])

.controller('MotionBlockListCtrl', [
    '$scope',
    'ngDialog',
    'MotionBlock',
    'MotionBlockForm',
    'Projector',
    'ProjectionDefault',
    function ($scope, ngDialog, MotionBlock, MotionBlockForm, Projector, ProjectionDefault) {
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motionBlocks'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        // Two-way data binding for all MotionBlock instances.
        MotionBlock.bindAll({}, $scope, 'motionBlocks');

        // Dialog with a form to create or update a MotionBlock instance.
        $scope.openFormDialog = function (motionBlock) {
            ngDialog.open(MotionBlockForm.getDialog(motionBlock));
        };

        // Confirm dialog to delete a MotionBlock instance.
        $scope.delete = function (motionBlock) {
            MotionBlock.destroy(motionBlock.id);
        };
    }
])

.controller('MotionBlockDetailCtrl', [
    '$scope',
    '$http',
    'ngDialog',
    'Motion',
    'MotionBlockForm',
    'MotionBlock',
    'motionBlockId',
    'Projector',
    'ProjectionDefault',
    'ErrorMessage',
    function($scope, $http, ngDialog, Motion, MotionBlockForm, MotionBlock, motionBlockId, Projector, ProjectionDefault, ErrorMessage) {
        MotionBlock.bindOne(motionBlockId, $scope, 'motionBlock');
        Motion.bindAll({}, $scope, 'motions');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motionBlocks'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.openDialog = function (motionBlock) {
            ngDialog.open(MotionBlockForm.getDialog(motionBlock));
        };
        $scope.followRecommendations = function () {
            $http.post('/rest/motions/motion-block/' + motionBlockId + '/follow_recommendations/').then(
                function (success) {
                $scope.alert = { type: 'success', msg: success.data.detail, show: true };
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
        $scope.delete = function (motion) {
            motion.motion_block_id = null;
            motion.title = motion.getTitle(-1);
            motion.text = motion.getText(-1);
            motion.reason = motion.getReason(-1);
            Motion.save(motion);
        };
    }
])

.controller('MotionBlockCreateCtrl', [
    '$scope',
    'MotionBlock',
    'MotionBlockForm',
    'AgendaUpdate',
    function($scope, MotionBlock, MotionBlockForm, AgendaUpdate) {
        // Prepare form.
        $scope.model = {};
        $scope.model.showAsAgendaItem = true;

        // Get all form fields.
        $scope.formFields = MotionBlockForm.getFormFields();

        // Save form.
        $scope.save = function (motionBlock) {
            MotionBlock.create(motionBlock).then(
                function (success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (motionBlock.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: motionBlock.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id, changes);
                    $scope.closeThisDialog();
                },
                function (error) {
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };
    }
])

.controller('MotionBlockUpdateCtrl', [
    '$scope',
    '$state',
    'MotionBlock',
    'MotionBlockForm',
    'AgendaUpdate',
    'motionBlockId',
    function($scope, $state, MotionBlock, MotionBlockForm, AgendaUpdate, motionBlockId) {
        // TODO: Check #2486 and remove some agenda related code.
        //MotionBlock.loadRelations(motionBlock, 'agenda_item');
        $scope.alert = {};

        // Prepare form. Set initial values by creating a deep copy of
        // motionBlock object so list/detail view is not updated while editing.
        var motionBlock = MotionBlock.get(motionBlockId);
        $scope.model = angular.copy(motionBlock);

        // Get all form fields.
        $scope.formFields = MotionBlockForm.getFormFields();
        for (var i = 0; i < $scope.formFields.length; i++) {
            if ($scope.formFields[i].key == 'showAsAgendaItem') {
                // Get state from agenda item (hidden/internal or agenda item).
                $scope.formFields[i].defaultValue = !motionBlock.agenda_item.is_hidden;
            } else if ($scope.formFields[i].key == 'agenda_parent_item_id') {
                $scope.formFields[i].defaultValue = motionBlock.agenda_item.parent_id;
            }
        }
        // Save form.
        $scope.save = function (motionBlock) {
            MotionBlock.create(motionBlock).then(
                function (success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (motionBlock.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: motionBlock.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    $scope.closeThisDialog();
                },
                function (error) {
                    // Save error: revert all changes by restore
                    // (refresh) original motionBlock object from server
                    MotionBlock.refresh(motionBlock);  // TODO: Why do we need a refresh here?
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.motionservices', ['OpenSlidesApp.motions', 'OpenSlidesApp.motions.lineNumbering'])

.factory('MotionPDFExport', [
    'HTMLValidizer',
    'Motion',
    'User',
    'PdfMakeConverter',
    'PdfMakeDocumentProvider',
    'PdfMakeBallotPaperProvider',
    'MotionContentProvider',
    'PollContentProvider',
    'gettextCatalog',
    '$http',
    'PdfCreate',
    function (HTMLValidizer, Motion, User, PdfMakeConverter, PdfMakeDocumentProvider, PdfMakeBallotPaperProvider,
            MotionContentProvider, PollContentProvider, gettextCatalog, $http, PdfCreate) {
        var obj = {};

        var $scope;

        obj.createMotion = function() {
            var text = $scope.motion.getTextByMode($scope.viewChangeRecommendations.mode, $scope.version);
            var content = HTMLValidizer.validize(text) + HTMLValidizer.validize($scope.motion.getReason($scope.version));
            var map = Function.prototype.call.bind([].map);
            var image_sources = map($(content).find("img"), function(element) {
                return element.getAttribute("src");
            });

            $http.post('/core/encode_media/', JSON.stringify(image_sources)).then(function (success) {
                var converter = PdfMakeConverter.createInstance(success.data.images);
                var motionContentProvider = MotionContentProvider.createInstance(converter, $scope.motion, $scope, User, $http);
                var documentProvider = PdfMakeDocumentProvider.createInstance(motionContentProvider);
                var identifier = $scope.motion.identifier ? '-' + $scope.motion.identifier : '';
                var filename = gettextCatalog.getString("Motion") + identifier + ".pdf";
                PdfCreate.download(documentProvider.getDocument(), filename);
            });
        };

        //make PDF for polls
        obj.createPoll = function() {
            var id = $scope.motion.identifier.replace(" ", "");
            var title = $scope.motion.getTitle($scope.version);
            var filename = gettextCatalog.getString("Motion") + "-" + id + "-" + gettextCatalog.getString("ballot-paper") + ".pdf";
            var pollContentProvider = PollContentProvider.createInstance(title, id, gettextCatalog);
            var documentProvider = PdfMakeBallotPaperProvider.createInstance(pollContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };

        obj.init = function (_scope) {
            $scope = _scope;
        };

        return obj;
    }
])

.factory('MotionInlineEditing', [
    'Editor',
    'Motion',
    '$timeout',
    'gettextCatalog',
    function (Editor, Motion, $timeout, gettextCatalog) {
        var createInstance = function ($scope, motion, selector, versioning, getOriginalData, saveData) {
            var obj = {
                active: false,
                changed: false,
                isEditable: false,
                trivialChange: false,
                originalHtml: null,
                ckeditorOptions: Editor.getOptions(),
            };
            obj.ckeditorOptions.readOnly = true;

            obj.setVersion = function (_motion, versionId) {
                motion = _motion; // If this is not updated,
                obj.originalHtml = motion.getTextWithLineBreaks(versionId);
                obj.changed = false;
                if (obj.editor) {
                    obj.editor.setReadOnly(true);
                    obj.editor.setData(obj.originalHtml);
                }
            };

            obj.enable = function () {
                obj.active = true;
                obj.isEditable = true;
                obj.ckeditorOptions.language = gettextCatalog.getCurrentLanguage();
                obj.editor = CKEDITOR.inline(selector, obj.ckeditorOptions);
                obj.editor.on('change', function () {
                    $timeout(function() {
                        if (obj.editor.getData() != obj.originalHtml) {
                            obj.changed = true;
                        } else {
                            obj.changed = false;
                        }
                    });
                });
                obj.revert();
            };

            obj.disable = function () {
                if (obj.editor) {
                    obj.editor.setReadOnly(true);
                    obj.editor.setData(obj.originalHtml, {
                        callback: function() {
                            obj.editor.destroy();
                        }
                    });
                }
                $timeout(function() {
                    obj.active = false;
                    obj.changed = false;
                    obj.isEditable = false;
                });
            };

            // sets editor content to the initial motion state
            obj.revert = function(originalData) {
                if (obj.editor) {
                    obj.originalHtml = getOriginalData(obj);
                    obj.editor.setData(
                        getOriginalData(obj), {
                        callback: function() {
                            obj.originalHtml = obj.editor.getData();
                            obj.editor.setReadOnly(false);
                            $timeout(function() {
                                obj.changed = false;
                            });
                            $timeout(function () {
                                obj.editor.focus();
                            }, 100);
                        }
                    });
                }
            };

            obj.save = function () {
                saveData(obj);
                obj.disable();

                Motion.inject(motion);
                // save change motion object on server
                Motion.save(motion, {method: 'PATCH'}).then(
                    function (success) {
                        if (versioning) {
                            $scope.showVersion(motion.getVersion(-1));
                        }
                        obj.revert();
                    },
                    function (error) {
                        // save error: revert all changes by restore
                        // (refresh) original motion object from server
                        Motion.refresh(motion);
                        obj.revert();
                        var message = '';
                        for (var e in error.data) {
                            message += e + ': ' + error.data[e] + ' ';
                        }
                        $scope.alert = {type: 'danger', msg: message, show: true};
                    }
                );
            };

            return obj;
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('MotionCommentsInlineEditing', [
    'MotionInlineEditing',
    function (MotionInlineEditing) {
        var createInstances = function ($scope, motion) {
            var commentsInlineEditing = {
                editors: []
            };
            _.forEach($scope.commentsFields, function (field) {
                var inlineEditing = MotionInlineEditing.createInstance($scope, motion,
                    'view-original-comment-inline-editor-' + field.name, false,
                    function (obj) {
                        return motion['comment ' + field.name];
                    },
                    function (obj) {
                        motion['comment ' + field.name] = obj.editor.getData();
                    }
                );
                commentsInlineEditing.editors.push(inlineEditing);
            });
            commentsInlineEditing.saveToolbarVisible = function () {
                return _.some(commentsInlineEditing.editors, function (instance) {
                    return instance.changed && instance.active;
                });
            };
            commentsInlineEditing.active = function () {
                return _.some(commentsInlineEditing.editors, function (instance) {
                    return instance.active;
                });
            };
            commentsInlineEditing.save = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.save();
                });
            };
            commentsInlineEditing.revert = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.revert();
                });
            };
            commentsInlineEditing.enable = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.enable();
                });
            };
            commentsInlineEditing.disable = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.disable();
                });
            };

            return commentsInlineEditing;
        };
        return {
            createInstances: createInstances,
        };
    }
])

.factory('ChangeRecommmendationCreate', [
    'ngDialog',
    'ChangeRecommendationForm',
    function(ngDialog, ChangeRecommendationForm) {
        var MODE_INACTIVE = 0,
            MODE_SELECTING_FROM = 1,
            MODE_SELECTING_TO = 2;

        var obj = {
            mode: MODE_INACTIVE,
            lineFrom: 1,
            lineTo: 2,
            html: '',
            reviewingHtml: ''
        };

        var $scope, motion, version;

        obj._getAffectedLineNumbers = function () {
            var changeRecommendations = motion.getChangeRecommendations(version),
                affectedLines = [];
            for (var i = 0; i < changeRecommendations.length; i++) {
                var change = changeRecommendations[i];
                for (var j = change.line_from; j < change.line_to; j++) {
                    affectedLines.push(j);
                }
            }
            return affectedLines;
        };

        obj.startCreating = function () {
            if (obj.mode > MODE_SELECTING_FROM || !motion.isAllowed('can_manage')) {
                return;
            }

            $(".tt_change_recommendation_create_help").removeClass("opened");
            var $lineNumbers = $(".motion-text-original .os-line-number");
            if ($lineNumbers.filter(".selectable").length === 0) {
                obj.mode = MODE_SELECTING_FROM;
                var alreadyAffectedLines = obj._getAffectedLineNumbers();
                $lineNumbers.each(function () {
                    var $this = $(this),
                        lineNumber = $this.data("line-number");
                    if (alreadyAffectedLines.indexOf(lineNumber) == -1) {
                        $(this).addClass("selectable");
                    }
                });
            }
        };

        obj.cancelCreating = function (ev) {
            var $target = $(ev.target),
                query = ".line-numbers-outside .os-line-number.selectable";
            if (!$target.is(query) && $target.parents(query).length === 0) {
                obj.mode = MODE_INACTIVE;
                obj.lineFrom = 0;
                obj.lineTo = 0;
                $(".motion-text-original .os-line-number").removeClass("selected selectable");
                obj.startCreating();
            }
        };

        obj.setFromLine = function (line) {
            obj.mode = MODE_SELECTING_TO;
            obj.lineFrom = line;

            var alreadyAffectedLines = obj._getAffectedLineNumbers(),
                foundCollission = false;

            $(".motion-text-original .os-line-number").each(function () {
                var $this = $(this);
                if ($this.data("line-number") >= line && !foundCollission) {
                    if (alreadyAffectedLines.indexOf($this.data("line-number")) == -1) {
                        $(this).addClass("selectable");
                    } else {
                        $(this).removeClass("selectable");
                        foundCollission = true;
                    }
                } else {
                    $(this).removeClass("selectable");
                }
            });

            var tt_pos = $(".motion-text-original .line-number-" + line).position().top - 45;
            $(".tt_change_recommendation_create_help").css("top", tt_pos).addClass("opened");
        };

        obj.setToLine = function (line) {
            if (line < obj.lineFrom) {
                return;
            }
            obj.mode = MODE_INACTIVE;
            obj.lineTo = line + 1;
            ngDialog.open(ChangeRecommendationForm.getCreateDialog(
                motion,
                version,
                obj.lineFrom,
                obj.lineTo
            ));

            obj.lineFrom = 0;
            obj.lineTo = 0;
            $(".motion-text-original .os-line-number").removeClass("selected selectable");
            obj.startCreating();
        };

        obj.lineClicked = function (ev) {
            if (obj.mode == MODE_INACTIVE) {
                return;
            }
            if (obj.mode == MODE_SELECTING_FROM) {
                obj.setFromLine($(ev.target).data("line-number"));
                $(ev.target).addClass("selected");
            } else if (obj.mode == MODE_SELECTING_TO) {
                obj.setToLine($(ev.target).data("line-number"));
            }
        };

        obj.mouseOver = function (ev) {
            if (obj.mode != MODE_SELECTING_TO) {
                return;
            }
            var hoverLine = $(ev.target).data("line-number");
            $(".motion-text-original .os-line-number").each(function () {
                var line = $(this).data("line-number");
                if (line >= obj.lineFrom && line <= hoverLine) {
                    $(this).addClass("selected");
                } else {
                    $(this).removeClass("selected");
                }
            });
        };

        obj.setVersion = function (_motion, _version) {
            motion = _motion;
            version = _version;
        };

        obj.editDialog = function(change_recommendation) {
            ngDialog.open(ChangeRecommendationForm.getEditDialog(change_recommendation));
        };

        obj.init = function (_scope, _motion) {
            $scope = _scope;
            motion = _motion;
            version = $scope.version;

            var $content = $("#content");
            $content.on("click", ".line-numbers-outside .os-line-number.selectable", obj.lineClicked);
            $content.on("click", obj.cancelCreating);
            $content.on("mouseover", ".line-numbers-outside .os-line-number.selectable", obj.mouseOver);
            $content.on("mouseover", ".motion-text-original", obj.startCreating);

            $scope.$watch(function () {
                return $scope.change_recommendations.length;
            }, function () {
                if (obj.mode == MODE_INACTIVE || obj.mode == MODE_SELECTING_FROM) {
                    // Recalculate the affected lines so we cannot select lines affected by a recommendation
                    // that has just been created
                    $(".motion-text-original .os-line-number").removeClass("selected selectable");
                    obj.startCreating();
                }
            });

            $scope.$on("$destroy", function () {
                obj.destroy();
            });
        };

        obj.destroy = function () {
            var $content = $("#content");
            $content.off("click", ".line-numbers-outside .os-line-number.selectable", obj.lineClicked);
            $content.off("click", obj.cancelCreating);
            $content.off("mouseover", ".line-numbers-outside .os-line-number.selectable", obj.mouseOver);
            $content.off("mouseover", ".motion-text-original", obj.startCreating);
        };

        return obj;
    }
])

.factory('ChangeRecommmendationView', [
    'Motion',
    'MotionChangeRecommendation',
    'Config',
    'lineNumberingService',
    'diffService',
    '$interval',
    '$timeout',
    function (Motion, MotionChangeRecommendation, Config, lineNumberingService, diffService, $interval, $timeout) {
        var $scope;

        var obj = {
            mode: 'original'
        };

        obj.diffFormatterCb = function (change, oldFragment, newFragment) {
            for (var i = 0; i < oldFragment.childNodes.length; i++) {
                diffService.addCSSClass(oldFragment.childNodes[i], 'delete');
            }
            for (i = 0; i < newFragment.childNodes.length; i++) {
                diffService.addCSSClass(newFragment.childNodes[i], 'insert');
            }
            var mergedFragment = document.createDocumentFragment(),
                diffSection = document.createElement('SECTION'),
                el;

            mergedFragment.appendChild(diffSection);
            diffSection.setAttribute('class', 'diff');
            diffSection.setAttribute('data-change-id', change.id);

            while (oldFragment.firstChild) {
                el = oldFragment.firstChild;
                oldFragment.removeChild(el);
                diffSection.appendChild(el);
            }
            while (newFragment.firstChild) {
                el = newFragment.firstChild;
                newFragment.removeChild(el);
                diffSection.appendChild(el);
            }

            return mergedFragment;
        };

        obj.delete = function (changeId) {
            MotionChangeRecommendation.destroy(changeId);
        };

        obj.rejectAll = function (motion) {
            var changeRecommendations = MotionChangeRecommendation.filter({
                'where': {'motion_version_id': {'==': motion.active_version}}
            });
            _.forEach(changeRecommendations, function(change) {
                change.rejected = true;
                change.saveStatus();
            });
        };

        obj.repositionOriginalAnnotations = function () {
            var $changeRecommendationList = $('.change-recommendation-list'),
                $lineNumberReference = $('.motion-text-original');

            $changeRecommendationList.children().each(function() {
                var $this = $(this),
                    lineFrom = $this.data('line-from'),
                    lineTo = ($this.data('line-to') - 1),
                    $lineFrom = $lineNumberReference.find('.line-number-' + lineFrom),
                    $lineTo = $lineNumberReference.find('.line-number-' + lineTo),
                    fromTop = $lineFrom.position().top + 3,
                    toTop = $lineTo.position().top + 20,
                    height = (toTop - fromTop);

                if (height < 10) {
                    height = 10;
                }

                // $lineFrom.position().top seems to depend on the scrolling position when the line numbers
                // have position: absolute. Maybe a bug in the used version of jQuery?
                // This cancels the effect.
                /*
                if ($lineNumberReference.hasClass('line-numbers-outside')) {
                    fromTop += window.scrollY;
                }
                */

                $this.css({ 'top': fromTop, 'height': height });
            });
        };

        obj.newVersionIncludingChanges = function (motion, version) {
            if (!motion.isAllowed('update')) {
                throw 'No permission to update motion';
            }

            var newHtml = motion.getTextByMode('agreed');
            motion.setTextStrippingLineBreaks(newHtml);

            Motion.inject(motion);
            // save change motion object on server
            Motion.save(motion, {method: 'PATCH'}).then(
                function (success) {
                    $scope.showVersion(motion.getVersion(-1));
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original motion object from server
                    Motion.refresh(motion);
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };

        obj.scrollToDiffBox = function (changeId) {
            obj.mode = 'diff';
            $timeout(function() {
                var $diffBox = $('.diff-box-' + changeId);
                $('html, body').animate({
                    scrollTop: $diffBox.offset().top - 50
                }, 300);
            }, 0, false);
        };

        obj.init = function (_scope, viewMode) {
            $scope = _scope;
            $scope.$evalAsync(function() {
                obj.repositionOriginalAnnotations();
            });
            $scope.$watch(function() {
                return $('.change-recommendation-list').children().length;
            }, obj.repositionOriginalAnnotations);
            $scope.$watch(function () {
                return $scope.change_recommendations.length;
            }, function () {
                if ($scope.change_recommendations.length === 0) {
                    obj.mode = 'original';
                }
            });

            var sizeCheckerLastSize = null,
                sizeCheckerLastClass = null,
                sizeChecker = $interval(function() {
                    var $holder = $(".motion-text-original"),
                        newHeight = $holder.height(),
                        classes = $holder.attr("class");
                    if (newHeight != sizeCheckerLastSize || sizeCheckerLastClass != classes) {
                        sizeCheckerLastSize = newHeight;
                        sizeCheckerLastClass = classes;
                        obj.repositionOriginalAnnotations();
                    }
                }, 100, 0, false);

            $scope.$on('$destroy', function() {
                $interval.cancel(sizeChecker);
            });

            obj.mode = viewMode;
        };

        return obj;
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.pdf', ['OpenSlidesApp.core.pdf'])

.factory('MotionContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Category',
    'Config',
    'Motion',
    function(gettextCatalog, PDFLayout, Category, Config, Motion) {
    /**
     * Provides the content as JS objects for Motions in pdfMake context
     * @constructor
     */

    var createInstance = function(converter, motion, $scope) {

        // title
        var identifier = motion.identifier ? ' ' + motion.identifier : '';
        var title = PDFLayout.createTitle(
                gettextCatalog.getString('Motion') + identifier + ': ' +
                motion.getTitle($scope.version)
        );

        // subtitle
        var subtitleLines = [];
        if (motion.parent_id) {
            var parentMotion = Motion.get(motion.parent_id);
            subtitleLines.push(
                gettextCatalog.getString('Amendment of motion') + ': ' +
                (parentMotion.identifier ? parentMotion.identifier : parentMotion.getTitle())
            );
        }
        subtitleLines.push(gettextCatalog.getString('Sequential number') + ': ' +  motion.id);
        var subtitle = PDFLayout.createSubtitle(subtitleLines);

        // meta data table
        var metaTable = function() {
            var metaTableBody = [];

            // submitters
            var submitters = _.map(motion.submitters, function (submitter) {
                return submitter.get_full_name();
            }).join(', ');
            metaTableBody.push([
                {
                    text: gettextCatalog.getString('Submitters') + ':',
                    style: ['bold', 'grey'],
                },
                {
                    text: submitters,
                    style: 'grey'
                }
            ]);

            // state
            metaTableBody.push([
                {
                    text: gettextCatalog.getString('State') + ':',
                    style: ['bold', 'grey']
                },
                {
                    text: motion.getStateName(),
                    style: 'grey'
                }
            ]);

            // recommendation
            if (motion.getRecommendationName()) {
                metaTableBody.push([
                    {
                        text: Config.get('motions_recommendations_by').value + ':',
                        style: ['bold', 'grey']
                    },
                    {
                        text: motion.getRecommendationName(),
                        style: 'grey'
                    }
                ]);
            }

            // category
            if (motion.category) {
                metaTableBody.push([
                    {
                        text: gettextCatalog.getString('Category') + ':',
                        style: ['bold', 'grey'] },
                    {
                        text: motion.category.name,
                        style: 'grey'
                    }
                ]);
            }

            // voting result
            if (motion.polls.length > 0 && motion.polls[0].has_votes) {
                var column1 = [];
                var column2 = [];
                var column3 = [];
                motion.polls.map(function(poll, index) {
                    if (poll.has_votes) {
                        // votenumber
                        if (motion.polls.length > 1) {
                            column1.push(index + 1 + '. ' + gettextCatalog.getString('Vote'));
                            column2.push('');
                            column3.push('');
                        }
                        // yes
                        var yes = poll.getVote(poll.yes, 'yes');
                        column1.push(gettextCatalog.getString('Yes') + ':');
                        column2.push(yes.value);
                        column3.push(yes.percentStr);
                        // no
                        var no = poll.getVote(poll.no, 'no');
                        column1.push(gettextCatalog.getString('No') + ':');
                        column2.push(no.value);
                        column3.push(no.percentStr);
                        // abstain
                        var abstain = poll.getVote(poll.abstain, 'abstain');
                        column1.push(gettextCatalog.getString('Abstain') + ':');
                        column2.push(abstain.value);
                        column3.push(abstain.percentStr);
                        // votes valid
                        if (poll.votesvalid) {
                            var valid = poll.getVote(poll.votesvalid, 'votesvalid');
                            column1.push(gettextCatalog.getString('Valid votes') + ':');
                            column2.push(valid.value);
                            column3.push(valid.percentStr);
                        }
                        // votes invalid
                        if (poll.votesvalid) {
                            var invalid = poll.getVote(poll.votesinvalid, 'votesinvalid');
                            column1.push(gettextCatalog.getString('Invalid votes') + ':');
                            column2.push(invalid.value);
                            column3.push(invalid.percentStr);
                        }
                        // votes cast
                        if (poll.votescast) {
                            var cast = poll.getVote(poll.votescast, 'votescast');
                            column1.push(gettextCatalog.getString('Votes cast') + ':');
                            column2.push(cast.value);
                            column3.push(cast.percentStr);
                        }
                    }
                });
                metaTableBody.push([
                    {
                        text: gettextCatalog.getString('Voting result') + ':',
                        style: ['bold', 'grey']
                    },
                    {
                        columns: [
                            {
                                text: column1.join('\n'),
                                width: 'auto'
                            },
                            {
                                text: column2.join('\n'),
                                width: 'auto',
                                alignment: 'right'
                            },
                            {
                                text: column3.join('\n'),
                                width: 'auto',
                                alignment: 'right'
                            },
                        ],
                        columnGap: 7,
                        style: 'grey'
                    }
                ]);
            }

            // summary of change recommendations (for motion diff version only)
            if ($scope.viewChangeRecommendations) {
                if ($scope.viewChangeRecommendations.mode == "diff") {
                    var columnLineNumbers = [];
                    var columnChangeType = [];
                    angular.forEach(_.orderBy($scope.change_recommendations, ['line_from']), function(change) {
                        // line numbers column
                        var line;
                        if (change.line_from >= change.line_to - 1) {
                            line = change.line_from;
                        } else {
                            line = change.line_from + ' - ' + (change.line_to - 1);
                        }
                        columnLineNumbers.push(
                            gettextCatalog.getString('Line') + ' ' + line + ': '
                        );
                        // change type column
                        if (change.getType(motion.getVersion($scope.version).text) === 0) {
                            columnChangeType.push(gettextCatalog.getString("Replacement"));
                        } else if (change.getType(motion.getVersion($scope.version).text) === 1) {
                            columnChangeType.push(gettextCatalog.getString("Insertion"));
                        } else if (change.getType(motion.getVersion($scope.version).text) === 2) {
                            columnChangeType.push(gettextCatalog.getString("Deletion"));
                        }
                    });
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Summary of change recommendations'),
                            style: ['bold', 'grey']
                        },
                        {
                            columns: [
                                {
                                    text: columnLineNumbers.join('\n'),
                                    width: 'auto'
                                },
                                {
                                    text: columnChangeType.join('\n'),
                                    width: 'auto'
                                }
                            ],
                            columnGap: 7,
                            style: 'grey'
                        }
                    ]);
                }
            }

            // build table
            // Used placeholder for 'layout' functions whiche are
            // replaced by lineWitdh/lineColor function in pfd-worker.js.
            // TODO: Remove placeholder and us static values for LineWidth and LineColor
            // if pdfmake has fixed this.
            var metaTableJsonString = {
                table: {
                    widths: ['30%','70%'],
                    body: metaTableBody,
                },
                margin: [0, 0, 0, 20],
                layout: '{{motion-placeholder-to-insert-functions-here}}'
            };
            return metaTableJsonString;
        };

        // motion title
        var motionTitle = function() {
            return [{
                text: motion.getTitle($scope.version),
                style: 'heading3'
            }];
        };

        // motion preamble
        var motionPreamble = function () {
            return {
                text: Config.translate(Config.get('motions_preamble').value),
                margin: [0, 10, 0, 0]
            };
        };


        // motion text (with line-numbers)
        var motionText = function() {
            var motionTextContent = motion.getTextByMode($scope.viewChangeRecommendations.mode, $scope.version);
            return converter.convertHTML(motionTextContent, $scope.lineNumberMode);
        };

        // motion reason heading
        var motionReason = function() {
            var reason = [];
            if (motion.getReason($scope.version)) {
                reason.push({
                    text:  gettextCatalog.getString('Reason'),
                    style: 'heading3',
                    marginTop: 25,
                });
                reason.push(converter.convertHTML(motion.getReason($scope.version), $scope.lineNumberMode));
            }
            return reason;
        };


        // getters
        var getTitle = function() {
            return motion.getTitle($scope.version);
        };

        var getIdentifier = function() {
            return motion.identifier ? motion.identifier : '';
        };

        var getCategory = function() {
            return motion.category;
        };

        // Generates content as a pdfmake consumable
        var getContent = function() {
            var content = [
                title,
                subtitle,
                metaTable(),
                motionTitle(),
                motionPreamble(),
                motionText(),
            ];
            if (motionReason()) {
                content.push(motionReason());
            }
            return content;
        };
        return {
            getContent: getContent,
            getTitle: getTitle,
            getIdentifier: getIdentifier,
            getCategory: getCategory
        };
    };

    return {
        createInstance: createInstance
    };
}])

.factory('PollContentProvider', [
    'PDFLayout',
    'Config',
    'User',
    function(PDFLayout, Config, User) {
    /**
    * Generates a content provider for polls
    * @constructor
    * @param {string} title - title of poll
    * @param {string} id - if of poll
    * @param {object} gettextCatalog - for translation
    */
    var createInstance = function(title, id, gettextCatalog) {

        /**
        * Returns a single section on the ballot paper
        * @function
        */
        var createSection = function() {
            var sheetend = 75;
            return {
                stack: [{
                    text: gettextCatalog.getString("Motion") + " " + id,
                    style: 'title',
                }, {
                    text: title,
                    style: 'description'
                },
                PDFLayout.createBallotEntry(gettextCatalog.getString("Yes")),
                PDFLayout.createBallotEntry(gettextCatalog.getString("No")),
                PDFLayout.createBallotEntry(gettextCatalog.getString("Abstain")),
                ],
                margin: [0, 0, 0, sheetend]
            };
        };

        /**
        * Returns Content for single motion
        * @function
        * @param {string} id - if of poll
        */
        var getContent = function() {
            var content = [];
            var amount;
            var amount_method = Config.get('motions_pdf_ballot_papers_selection').value;
            switch (amount_method) {
                    case 'NUMBER_OF_ALL_PARTICIPANTS':
                        amount = User.getAll().length;
                        break;
                    case 'NUMBER_OF_DELEGATES':
                        //TODO: assumption that DELEGATES is always group id 2. This may not be true
                        var group_id = 2;
                        amount = User.filter({where: {'groups_id': {contains:group_id} }}).length;
                        break;
                    case 'CUSTOM_NUMBER':
                        amount = Config.get('motions_pdf_ballot_papers_number').value;
                        break;
                    default:
                        // should not happen.
                        amount = 0;
            }
            var fullpages = Math.floor(amount / 8);

            for (var i=0; i < fullpages; i++) {
                content.push({
                    table: {
                        headerRows: 1,
                        widths: ['*', '*'],
                        body: [
                            [createSection(), createSection()],
                            [createSection(), createSection()],
                            [createSection(), createSection()],
                            [createSection(), createSection()]
                        ],
                        pageBreak: 'after'
                    },
                    layout: PDFLayout.getBallotLayoutLines(),
                    rowsperpage: 4
                });
            }
            amount = amount  - (fullpages * 8);
            if (amount > 0) {
                var partialpagebody = [];
                while (amount > 1) {
                    partialpagebody.push([createSection(), createSection()]);
                    amount -=2;
                }
                if (amount == 1) {
                    partialpagebody.push([createSection(), '']);
                }
                content.push({
                    table: {
                        headerRows: 1,
                        widths: ['50%', '50%'],
                        body: partialpagebody
                    },
                    layout: PDFLayout.getBallotLayoutLines(),
                    rowsperpage: 4
                });
            }
            return content;
        };

        return {
            getContent: getContent,
        };
    };
    return {
        createInstance: createInstance
    };
}])

.factory('MotionCatalogContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Category',
    'Config',
    function(gettextCatalog, PDFLayout, Category, Config) {

    /**
    * Constructor
    * @function
    * @param {object} allMotions - A sorted array of all motions to parse
    * @param {object} $scope - Current $scope
    */
    var createInstance = function(allMotions, $scope) {

        var title = PDFLayout.createTitle(
                Config.translate(Config.get('motions_export_title').value)
        );

        var createPreamble = function() {
            var preambleText = Config.get('motions_export_preamble').value;
            if (preambleText) {
                return {
                    text: preambleText,
                    style: "preamble"
                };
            } else {
                return "";
            }
        };

        var createTOContent = function() {
            var heading = {
                text: gettextCatalog.getString("Table of contents"),
                style: "heading2"
            };

            var toc = [];
            angular.forEach(allMotions, function(motion) {
                var identifier = motion.getIdentifier() ? motion.getIdentifier() : '';
                toc.push(
                    {
                        columns: [
                            {
                                text: identifier,
                                style: 'tableofcontent',
                                width: 70
                            },
                            {
                                text: motion.getTitle(),
                                style: 'tableofcontent'
                            }
                        ]
                    }
                );
            });

            return [
                heading,
                toc,
                PDFLayout.addPageBreak()
            ];
        };

        // function to create the table of catergories (if any)
        var createTOCategories = function() {
            var categories = [];
            _.forEach(allMotions, function(motion) {
                var category = motion.getCategory();
                if (category) {
                    categories.push(category);
                }
            });
            categories = _.uniqBy(categories, 'id');
            if (categories.length > 1) {
                var heading = {
                    text: gettextCatalog.getString("Categories"),
                    style: "heading2"
                };

                var toc = [];
                angular.forEach(_.orderBy(categories, ['prefix']), function(cat) {
                    toc.push(
                        {
                            columns: [
                                {
                                    text: cat.prefix,
                                    style: 'tableofcontent',
                                    width: 50
                                },
                                {
                                    text: cat.name,
                                    style: 'tableofcontent'
                                }
                            ]
                        }
                    );
                });

                return [
                    heading,
                    toc,
                    PDFLayout.addPageBreak()
                ];
            } else {
                // if there are no categories, return "empty string"
                // pdfmake takes "null" literally and throws an error
                return "";
            }
        };

        // returns the pure content of the motion, parseable by pdfmake
        var getContent = function() {
            var motionContent = [];
            angular.forEach(allMotions, function(motion, key) {
                motionContent.push(motion.getContent());
                if (key < allMotions.length - 1) {
                    motionContent.push(PDFLayout.addPageBreak());
                }
            });

            return [
                title,
                createPreamble(),
                createTOCategories(),
                createTOContent(),
                motionContent
            ];
        };
        return {
            getContent: getContent
        };
    };

    return {
        createInstance: createInstance
    };
}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.projector', [
    'OpenSlidesApp.motions',
    'OpenSlidesApp.motions.motionBlockProjector',
])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('motions/motion', {
            template: 'static/templates/motions/slide_motion.html',
        });
    }
])

.controller('SlideMotionCtrl', [
    '$scope',
    'Motion',
    'MotionChangeRecommendation',
    'User',
    function($scope, Motion, MotionChangeRecommendation, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        $scope.mode = $scope.element.mode || 'original';

        Motion.bindOne(id, $scope, 'motion');
        User.bindAll({}, $scope, 'users');
        MotionChangeRecommendation.bindAll({}, $scope, 'change_recommendations');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.site', [
    'OpenSlidesApp.motions',
    'OpenSlidesApp.motions.motionservices',
    'OpenSlidesApp.poll.majority',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.motions.docx',
    'OpenSlidesApp.motions.pdf',
    'OpenSlidesApp.motions.csv',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'motions.motion.list',
            'img_class': 'file-text',
            'title': gettext('Motions'),
            'weight': 300,
            'perm': 'motions.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Motions'),
            'collectionName': 'motions/motion',
            'urlDetailState': 'motions.motion.detail',
            'weight': 300,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('motions', {
                url: '/motions',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Motions'),
                    basePerm: 'motions.can_see',
                },
            })
            .state('motions.motion', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('motions.motion.list', {})
            .state('motions.motion.detail', {
                resolve: {
                    motionId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to motion detail and opens motion edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from motion controller use MotionForm factory instead to open dialog in front of
            // current view without redirect)
            .state('motions.motion.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog', 'Motion',
                    function($stateParams, $state, ngDialog, Motion) {
                        ngDialog.open({
                            template: 'static/templates/motions/motion-form.html',
                            controller: 'MotionUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                motionId: function () {return $stateParams.id;},
                            },
                            preCloseCallback: function () {
                                $state.go('motions.motion.detail', {motion: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ]
            })
            .state('motions.motion.import', {
                url: '/import',
                controller: 'MotionImportCtrl',
            })
            // categories
            .state('motions.category', {
                url: '/category',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Categories'),
                },
            })
            .state('motions.category.list', {})
            .state('motions.category.sort', {
                url: '/sort/{id}',
                controller: 'CategorySortCtrl',
                templateUrl: 'static/templates/motions/category-sort.html',
                resolve: {
                    categoryId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                },
            })
            // MotionBlock
            .state('motions.motionBlock', {
                url: '/blocks',
                abstract: true,
                template: '<ui-view/>',
                data: {
                    title: gettext('Motion blocks'),
                },
            })
            .state('motions.motionBlock.list', {})
            .state('motions.motionBlock.detail', {
                resolve: {
                    motionBlockId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to motionBlock detail and opens motionBlock edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from motionBlock controller use MotionBlockForm factory instead to open dialog in front
            // of current view without redirect)
            .state('motions.motionBlock.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/motions/motion-block-form.html',
                            controller: 'MotionBlockUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                motionBlockId: function () {
                                    return $stateParams.id;
                                }
                            },
                            preCloseCallback: function() {
                                $state.go('motions.motionBlock.detail', {motionBlock: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ],
            });
    }
])

.factory('ChangeRecommendationForm', [
    'gettextCatalog',
    'Editor',
    'Config',
    function(gettextCatalog, Editor, Config) {
        return {
            // ngDialog for motion form
            getCreateDialog: function (motion, version, lineFrom, lineTo) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motion: function() {
                            return motion;
                        },
                        version: function() {
                            return version;
                        },
                        lineFrom: function() {
                            return lineFrom;
                        },
                        lineTo: function() {
                            return lineTo;
                        }
                    }
                };
            },
            getEditDialog: function(change) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationUpdateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        change: function() {
                            return change;
                        }
                    }
                };
            },
            // angular-formly fields for motion form
            getFormFields: function (line_from, line_to) {
                return [
                    {
                        key: 'identifier',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Identifier')
                        },
                        hide: true
                    },
                    {
                        key: 'motion_version_id',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Motion')
                        },
                        hide: true
                    },
                    {
                        key: 'line_from',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('From Line')
                        },
                        hide: true
                    },
                    {
                        key: 'line_to',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('To Line')
                        },
                        hide: true
                    },
                    {
                        key: 'type',
                        type: 'radio-buttons',
                        templateOptions: {
                            name: 'type',
                            options: [
                                {name: gettextCatalog.getString('Replacement'), value: 0},
                                {name: gettextCatalog.getString('Insertion'), value: 1},
                                {name: gettextCatalog.getString('Deletion'), value: 2}
                            ]
                        }
                    },
                    {
                        key: 'text',
                        type: 'editor',
                        templateOptions: {
                            label: (
                                line_from == line_to - 1 ?
                                gettextCatalog.getString('Text in line %from%').replace(/%from%/, line_from) :
                                gettextCatalog.getString('Text from line %from% to %to%')
                                  .replace(/%from%/, line_from).replace(/%to%/, line_to - 1)
                            ),
                            required: false
                        },
                        data: {
                            ckeditorOptions: Editor.getOptions()
                        }
                    }
                ];
            }
        };
    }
])

// Service for generic motion form (create and update)
.factory('MotionForm', [
    'gettextCatalog',
    'operator',
    'Editor',
    'MotionComment',
    'Category',
    'Config',
    'Mediafile',
    'MotionBlock',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'AgendaTree',
    function (gettextCatalog, operator, Editor, MotionComment, Category, Config, Mediafile, MotionBlock, Tag, User, Workflow, Agenda, AgendaTree) {
        return {
            // ngDialog for motion form
            getDialog: function (motion) {
                return {
                    template: 'static/templates/motions/motion-form.html',
                    controller: motion ? 'MotionUpdateCtrl' : 'MotionCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motionId: function () {return motion ? motion.id : void 0;},
                    },
                };
            },
            // angular-formly fields for motion form
            getFormFields: function (isCreateForm) {
                var workflows = Workflow.getAll();
                var images = Mediafile.getAllImages();
                var formFields = [
                {
                    key: 'identifier',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Identifier')
                    },
                    hide: true
                },
                {
                    key: 'submitters_id',
                    type: 'select-multiple',
                    templateOptions: {
                        label: gettextCatalog.getString('Submitters'),
                        options: User.getAll(),
                        ngOptions: 'option.id as option.full_name for option in to.options',
                        placeholder: gettextCatalog.getString('Select or search a submitter ...')
                    },
                    hide: !operator.hasPerms('motions.can_manage')
                },
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    template: '<p class="spacer-top-lg no-padding">' + Config.translate(Config.get('motions_preamble').value) + '</p>'
                },
                {
                    key: 'text',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Text'),
                        required: true
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    }
                },
                {
                    key: 'reason',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Reason'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    }
                },
                {
                    key: 'disable_versioning',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Trivial change'),
                        description: gettextCatalog.getString("Don't create a new version.")
                    },
                    hide: true
                },
                {
                    key: 'showAsAgendaItem',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Show as agenda item'),
                        description: gettextCatalog.getString('If deactivated the motion appears as internal item on agenda.')
                    },
                    hide: !(operator.hasPerms('motions.can_manage') && operator.hasPerms('agenda.can_manage'))
                }];

                // parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'agenda_parent_item_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }

                // motion comments
                formFields = formFields.concat(MotionComment.getFormFields());

                // more
                formFields.push(
                    {
                        key: 'more',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show extended fields')
                        },
                        hide: !operator.hasPerms('motions.can_manage')
                    },
                    {
                        template: '<hr class="smallhr">',
                        hideExpression: '!model.more'
                    }
                );
                // attachments
                if (Mediafile.getAll().length > 0) {
                    formFields.push({
                        key: 'attachments_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Attachment'),
                            options: Mediafile.getAll(),
                            ngOptions: 'option.id as option.title_or_filename for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search an attachment ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // category
                if (Category.getAll().length > 0) {
                    formFields.push({
                        key: 'category_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Category'),
                            options: Category.getAll(),
                            ngOptions: 'option.id as option.name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a category ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // motion block
                if (MotionBlock.getAll().length > 0) {
                    formFields.push({
                        key: 'motion_block_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Motion block'),
                            options: MotionBlock.getAll(),
                            ngOptions: 'option.id as option.title for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a motion block ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // origin
                formFields.push({
                    key: 'origin',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Origin'),
                    },
                    hideExpression: '!model.more'
                });
                // tags
                if (Tag.getAll().length > 0) {
                    formFields.push({
                        key: 'tags_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Tags'),
                            options: Tag.getAll(),
                            ngOptions: 'option.id as option.name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a tag ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // supporters
                if (Config.get('motions_min_supporters').value > 0) {
                    formFields.push({
                        key: 'supporters_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Supporters'),
                            options: User.getAll(),
                            ngOptions: 'option.id as option.full_name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a supporter ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // workflows
                if (workflows.length > 1) {
                    formFields.push({
                        key: 'workflow_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Workflow'),
                            optionsAttr: 'bs-options',
                            options: workflows,
                            ngOptions: 'option.id as option.name | translate for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a workflow ...')
                        },
                        hideExpression: '!model.more',
                    });
                }

                return formFields;
            }
        };
    }
])

.factory('CategoryForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getDialog: function (category) {
                return {
                    template: 'static/templates/motions/category-form.html',
                    controller: category ? 'CategoryUpdateCtrl' : 'CategoryCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        categoryId: function () {return category ? category.id : void 0;},
                    },
                };

            },
            getFormFields: function () {
                return [
                    {
                        key: 'prefix',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Prefix')
                        },
                    },
                    {
                        key: 'name',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Name')
                        },
                    }
                ];
            },
        };
    }
])

// Provide generic motionpoll form fields for poll update view
.factory('MotionPollForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getFormFields: function () {
                return [
                {
                    key: 'yes',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Yes'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'no',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('No'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'abstain',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Abstain'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'votesvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Valid votes'),
                        type: 'number'
                    }
                },
                {
                    key: 'votesinvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Invalid votes'),
                        type: 'number'
                    }
                },
                {
                    key: 'votescast',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Votes cast'),
                        type: 'number'
                    }
                }];
            }
        };
    }
])

// Cache for MotionPollDetailCtrl so that users choices are keeped during user actions (e. g. save poll form).
.value('MotionPollDetailCtrlCache', {})

// Child controller of MotionDetailCtrl for each single poll.
.controller('MotionPollDetailCtrl', [
    '$scope',
    'MajorityMethodChoices',
    'Config',
    'MotionPollDetailCtrlCache',
    function ($scope, MajorityMethodChoices, Config, MotionPollDetailCtrlCache) {
        // Define choices.
        $scope.methodChoices = MajorityMethodChoices;
        // TODO: Get $scope.baseChoices from config_variables.py without copying them.

        // Setup empty cache with default values.
        if (typeof MotionPollDetailCtrlCache[$scope.poll.id] === 'undefined') {
            MotionPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.config('motions_poll_default_majority_method'),
            };
        }

        // Fetch users choices from cache.
        $scope.method = MotionPollDetailCtrlCache[$scope.poll.id].method;

        // Define result function.
        $scope.isReached = function () {
            return $scope.poll.isReached($scope.method);
        };

        // Define template controll function
        $scope.hideMajorityCalculation = function () {
            return typeof $scope.isReached() === 'undefined' && $scope.method !== 'disabled';
        };

        // Save current values to cache on detroy of this controller.
        $scope.$on('$destroy', function() {
            MotionPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.method,
            };
        });
    }
])

.controller('MotionListCtrl', [
    '$scope',
    '$state',
    '$http',
    'gettext',
    'gettextCatalog',
    'ngDialog',
    'MotionForm',
    'Motion',
    'Category',
    'Config',
    'Tag',
    'Workflow',
    'User',
    'Agenda',
    'MotionBlock',
    'MotionChangeRecommendation',
    'MotionCsvExport',
    'MotionDocxExport',
    'MotionContentProvider',
    'MotionCatalogContentProvider',
    'PdfMakeConverter',
    'PdfMakeDocumentProvider',
    'HTMLValidizer',
    'Projector',
    'ProjectionDefault',
    'osTableFilter',
    'osTableSort',
    'PdfCreate',
    function($scope, $state, $http, gettext, gettextCatalog, ngDialog, MotionForm, Motion,
                Category, Config, Tag, Workflow, User, Agenda, MotionBlock, MotionChangeRecommendation,
                MotionCsvExport, MotionDocxExport, MotionContentProvider, MotionCatalogContentProvider,
                PdfMakeConverter, PdfMakeDocumentProvider, HTMLValidizer, Projector, ProjectionDefault,
                osTableFilter, osTableSort, PdfCreate) {
        Motion.bindAll({}, $scope, 'motions');
        Category.bindAll({}, $scope, 'categories');
        MotionBlock.bindAll({}, $scope, 'motionBlocks');
        Tag.bindAll({}, $scope, 'tags');
        Workflow.bindAll({}, $scope, 'workflows');
        User.bindAll({}, $scope, 'users');
        Projector.bindAll({}, $scope, 'projectors');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motions'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.alert = {};

        // collect all states and all recommendations of all workflows
        $scope.states = [];
        $scope.recommendations = [];
        var workflows = Workflow.getAll();
        _.forEach(workflows, function (workflow) {
            var workflowHeader = {
                headername: workflow.name,
                workflowHeader: true,
            };
            $scope.states.push(workflowHeader);
            $scope.recommendations.push(workflowHeader);
            _.forEach(workflow.states, function (state) {
                $scope.states.push(state);
                if (state.recommendation_label) {
                    $scope.recommendations.push(state);
                }
            });
        });

        $scope.stateFilter = [];
        var updateStateFilter = function () {
            if (_.indexOf($scope.filter.multiselectFilters.state, -1) > -1) { // contains -1
                $scope.stateFilter = _.filter($scope.filter.multiselectFilters.state, function (id) {
                    return id >= 0;
                }); // remove -1
                _.forEach($scope.states, function (state) {
                    if (!state.workflowHeader) {
                        if (state.getNextStates().length === 0) { // done state
                            $scope.stateFilter.push(state.id);
                        }
                    }
                });
            } else {
                $scope.stateFilter = _.clone($scope.filter.multiselectFilters.state);
            }
        };

        // Filtering
        $scope.filter = osTableFilter.createInstance('MotionTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                state: [],
                category: [],
                motionBlock: [],
                tag: [],
                recommendation: [],
            };
        }
        updateStateFilter();
        $scope.filter.propertyList = ['identifier', 'origin'];
        $scope.filter.propertyFunctionList = [
            function (motion) {return motion.getTitle();},
            function (motion) {return motion.getText();},
            function (motion) {return motion.getReason();},
            function (motion) {return motion.category ? motion.category.name : '';},
            function (motion) {return motion.motionBlock ? motion.motionBlock.name : '';},
            function (motion) {return motion.recommendation ? motion.getRecommendationName() : '';},
        ];
        $scope.filter.propertyDict = {
            'submitters' : function (submitter) {
                return submitter.get_short_name();
            },
            'supporters' : function (submitter) {
                return supporter.get_short_name();
            },
            'tags' : function (tag) {
                return tag.name;
            },
        };
        $scope.getItemId = {
            state: function (motion) {return motion.state_id;},
            category: function (motion) {return motion.category_id;},
            motionBlock: function (motion) {return motion.motion_block_id;},
            tag: function (motion) {return motion.tags_id;},
            recommendation: function (motion) {return motion.recommendation_id;},
        };
        $scope.operateStateFilter = function (id, danger) {
            $scope.filter.operateMultiselectFilter('state', id, danger);
            updateStateFilter();
        };
        $scope.resetFilters = function () {
            $scope.filter.reset();
            updateStateFilter();
        };
        // Sorting
        $scope.sort = osTableSort.createInstance();
        $scope.sort.column = 'identifier';
        $scope.sortOptions = [
            {name: 'identifier',
             display_name: gettext('Identifier')},
            {name: 'getTitle()',
             display_name: gettext('Title')},
            {name: 'submitters',
             display_name: gettext('Submitters')},
            {name: 'category.name',
             display_name: gettext('Category')},
            {name: 'motionBlock.title',
             display_name: gettext('Motion block')},
            {name: 'state.name',
             display_name: gettext('State')},
            {name: 'log_messages[log_messages.length-1].time',
             display_name: gettext('Creation date')},
            {name: 'log_messages[0].time',
             display_name: gettext('Last modified')},
        ];

        // pagination
        $scope.currentPage = 1;
        $scope.itemsPerPage = 25;
        $scope.limitBegin = 0;
        $scope.pageChanged = function() {
            $scope.limitBegin = ($scope.currentPage - 1) * $scope.itemsPerPage;
        };


        // update state
        $scope.updateState = function (motion, state_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {'state': state_id});
        };
        // reset state
        $scope.resetState = function (motion) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {});
        };
        // update recommendation
        $scope.updateRecommendation = function (motion, recommendation_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {'recommendation': recommendation_id});
        };
        // reset recommendation
        $scope.resetRecommendation = function (motion) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {});
        };

        $scope.hasTag = function (motion, tag) {
            return _.indexOf(motion.tags_id, tag.id) > -1;
        };

        $scope.save = function (motion) {
            Motion.save(motion, {method: 'PATCH'});
        };
        // delete single motion
        $scope.delete = function (motion) {
            Motion.destroy(motion.id);
        };
        $scope.toggleTag = function (motion, tag) {
            if ($scope.hasTag(motion, tag)) {
                // remove
                motion.tags_id = _.filter(motion.tags_id, function (tag_id){
                    return tag_id != tag.id;
                });
            } else {
                motion.tags_id.push(tag.id);
            }
            $scope.save(motion);
        };
        $scope.toggleCategory = function (motion, category) {
            if (motion.category_id == category.id) {
                motion.category_id = null;
            } else {
                motion.category_id = category.id;
            }
            $scope.save(motion);
        };
        $scope.toggleMotionBlock = function (motion, block) {
            if (motion.motion_block_id == block.id) {
                motion.motion_block_id = null;
            } else {
                motion.motion_block_id = block.id;
            }
            $scope.save(motion);
        };

        // open new/edit dialog
        $scope.openDialog = function (motion) {
            ngDialog.open(MotionForm.getDialog(motion));
        };

        // Export as a pdf file
        $scope.pdfExport = function() {
            var filename = gettextCatalog.getString("Motions") + ".pdf";
            var image_sources = [];
            $scope.viewChangeRecommendations = {};
            $scope.viewChangeRecommendations.mode = Config.get('motions_recommendation_text_mode').value;
            $scope.lineNumberMode = Config.get('motions_default_line_numbering').value;

            //save the arrays of the filtered motions to an array
            angular.forEach($scope.motionsFiltered, function (motion) {
                $scope.change_recommendations = MotionChangeRecommendation.filter({
                    'where': {'motion_version_id': {'==': motion.active_version}}
                });
                var text = motion.getTextByMode($scope.viewChangeRecommendations.mode, null);
                var content = HTMLValidizer.validize(text) + HTMLValidizer.validize(motion.getReason());
                var map = Function.prototype.call.bind([].map);
                var tmp_image_sources = map($(content).find("img"), function(element) {
                    return element.getAttribute("src");
                });
                image_sources = image_sources.concat(tmp_image_sources);
            });

            //post-request to convert the images. Async.
            $http.post('/core/encode_media/', JSON.stringify(image_sources)).then(function (success) {
                var converter = PdfMakeConverter.createInstance(success.data.images);
                var motionContentProviderArray = [];

                //convert the filtered motions to motionContentProviders
                angular.forEach($scope.motionsFiltered, function (motion) {
                    motionContentProviderArray.push(MotionContentProvider.createInstance(converter, motion, $scope, User, $http));
                });
                var motionCatalogContentProvider = MotionCatalogContentProvider.createInstance(motionContentProviderArray, $scope, User, Category);
                var documentProvider = PdfMakeDocumentProvider.createInstance(motionCatalogContentProvider);

                PdfCreate.download(documentProvider.getDocument(), filename);
            });
        };

        // Export as a csv file
        $scope.csvExport = function () {
            MotionCsvExport.export($scope.motionsFiltered);
        };
        // Export as docx file
        $scope.docxExport = function () {
            MotionDocxExport.export($scope.motionsFiltered, $scope.categories);
        };

        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes from filtered motions
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            angular.forEach($scope.motionsFiltered, function (motion) {
                motion.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.motions, function (motion) {
                    motion.selected = false;
                });
            }
        };
        var selectModeAction = function (predicate) {
            angular.forEach($scope.motionsFiltered, function (motion) {
                if (motion.selected) {
                    predicate(motion);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete selected motions
        $scope.deleteMultiple = function () {
            selectModeAction(function (motion) {
                $scope.delete(motion);
            });
        };
        // set status for selected motions
        $scope.setStatusMultiple = function (stateId) {
            selectModeAction(function (motion) {
                $scope.updateState(motion, stateId);
            });
        };
        // set category for selected motions
        $scope.setCategoryMultiple = function (categoryId) {
            selectModeAction(function (motion) {
                motion.category_id = categoryId === 'no_category_selected' ? null : categoryId;
                $scope.save(motion);
            });
        };
        // set status for selected motions
        $scope.setMotionBlockMultiple = function (motionBlockId) {
            selectModeAction(function (motion) {
                motion.motion_block_id = motionBlockId === 'no_motionBlock_selected' ? null : motionBlockId;
                $scope.save(motion);
            });
        };
    }
])

.controller('MotionDetailCtrl', [
    '$scope',
    '$http',
    '$timeout',
    'operator',
    'ngDialog',
    'MotionForm',
    'ChangeRecommmendationCreate',
    'ChangeRecommmendationView',
    'MotionChangeRecommendation',
    'MotionPDFExport',
    'Motion',
    'MotionComment',
    'Category',
    'Mediafile',
    'Tag',
    'User',
    'Workflow',
    'Config',
    'motionId',
    'MotionInlineEditing',
    'MotionCommentsInlineEditing',
    'Projector',
    'ProjectionDefault',
    'MotionBlock',
    function($scope, $http, $timeout, operator, ngDialog, MotionForm,
             ChangeRecommmendationCreate, ChangeRecommmendationView, MotionChangeRecommendation, MotionPDFExport,
             Motion, MotionComment, Category, Mediafile, Tag, User, Workflow, Config, motionId, MotionInlineEditing,
             MotionCommentsInlineEditing, Projector, ProjectionDefault, MotionBlock) {
        var motion = Motion.get(motionId);
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');
        MotionBlock.bindAll({}, $scope, 'motionBlocks');
        $scope.$watch(function () {
            return MotionChangeRecommendation.lastModified();
        }, function () {
            $scope.change_recommendations = MotionChangeRecommendation.filter({
                'where': {'motion_version_id': {'==': motion.active_version}}
            });
            if ($scope.change_recommendations.length === 0) {
                $scope.setProjectionMode($scope.projectionModes[0]);
            }
            if ($scope.change_recommendations.length > 0) {
                $scope.inlineEditing.disable();
            }
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            var projectiondefault = ProjectionDefault.filter({name: 'motions'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.$watch(function () {
            return Motion.lastModified(motionId);
        }, function () {
            $scope.motion = Motion.get(motionId);
            MotionComment.populateFields($scope.motion);
        });
        $scope.projectionModes = [
            {mode: 'original',
            label: 'Original version'},
            {mode: 'changed',
            label: 'Changed version'},
            {mode: 'diff',
            label: 'Diff version'},
            {mode: 'agreed',
            label: 'Final version'},
        ];
        $scope.projectionMode = $scope.projectionModes[0];
        if (motion.isProjected().length) {
            var modeMapping = motion.isProjectedWithMode();
            _.forEach($scope.projectionModes, function (mode) {
                if (mode.mode === modeMapping[0].mode) {
                    $scope.projectionMode = mode;
                }
            });
        }
        $scope.setProjectionMode = function (mode) {
            $scope.projectionMode = mode;
            var isProjected = motion.isProjectedWithMode();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.mode != mode.mode) { // change the mode if it is different
                        motion.project(mapping.projectorId, mode.mode);
                    }
                });
            }
        };
        $scope.commentsFields = Config.get('motions_comments').value;
        $scope.commentFieldForState = MotionComment.getFieldNameForFlag('forState');
        $scope.commentFieldForRecommendation = MotionComment.getFieldNameForFlag('forRecommendation');
        $scope.version = motion.active_version;
        $scope.isCollapsed = true;
        $scope.lineNumberMode = Config.get('motions_default_line_numbering').value;
        $scope.setLineNumberMode = function(mode) {
            $scope.lineNumberMode = mode;
        };

        if (motion.parent_id) {
            Motion.bindOne(motion.parent_id, $scope, 'parent');
        }
        $scope.amendments = Motion.filter({parent_id: motion.id});

        $scope.highlight = 0;
        $scope.scrollToAndHighlight = function (line) {
            $scope.highlight = line;

            // The same line number can occur twice in diff view; we scroll to the first one in this case
            var scrollTop = null;
            $(".line-number-" + line).each(function() {
                var top = $(this).offset().top;
                if (top > 0 && (scrollTop === null || top < scrollTop)) {
                    scrollTop = top;
                }
            });

            if (scrollTop) {
                // Scroll local; 50 pixel above the line, so it's not completely squeezed to the screen border
                $('html, body').animate({
                    'scrollTop': scrollTop - 50
                }, 1000);
                // remove the line highlight after 2 seconds.
                $timeout(function () {
                    $scope.highlight = 0;
                }, 2000);
            }
        };

        // open edit dialog
        $scope.openDialog = function (motion) {
            if ($scope.inlineEditing.active) {
                $scope.inlineEditing.disable();
            }
            ngDialog.open(MotionForm.getDialog(motion));
        };
        $scope.save = function (motion) {
            Motion.save(motion, {method: 'PATCH'});
        };
        // support
        $scope.support = function () {
            $http.post('/rest/motions/motion/' + motion.id + '/support/');
        };
        // unsupport
        $scope.unsupport = function () {
            $http.delete('/rest/motions/motion/' + motion.id + '/support/');
        };
        // open dialog for new amendment
        $scope.newAmendment = function () {
            var dialog = MotionForm.getDialog();
            if (typeof dialog.scope === 'undefined') {
                dialog.scope = {};
            }
            dialog.scope = $scope;
            ngDialog.open(dialog);
        };
        // update state
        $scope.updateState = function (state_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {'state': state_id});
        };
        // reset state
        $scope.reset_state = function () {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {});
        };
        // toggle functions for meta information
        $scope.toggleCategory = function (category) {
            if ($scope.motion.category_id == category.id) {
                $scope.motion.category_id = null;
            } else {
                $scope.motion.category_id = category.id;
            }
            $scope.save($scope.motion);
        };
        $scope.toggleMotionBlock = function (block) {
            if ($scope.motion.motion_block_id == block.id) {
                $scope.motion.motion_block_id = null;
            } else {
                $scope.motion.motion_block_id = block.id;
            }
            $scope.save($scope.motion);

        };
        $scope.toggleTag = function (tag) {
            if (_.indexOf($scope.motion.tags_id, tag.id) > -1) {
                // remove
                $scope.motion.tags_id = _.filter($scope.motion.tags_id,
                    function (tag_id){
                        return tag_id != tag.id;
                    }
                );
            } else {
                $scope.motion.tags_id.push(tag.id);
            }
            $scope.save($scope.motion);
        };
        // save additional state field
        $scope.saveAdditionalStateField = function (stateExtension) {
            if (stateExtension) {
                motion["comment " + $scope.commentFieldForState] = stateExtension;
                $scope.save(motion);
            }
        };
        // save additional recommendation field
        $scope.saveAdditionalRecommendationField = function (recommendationExtension) {
            if (recommendationExtension) {
                motion["comment " + $scope.commentFieldForRecommendation] = recommendationExtension;
                $scope.save(motion);
            }
        };
        // update recommendation
        $scope.updateRecommendation = function (recommendation_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {'recommendation': recommendation_id});
        };
        // reset recommendation
        $scope.resetRecommendation = function () {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {});
        };
        // create poll
        $scope.create_poll = function () {
            $http.post('/rest/motions/motion/' + motion.id + '/create_poll/', {});
        };
        // open poll update dialog
        $scope.openPollDialog = function (poll, voteNumber) {
            ngDialog.open({
                template: 'static/templates/motions/motion-poll-form.html',
                controller: 'MotionPollUpdateCtrl',
                className: 'ngdialog-theme-default',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    motionpollId: function () {
                        return poll.id;
                    },
                    voteNumber: function () {
                        return voteNumber;
                    }
                }
            });
        };
        // delete poll
        $scope.delete_poll = function (poll) {
            poll.DSDestroy();
        };
        // show specific version
        $scope.showVersion = function (version) {
            $scope.version = version.id;
            $scope.inlineEditing.setVersion(motion, version.id);
            $scope.createChangeRecommendation.setVersion(motion, version.id);
        };
        // permit specific version
        $scope.permitVersion = function (version) {
            $http.put('/rest/motions/motion/' + motion.id + '/manage_version/',
                {'version_number': version.version_number})
                .then(function(success) {
                    $scope.showVersion(version);
                });
        };
        // delete specific version
        $scope.deleteVersion = function (version) {
            $http.delete('/rest/motions/motion/' + motion.id + '/manage_version/',
                    {headers: {'Content-Type': 'application/json'},
                     data: JSON.stringify({version_number: version.version_number})})
                .then(function(success) {
                    $scope.showVersion({id: motion.active_version});
                });
        };
        // check if user is allowed to see at least one comment field
        $scope.isAllowedToSeeCommentField = function () {
            var isAllowed = false;
            if ($scope.commentsFields.length > 0) {
                isAllowed = operator.hasPerms('motions.can_see_and_manage_comments') || _.find(
                        $scope.commentsFields,
                        function(field) {
                            return field.public && !field.forState && !field.forRecommendation;
                        }
                );
            }
            return Boolean(isAllowed);
        };

        // Inline editing functions
        $scope.inlineEditing = MotionInlineEditing.createInstance($scope, motion,
            'view-original-text-inline-editor', true,
            function (obj) {
                return motion.getTextWithLineBreaks($scope.version);
            },
            function (obj) {
                motion.reason = motion.getReason(-1);
                motion.setTextStrippingLineBreaks(obj.editor.getData());
                motion.disable_versioning = (obj.trivialChange &&
                    Config.get('motions_allow_disable_versioning').value);
            }
        );
        $scope.commentsInlineEditing = MotionCommentsInlineEditing.createInstances($scope, motion);

        // Change recommendation creation functions
        $scope.createChangeRecommendation = ChangeRecommmendationCreate;
        $scope.createChangeRecommendation.init($scope, motion);

        // Change recommendation viewing
        $scope.viewChangeRecommendations = ChangeRecommmendationView;
        $scope.viewChangeRecommendations.init($scope, Config.get('motions_recommendation_text_mode').value);

        // PDF creating functions
        $scope.pdfExport = MotionPDFExport;
        $scope.pdfExport.init($scope);
    }
])

.controller('ChangeRecommendationUpdateCtrl', [
    '$scope',
    'MotionChangeRecommendation',
    'ChangeRecommendationForm',
    'change',
    'ErrorMessage',
    function ($scope, MotionChangeRecommendation, ChangeRecommendationForm, change, ErrorMessage) {
        $scope.alert = {};
        $scope.model = angular.copy(change);

        // get all form fields
        $scope.formFields = ChangeRecommendationForm.getFormFields(change.line_from, change.line_to);
        // save motion
        $scope.save = function (change) {
            // inject the changed change recommendation (copy) object back into DS store
            MotionChangeRecommendation.inject(change);
            // save changed change recommendation object on server
            MotionChangeRecommendation.save(change).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    MotionChangeRecommendation.refresh(change);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('ChangeRecommendationCreateCtrl', [
    '$scope',
    'Motion',
    'MotionChangeRecommendation',
    'ChangeRecommendationForm',
    'Config',
    'diffService',
    'motion',
    'version',
    'lineFrom',
    'lineTo',
    function($scope, Motion, MotionChangeRecommendation, ChangeRecommendationForm, Config, diffService, motion,
             version, lineFrom, lineTo) {
        $scope.alert = {};

        var html = motion.getTextWithLineBreaks(version),
            lineData = diffService.extractRangeByLineNumbers(html, lineFrom, lineTo);

        $scope.model = {
            text: lineData.outerContextStart + lineData.innerContextStart +
                lineData.html + lineData.innerContextEnd + lineData.outerContextEnd,
            line_from: lineFrom,
            line_to: lineTo,
            motion_version_id: version,
            type: 0
        };

        // get all form fields
        $scope.formFields = ChangeRecommendationForm.getFormFields(lineFrom, lineTo);
        // save motion
        $scope.save = function (motion) {
            MotionChangeRecommendation.create(motion).then(
                function(success) {
                    $scope.closeThisDialog();
                }
            );
        };
    }
])

.controller('MotionCreateCtrl', [
    '$scope',
    '$state',
    'gettext',
    'gettextCatalog',
    'operator',
    'Motion',
    'MotionForm',
    'Category',
    'Config',
    'Mediafile',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'AgendaUpdate',
    'ErrorMessage',
    function($scope, $state, gettext, gettextCatalog, operator, Motion, MotionForm,
        Category, Config, Mediafile, Tag, User, Workflow, Agenda, AgendaUpdate, ErrorMessage) {
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');

        $scope.model = {};
        $scope.alert = {};

        // Check whether this is a new amendment.
        var isAmendment = $scope.$parent.motion && $scope.$parent.motion.id;

        // Set default values for create form
        // ... for amendments add parent_id
        if (isAmendment) {
            if (Config.get('motions_amendments_apply_text').value) {
                $scope.model.text = $scope.$parent.motion.getText();
            }
            $scope.model.title = $scope.$parent.motion.getTitle();
            $scope.model.parent_id = $scope.$parent.motion.id;
            $scope.model.category_id = $scope.$parent.motion.category_id;
            $scope.model.motion_block_id = $scope.$parent.motion.motion_block_id;
            Motion.bindOne($scope.model.parent_id, $scope, 'parent');
        }
        // ... preselect default workflow
        if (operator.hasPerms('motions.can_manage')) {
            $scope.model.workflow_id = Config.get('motions_workflow').value;
        }
        // get all form fields
        $scope.formFields = MotionForm.getFormFields(true);

        // save motion
        $scope.save = function (motion, gotoDetailView) {
            Motion.create(motion).then(
                function(success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (motion.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: motion.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id, changes);
                    if (isAmendment || gotoDetailView) {
                        $state.go('motions.motion.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('MotionUpdateCtrl', [
    '$scope',
    '$state',
    'Motion',
    'Category',
    'Config',
    'Mediafile',
    'MotionForm',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'AgendaUpdate',
    'motionId',
    'ErrorMessage',
    function($scope, $state, Motion, Category, Config, Mediafile, MotionForm, Tag,
        User, Workflow, Agenda, AgendaUpdate, motionId, ErrorMessage) {
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');
        $scope.alert = {};

        // set initial values for form model by create deep copy of motion object
        // so list/detail view is not updated while editing
        var motion = Motion.get(motionId);
        $scope.model = angular.copy(motion);
        $scope.model.disable_versioning = false;
        $scope.model.more = false;

        // get all form fields
        $scope.formFields = MotionForm.getFormFields();
        // override default values for update form
        for (var i = 0; i < $scope.formFields.length; i++) {
            if ($scope.formFields[i].key == "identifier") {
                // show identifier field
               $scope.formFields[i].hide = false;
            }
            if ($scope.formFields[i].key == "title") {
                // get title of latest version
                $scope.formFields[i].defaultValue = motion.getTitle(-1);
            }
            if ($scope.formFields[i].key == "text") {
                // get text of latest version
                $scope.formFields[i].defaultValue = motion.getText(-1);
            }
            if ($scope.formFields[i].key == "reason") {
                // get reason of latest version
                $scope.formFields[i].defaultValue = motion.getReason(-1);
            }
            if ($scope.formFields[i].key == "disable_versioning") {
                if (Config.get('motions_allow_disable_versioning').value && motion.state.versioning) {
                    // check current state if versioning is active
                    $scope.formFields[i].hide = false;
                }
            }
            if ($scope.formFields[i].key == "showAsAgendaItem" && motion.agenda_item) {
                // get state from agenda item (hidden/internal or agenda item)
                $scope.formFields[i].defaultValue = !motion.agenda_item.is_hidden;
            }
            if ($scope.formFields[i].key == "workflow_id") {
               // get saved workflow id from state
               $scope.formFields[i].defaultValue = motion.state.workflow_id;
            }
            if ($scope.formFields[i].key == "agenda_parent_item_id") {
                // get current parent_id of the agenda item
                $scope.formFields[i].defaultValue = motion.agenda_item.parent_id;
            }
        }

        // save motion
        $scope.save = function (motion, gotoDetailView) {
            // inject the changed motion (copy) object back into DS store
            Motion.inject(motion);
            // save change motion object on server
            Motion.save(motion).then(
                function(success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (motion.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: motion.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    if (gotoDetailView) {
                        $state.go('motions.motion.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original motion object from server
                    Motion.refresh(motion);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('MotionPollUpdateCtrl', [
    '$scope',
    'gettextCatalog',
    'MotionPoll',
    'MotionPollForm',
    'motionpollId',
    'voteNumber',
    'ErrorMessage',
    function($scope, gettextCatalog, MotionPoll, MotionPollForm, motionpollId,
        voteNumber, ErrorMessage) {
        // set initial values for form model by create deep copy of motionpoll object
        // so detail view is not updated while editing poll
        var motionpoll = MotionPoll.get(motionpollId);
        $scope.model = angular.copy(motionpoll);
        $scope.voteNumber = voteNumber;
        $scope.formFields = MotionPollForm.getFormFields();
        $scope.alert = {};

        // save motionpoll
        $scope.save = function (poll) {
            poll.DSUpdate({
                    motion_id: poll.motion_id,
                    votes: {"Yes": poll.yes, "No": poll.no, "Abstain": poll.abstain},
                    votesvalid: poll.votesvalid,
                    votesinvalid: poll.votesinvalid,
                    votescast: poll.votescast
            })
            .then(function(success) {
                $scope.alert.show = false;
                $scope.closeThisDialog();
            }, function(error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

.controller('MotionImportCtrl', [
    '$scope',
    '$q',
    'gettext',
    'Category',
    'Motion',
    'User',
    'MotionCsvExport',
    function($scope, $q, gettext, Category, Motion, User, MotionCsvExport) {
        // set initial data for csv import
        $scope.motions = [];

        // set csv
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };

        var FIELDS = ['identifier', 'title', 'text', 'reason', 'submitter', 'category', 'origin'];
        $scope.motions = [];
        $scope.onCsvChange = function (csv) {
            $scope.motions = [];
            var motions = [];
            _.forEach(csv.data, function (row) {
                if (row.length >= 3) {
                    var filledRow = _.zipObject(FIELDS, row);
                    motions.push(filledRow);
                }
            });

            _.forEach(motions, function (motion) {
                motion.selected = true;
                // identifier
                if (motion.identifier !== '') {
                    // All motion objects are already loaded via the resolve statement from ui-router.
                    var motions = Motion.getAll();
                    if (_.find(motions, function (item) {
                        return item.identifier === motion.identifier;
                    })) {
                        motion.importerror = true;
                        motion.identifier_error = gettext('Error: Identifier already exists.');
                    }
                }
                // title
                if (!motion.title) {
                    motion.importerror = true;
                    motion.title_error = gettext('Error: Title is required.');
                }
                // text
                if (!motion.text) {
                    motion.importerror = true;
                    motion.text_error = gettext('Error: Text is required.');
                } else if (!motion.text.startsWith('<p>')) {
                    motion.text = '<p>' + motion.text + '</p>';
                }
                // Reason
                if (motion.reason && !motion.reason.startsWith('<p>')) {
                    motion.reason = '<p>' + motion.reason + '</p>';
                }
                // submitter
                if (motion.submitter) {
                    if (motion.submitter !== '') {
                        // All user objects are already loaded via the resolve statement from ui-router.
                        var users = User.getAll();
                        angular.forEach(users, function (user) {
                            if (user.short_name == motion.submitter.trim()) {
                                motion.submitters_id = [user.id];
                                motion.submitter = User.get(user.id).full_name;
                            }
                        });
                    }
                }
                if (motion.submitter && motion.submitter !== '' && !motion.submitters_id) {
                    motion.submitter_create = gettext('New participant will be created.');
                }
                // category
                if (motion.category) {
                    if (motion.category !== '') {
                        // All categore objects are already loaded via the resolve statement from ui-router.
                        var categories = Category.getAll();
                        angular.forEach(categories, function (category) {
                            // search for existing category
                            if (category.name == motion.category) {
                                motion.category_id = category.id;
                                motion.category = Category.get(category.id).name;
                            }
                        });
                    }
                }
                if (motion.category && motion.category !== '' && !motion.category_id) {
                    motion.category_create = gettext('New category will be created.');
                }

                $scope.motions.push(motion);
            });
            $scope.calcStats();
        };

        $scope.calcStats = function () {
            $scope.motionsWillNotBeImported = 0;
            $scope.motionsWillBeImported = 0;

            $scope.motions.forEach(function(motion) {
                if (!motion.importerror && motion.selected) {
                    $scope.motionsWillBeImported++;
                } else {
                    $scope.motionsWillNotBeImported++;
                }
            });
        };

        // Counter for creations
        $scope.usersCreated = 0;
        $scope.categoriesCreated = 0;

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;

            // Reset counters
            $scope.usersCreated = 0;
            $scope.categoriesCreated = 0;

            var importedUsers = [];
            var importedCategories = [];
            // collect users and categories
            angular.forEach($scope.motions, function (motion) {
                if (motion.selected && !motion.importerror) {
                    // collect user if not exists
                    if (!motion.submitters_id && motion.submitter) {
                        var index = motion.submitter.indexOf(' ');
                        var user = {
                            first_name: motion.submitter.substr(0, index),
                            last_name: motion.submitter.substr(index+1),
                            groups_id: []
                        };
                        importedUsers.push(user);
                    }
                    // collect category if not exists
                    if (!motion.category_id && motion.category) {
                        var category = {
                            name: motion.category,
                            prefix: motion.category.charAt(0)
                        };
                        importedCategories.push(category);
                    }
                }
            });

            // unique users and categories
            var importedUsersUnique = _.uniqWith(importedUsers, function (u1, u2) {
                return u1.first_name == u2.first_name &&
                    u1.last_name == u2.last_name;
            });
            var importedCategoriesUnique = _.uniqWith(importedCategories, function (c1, c2) {
                return c1.name == c2.name;
            });

            // Promises for users and categories
            var createPromises = [];

            // create users and categories
            importedUsersUnique.forEach(function (user) {
                createPromises.push(User.create(user).then(
                    function (success) {
                        user.id = success.id;
                        $scope.usersCreated++;
                    }
                ));
            });
            importedCategoriesUnique.forEach(function (category) {
                createPromises.push(Category.create(category).then(
                    function (success) {
                        category.id = success.id;
                        $scope.categoriesCreated++;
                    }
                ));
            });

            // wait for users and categories to create
            $q.all(createPromises).then( function() {
                angular.forEach($scope.motions, function (motion) {
                    if (motion.selected && !motion.importerror) {
                        // now, add user
                        if (!motion.submitters_id && motion.submitter) {
                            var index = motion.submitter.indexOf(' ');
                            var first_name = motion.submitter.substr(0, index);
                            var last_name = motion.submitter.substr(index+1);

                            // search for user, set id.
                            importedUsersUnique.forEach(function (user) {
                                if (user.first_name == first_name &&
                                    user.last_name == last_name) {
                                    motion.submitters_id = [user.id];
                                }
                            });
                        }
                        // add category
                        if (!motion.category_id && motion.category) {
                            var name = motion.category;

                            // search for category, set id.
                            importedCategoriesUnique.forEach(function (category) {
                                if (category.name == name) {
                                    motion.category_id = category.id;
                                }
                            });
                        }

                        // finally create motion
                        Motion.create(motion).then(
                            function(success) {
                                motion.imported = true;
                            }
                        );
                    }
                });
            });
            $scope.csvimported = true;
        };
        $scope.clear = function () {
            $scope.motions = [];
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            MotionCsvExport.downloadExample();
        };
    }
])


.controller('CategoryListCtrl', [
    '$scope',
    'Category',
    'ngDialog',
    'CategoryForm',
    function($scope, Category, ngDialog, CategoryForm) {
        Category.bindAll({}, $scope, 'categories');

        // setup table sorting
        $scope.sortColumn = 'name';
        $scope.reverse = false;
        // function to sort by clicked column
        $scope.toggleSort = function ( column ) {
            if ( $scope.sortColumn === column ) {
                $scope.reverse = !$scope.reverse;
            }
            $scope.sortColumn = column;
        };

        // delete selected category
        $scope.delete = function (category) {
            Category.destroy(category.id);
        };
        $scope.editOrCreate = function (category) {
            ngDialog.open(CategoryForm.getDialog(category));
        };
    }
])

.controller('CategoryCreateCtrl', [
    '$scope',
    'Category',
    'CategoryForm',
    'ErrorMessage',
    function($scope, Category, CategoryForm, ErrorMessage) {
        $scope.model = {};
        $scope.alert = {};
        $scope.formFields = CategoryForm.getFormFields();
        $scope.save = function (category) {
            Category.create(category).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('CategoryUpdateCtrl', [
    '$scope',
    'Category',
    'categoryId',
    'CategoryForm',
    'ErrorMessage',
    function($scope, Category, categoryId, CategoryForm, ErrorMessage) {
        $scope.alert = {};
        $scope.model = angular.copy(Category.get(categoryId));
        $scope.formFields = CategoryForm.getFormFields();
        $scope.save = function (category) {
            Category.inject(category);
            Category.save(category).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original category object from server
                    Category.refresh(category);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('CategorySortCtrl', [
    '$scope',
    '$stateParams',
    '$http',
    'MotionList',
    'Category',
    'categoryId',
    'Motion',
    'ErrorMessage',
    function($scope, $stateParams, $http, MotionList, Category, categoryId, Motion, ErrorMessage) {
        Category.bindOne(categoryId, $scope, 'category');
        Motion.bindAll({}, $scope, 'motions');
        $scope.filter = { category_id: categoryId,
                          parent_id: null,
                          orderBy: 'identifier' };

        $scope.$watch(
            function () {
                return Motion.lastModified();
            },
            function () {
                $scope.items = MotionList.getList(Motion.filter($scope.filter));
            }
        );

        $scope.alert = {};
        // Numbers all motions in this category by the given order in $scope.items
        $scope.numbering = function () {
            // Create a list of all motion ids in the current order.
            var sorted_motions = [];
            $scope.items.forEach(function (item) {
                sorted_motions.push(item.item.id);
            });

            // renumber them
            $http.post('/rest/motions/category/' + $scope.category.id + '/numbering/',
                {'motions': sorted_motions} ).then(
            function (success) {
                $scope.alert = { type: 'success', msg: success.data.detail, show: true };
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

//mark all motions config strings for translation in javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Motions');

        // subgroup General
        gettext('General');
        gettext('Workflow of new motions');
        gettext('Identifier');
        gettext('Numbered per category');
        gettext('Serially numbered');
        gettext('Set it manually');
        gettext('Motion preamble');
        gettext('The assembly may decide:');
        gettext('Default line numbering');
        /// Line numbering: Outside
        gettext('Outside');
        /// Line numbering: Inline
        gettext('Inline');
        /// Line numbering: None
        gettext('None');
        gettext('Line length');
        gettext('The maximum number of characters per line. Relevant when line numbering is enabled. Min: 40');
        gettext('Stop submitting new motions by non-staff users');
        gettext('Allow to disable versioning');
        gettext('Name of recommender');
        gettext('Default text version for change recommendations');
        gettext('Will be displayed as label before selected recommendation. Use an empty value to disable the recommendation system.');

        // subgroup Amendments
        gettext('Amendments');
        gettext('Activate amendments');
        gettext('Prefix for the identifier for amendments');
        gettext('Apply text for new amendments');
        gettext('The title of the motion is always applied.');

        // subgroup Suppoerters
        gettext('Supporters');
        gettext('Number of (minimum) required supporters for a motion');
        gettext('Choose 0 to disable the supporting system.');
        gettext('Remove all supporters of a motion if a submitter edits his ' +
                'motion in early state');

        // subgroup Supporters
        gettext('Comments');
        gettext('Comment fields for motions');
        gettext('Public');
        gettext('Private');

        // subgroup Voting and ballot papers
        gettext('Voting and ballot papers');
        gettext('The 100 % base of a voting result consists of');
        gettext('Yes/No/Abstain');
        gettext('Yes/No');
        gettext('All valid ballots');
        gettext('All casted ballots');
        gettext('Disabled (no percents)');
        gettext('Required majority');
        gettext('Default method to check whether a motion has reached the required majority.');
        gettext('Simple majority');
        gettext('Two-thirds majority');
        gettext('Three-quarters majority');
        gettext('Disabled');
        gettext('Number of ballot papers (selection)');
        gettext('Number of all delegates');
        gettext('Number of all participants');
        gettext('Use the following custom number');
        gettext('Custom number of ballot papers');

        // subgroup PDF
        gettext('Title for PDF and DOCX documents (all motions)');
        gettext('Preamble text for PDF and DOCX documents (all motions)');

        // misc strings (used dynamically in templates by translate filter)
        gettext('needed');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.poll.majority', [])

.value('MajorityMethodChoices', [
    {'value': 'simple_majority', 'display_name': 'Simple majority'},
    {'value': 'two-thirds_majority', 'display_name': 'Two-thirds majority'},
    {'value': 'three-quarters_majority', 'display_name': 'Three-quarters majority'},
    {'value': 'disabled', 'display_name': 'Disabled'},
])

.factory('MajorityMethods', [
    function () {
        return {
            'simple_majority': function (vote, base) {
                return Math.ceil(-(base / 2 - vote)) - 1;
            },
            'two-thirds_majority': function (vote, base) {
                var result = -(base * 2 - vote * 3) / 3;
                if (result % 1 !== 0) {
                    result = Math.ceil(result) - 1;
                }
                return result;
            },
            'three-quarters_majority': function (vote, base) {
                var result = -(base * 3 - vote * 4) / 4;
                if (result % 1 !== 0) {
                    result = Math.ceil(result) - 1;
                }
                return result;
            },
            'disabled': function () {
                return undefined;
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics', [])

.factory('Topic', [
    'DS',
    'jsDataModel',
    'gettext',
    function(DS, jsDataModel, gettext) {
        var name = 'topics/topic';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Topic'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
                getCSVExportText: function () {
                    return this.text;
                },
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'mediafiles/mediafile': {
                        localField: 'attachments',
                        localKeys: 'attachments_id',
                    }
                }
            }
        });
    }
])

.run(['Topic', function(Topic) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.csv', [])

.factory('TopicsCsvExample', [
    'gettextCatalog',
    'CsvDownload',
    function (gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Text', 'Duration', 'Comment', 'Internal item'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            downloadExample: function () {
                var csvRows = [makeHeaderline(),
                    // example entries
                    ['Demo 1', 'Demo text 1', '1:00', 'test comment', ''],
                    ['Break', '', '0:10', '', '1'],
                    ['Demo 2', 'Demo text 2', '1:30', '', '']

                ];
                CsvDownload(csvRows, 'agenda-example.csv');
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.projector', ['OpenSlidesApp.topics'])

.config([
    'slidesProvider',
    function (slidesProvider) {
        slidesProvider.registerSlide('topics/topic', {
            template: 'static/templates/topics/slide_topic.html'
        });
    }
])

.controller('SlideTopicCtrl', [
    '$scope',
    'Topic',
    function($scope, Topic) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        Topic.bindOne(id, $scope, 'topic');
    }
]);

})();

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.site', ['OpenSlidesApp.topics', 'OpenSlidesApp.topics.csv'])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('topics', {
                url: '/topics',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Topics'),
                },
            })
            .state('topics.topic', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('topics.topic.detail', {
                resolve: {
                    topicId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to topic detail and opens topic edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from topic controller use TopicForm factory instead to open dialog in front
            // of current view without redirect)
            .state('topics.topic.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/topics/topic-form.html',
                            controller: 'TopicUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                topicId: function() {
                                    return $stateParams.id;
                                },
                            },
                            preCloseCallback: function() {
                                $state.go('topics.topic.detail', {topic: $stateParams.id});
                                return true;
                            }
                        });
                    }],
            })
            .state('topics.topic.import', {
                url: '/import',
                controller: 'TopicImportCtrl',
            });
    }
])

.factory('TopicForm', [
    'gettextCatalog',
    'operator',
    'Editor',
    'Mediafile',
    'Agenda',
    'AgendaTree',
    function (gettextCatalog, operator, Editor, Mediafile, Agenda, AgendaTree) {
        return {
            // ngDialog for topic form
            getDialog: function (topic) {
                return {
                    template: 'static/templates/topics/topic-form.html',
                    controller: (topic) ? 'TopicUpdateCtrl' : 'TopicCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        topicId: function () {return topic ? topic.id: void 0;}
                    },
                };
            },
            getFormFields: function (isCreateForm) {
                var images = Mediafile.getAllImages();
                var formFields = [
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    key: 'text',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Text')
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    }
                }];
                // attachments
                if (Mediafile.getAll().length > 0) {
                    formFields.push({
                        key: 'attachments_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Attachment'),
                            options: Mediafile.getAll(),
                            ngOptions: 'option.id as option.title_or_filename for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search an attachment ...')
                        }
                    });
                }
                // show as agenda item
                formFields.push({
                    key: 'showAsAgendaItem',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Show as agenda item'),
                        description: gettextCatalog.getString('If deactivated it appears as internal item on agenda.')
                    },
                    hide: !operator.hasPerms('agenda.can_manage')
                });

                // parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'agenda_parent_item_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        }
                    });
                }

                return formFields;
            }
        };
    }
])

.controller('TopicDetailCtrl', [
    '$scope',
    'ngDialog',
    'TopicForm',
    'Topic',
    'topicId',
    'Projector',
    'ProjectionDefault',
    function($scope, ngDialog, TopicForm, Topic, topicId, Projector, ProjectionDefault) {
        Topic.bindOne(topicId, $scope, 'topic');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'topics'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.openDialog = function (topic) {
            ngDialog.open(TopicForm.getDialog(topic));
        };
    }
])

.controller('TopicCreateCtrl', [
    '$scope',
    '$state',
    'Topic',
    'TopicForm',
    'Agenda',
    'AgendaUpdate',
    'ErrorMessage',
    function($scope, $state, Topic, TopicForm, Agenda, AgendaUpdate, ErrorMessage) {
        $scope.topic = {};
        $scope.model = {};
        $scope.model.showAsAgendaItem = true;
        // get all form fields
        $scope.formFields = TopicForm.getFormFields(true);
        // save form
        $scope.save = function (topic) {
            Topic.create(topic).then(
                function (success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (topic.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: topic.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    $scope.closeThisDialog();
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TopicUpdateCtrl', [
    '$scope',
    '$state',
    'Topic',
    'TopicForm',
    'Agenda',
    'AgendaUpdate',
    'topicId',
    'ErrorMessage',
    function($scope, $state, Topic, TopicForm, Agenda, AgendaUpdate, topicId, ErrorMessage) {
        var topic = Topic.get(topicId);
        $scope.alert = {};
        // set initial values for form model by create deep copy of topic object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(topic);
        // get all form fields
        $scope.formFields = TopicForm.getFormFields();
        for (var i = 0; i < $scope.formFields.length; i++) {
            if ($scope.formFields[i].key == "showAsAgendaItem") {
                // get state from agenda item (hidden/internal or agenda item)
                $scope.formFields[i].defaultValue = !topic.agenda_item.is_hidden;
            } else if ($scope.formFields[i].key == "agenda_parent_item_id") {
                $scope.formFields[i].defaultValue = topic.agenda_item.parent_id;
            }
        }
        // save form
        $scope.save = function (topic) {
            Topic.create(topic).then(
                function(success) {
                    // type: Value 1 means a non hidden agenda item, value 2 means a hidden agenda item,
                    // see openslides.agenda.models.Item.ITEM_TYPE.
                    var changes = [{key: 'type', value: (topic.showAsAgendaItem ? 1 : 2)},
                                   {key: 'parent_id', value: topic.agenda_parent_item_id}];
                    AgendaUpdate.saveChanges(success.agenda_item_id,changes);
                    $scope.closeThisDialog();
                }, function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original topic object from server
                    Topic.refresh(topic);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TopicImportCtrl', [
    '$scope',
    'gettext',
    'Agenda',
    'Topic',
    'HumanTimeConverter',
    'TopicsCsvExample',
    'AgendaUpdate',
    function($scope, gettext, Agenda, Topic, HumanTimeConverter, TopicsCsvExample, AgendaUpdate) {
        // Big TODO: Change wording from "item" to "topic".
        // import from textarea
        $scope.importByLine = function () {
            if ($scope.itemlist) {
                $scope.titleItems = $scope.itemlist[0].split("\n");
                $scope.importcounter = 0;
                $scope.titleItems.forEach(function(title, index) {
                    var item = {title: title};
                    // TODO: create all items in bulk mode
                    Topic.create(item).then(
                        function(success) {
                            var changes = [{key: 'type', value: 1},
                                           {key: 'weight', value: 1000 + index}];
                            AgendaUpdate.saveChanges(success.agenda_item_id, changes);
                            $scope.importcounter++;
                        }
                    );
                });
            }
        };

        // *** CSV import ***
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };
        var FIELDS = ['title', 'text', 'duration', 'comment', 'is_hidden'];
        $scope.items = [];
        $scope.onCsvChange = function (csv) {
            $scope.items = [];

            var items = [];
            _.forEach(csv.data, function (row) {
                if (row.length > 1) {
                    var filledRow = _.zipObject(FIELDS, row);
                    items.push(filledRow);
                }
            });

            _.forEach(items, function (item, index) {
                item.selected = true;

                if (!item.title) {
                    item.importerror = true;
                    item.title_error = gettext('Error: Title is required.');
                }
                // duration
                if (item.duration) {
                    var time = HumanTimeConverter.humanTimeToSeconds(item.duration, {hours: true})/60;
                    if (time <= 0) { // null instead of 0 or negative duration
                        time = null;
                    }
                    item.duration = time;
                } else {
                    item.duration = null;
                }
                // is_hidden
                if (item.is_hidden) {
                    if (item.is_hidden == '1') {
                        item.type = 2;
                    } else {
                        item.type = 1;
                    }
                } else {
                    item.type = 1;
                }
                // set weight for right csv row order
                // (Use 1000+ to protect existing items and prevent collision
                // with new items which use weight 10000 as default.)
                item.weight = 1000 + index;
                $scope.items.push(item);
            });
            $scope.calcStats();
        };

        $scope.calcStats = function () {
            $scope.itemsWillNotBeImported = 0;
            $scope.itemsWillBeImported = 0;

            $scope.items.forEach(function(item) {
                if (item.selected && !item.importerror) {
                    $scope.itemsWillBeImported++;
                } else {
                    $scope.itemsWillNotBeImported++;
                }
            });
        };

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;
            angular.forEach($scope.items, function (item) {
                if (item.selected && !item.importerror) {
                    Topic.create(item).then(
                        function(success) {
                            var changes = [{key: 'duration', value: item.duration},
                                           {key: 'comment', value: item.comment},
                                           {key: 'type', value: item.type},
                                           {key: 'weight', value: item.weight}];
                            AgendaUpdate.saveChanges(success.agenda_item_id, changes);
                            item.imported = true;
                        }
                    );
                }
            });
            $scope.csvimported = true;
        };
        $scope.clear = function () {
            $scope.items = null;
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            TopicsCsvExample.downloadExample();
        };
     }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users', [])

.factory('User', [
    'DS',
    'Group',
    'jsDataModel',
    'gettext',
    'gettextCatalog',
    'Config',
    function(DS, Group, jsDataModel, gettext, gettextCatalog, Config) {
        var name = 'users/user';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Participants'),
            verboseNamePlural: gettext('Participants'),
            computed: {
                full_name: function () {
                    return this.get_full_name();
                },
                short_name: function () {
                    return this.get_short_name();
                },
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                /*
                 * Returns a short form of the name.
                 *
                 * Example:
                 * - Dr. Max Mustermann
                 * - Professor Dr. Enders, Christoph
                 */
                get_short_name: function() {
                    var title = _.trim(this.title),
                        firstName = _.trim(this.first_name),
                        lastName = _.trim(this.last_name),
                        name = '';
                    if (Config.get('users_sort_by') && Config.get('users_sort_by').value == 'last_name') {
                        if (lastName && firstName) {
                            name += [lastName, firstName].join(', ');
                        } else {
                            name += lastName || firstName;
                        }
                    } else {
                        name += [firstName, lastName].join(' ');
                    }
                    if (name.trim() === '') {
                        name = this.username;
                    }
                    if (title !== '') {
                        name = title + ' ' + name;
                    }
                    return name.trim();
                },
                /*
                 * Returns a long form of the name.
                 *
                 * Example:
                 * - Dr. Max Mustermann (Villingen)
                 * - Professor Dr. Enders, Christoph (Leipzig)
                 */
                get_full_name: function() {
                    var name = this.get_short_name(),
                        structure_level = _.trim(this.structure_level),
                        number = _.trim(this.number),
                        addition = [];

                    // addition: add number and structure level
                    if (structure_level) {
                        addition.push(structure_level);
                    }
                    if (number) {
                        addition.push(
                            /// abbreviation for number
                            gettextCatalog.getString('No.') + ' ' + number
                        );
                    }
                    if (addition.length > 0) {
                        name += ' (' + addition.join(' · ') + ')';
                    }
                    return name.trim();
                },
                getPerms: function() {
                    var allPerms = [];
                    var allGroups = [];
                    if (this.groups_id) {
                        allGroups = this.groups_id.slice(0);
                    }
                    if (allGroups.length === 0) {
                        allGroups.push(1); // add default group
                    }
                    _.forEach(allGroups, function(groupId) {
                        var group = Group.get(groupId);
                        if (group) {
                            _.forEach(group.permissions, function(perm) {
                                allPerms.push(perm);
                            });
                        }
                    });
                    return _.uniq(allPerms);
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.get_full_name();
                },
                // subtitle of search result
                getSearchResultSubtitle: function () {
                    return "Participant";
                },
            },
            relations: {
                hasMany: {
                    'users/group': {
                        localField: 'groups',
                        localKey: 'groups_id',
                    }
                }
            }
        });
    }
])

.factory('Group', [
    '$http',
    'DS',
    function($http, DS) {
        var permissions;
        return DS.defineResource({
            name: 'users/group',
            permissions: permissions,
            // TODO (Issue 2862): Do not query the permissions from server. They should be included
            // in the startup data. Then remove 'permission' injection from group list controller.
            getPermissions: function() {
                if (!this.permissions) {
                    this.permissions = $http({ 'method': 'OPTIONS', 'url': '/rest/users/group/' })
                        .then(function(result) {
                            return result.data.actions.POST.permissions.choices;
                        });
                }
                return this.permissions;
            }
        });
    }
])

.run([
    'User',
    'Group',
    function(User, Group) {}
])


// Mark strings for translation in JavaScript.
.config([
    'gettext',
    function (gettext) {
        // default group names (from users/signals.py)
        gettext('Default');
        gettext('Delegates');
        gettext('Staff');
        gettext('Committees');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.csv', [])

.factory('UserCsvExport', [
    '$filter',
    'Group',
    'gettextCatalog',
    'CsvDownload',
    function ($filter, Group, gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Given name', 'Surname', 'Structure level', 'Participant number', 'Groups',
                'Comment', 'Is active', 'Is present', 'Is a committee', 'Initial password'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (users) {
                var csvRows = [
                    makeHeaderline()
                ];
                _.forEach(users, function (user) {
                    var groups = _.map(user.groups_id, function (id) {
                        return gettextCatalog.getString(Group.get(id).name);
                    }).join(',');
                    var row = [];
                    row.push('"' + user.title + '"');
                    row.push('"' + user.first_name + '"');
                    row.push('"' + user.last_name + '"');
                    row.push('"' + user.structure_level + '"');
                    row.push('"' + user.number + '"');
                    row.push('"' + groups + '"');
                    row.push('"' + user.comment + '"');
                    row.push(user.is_active ? '1' : '0');
                    row.push(user.is_present ? '1' : '0');
                    row.push(user.is_committee ? '1' : '0');
                    row.push('"' + user.default_password + '"');
                    csvRows.push(row);
                });
                CsvDownload(csvRows, 'users-export.csv');
            },

            downloadExample: function () {
                // try to get an example with two groups and one with one group
                var groups = $filter('orderBy')(Group.getAll(), 'id');
                var csvGroups = '';
                var csvGroup = '';
                if (groups.length >= 3) { // do not pick groups[0], this is the default group
                    csvGroups = '"' + gettextCatalog.getString(groups[1].name) +
                                ', ' + gettextCatalog.getString(groups[2].name) + '"';
                }
                if (groups.length >= 2) {
                    csvGroup = gettextCatalog.getString(groups[groups.length - 1].name); // take last group
                }

                var csvRows = [makeHeaderline(),
                    // example entries
                    ['Dr.', 'Max', 'Mustermann', 'Berlin','1234567890', csvGroups, 'xyz', '1', '1', '', ''],
                    ['', 'John', 'Doe', 'Washington','75/99/8-2', csvGroup, 'abc', '1', '1', '', ''],
                    ['', 'Fred', 'Bloggs', 'London', '', '', '', '', '', '', ''],
                    ['', '', 'Executive Board', '', '', '', '', '', '', '1', ''],

                ];
                CsvDownload(csvRows, 'users-example.csv');
            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.pdf', ['OpenSlidesApp.core.pdf'])

.factory('UserListContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    function(gettextCatalog, PDFLayout) {

    var createInstance = function(userList, groups) {

        //use the Predefined Functions to create the title
        var title = PDFLayout.createTitle(gettextCatalog.getString("List of participants"));

        //function to generate the user list
        var createUserList = function() {
            var userJsonList = [];

            angular.forEach(userList, function (user, counter) {

                //parse for the group names
                var userGroups = [];
                angular.forEach(user.groups_id, function (id) {
                    if (id) {
                        angular.forEach(groups, function(group) {
                            if (id == group.id) {
                                userGroups.push(gettextCatalog.getString(group.name));
                            }
                        });
                    }
                });

                var userJsonObj = [
                    {
                        text: "" + (counter+1),
                        style: PDFLayout.flipTableRowStyle(userJsonList.length)
                    },
                    {
                        text: user.short_name,
                        style: PDFLayout.flipTableRowStyle(userJsonList.length)
                    },
                    {
                        text: user.structure_level,
                        style: PDFLayout.flipTableRowStyle(userJsonList.length)
                    },
                    {
                        text: userGroups.join(" "),
                        style: PDFLayout.flipTableRowStyle(userJsonList.length)
                    }
                ];
                userJsonList.push(userJsonObj);
            });

            var userTableBody = [
                [
                    {
                        text: '#',
                        style: 'tableHeader'
                    },
                    {
                        text: gettextCatalog.getString("Name"),
                        style: 'tableHeader'
                    },
                    {
                        text: gettextCatalog.getString("Structure level"),
                        style: 'tableHeader'
                    },
                    {
                        text: gettextCatalog.getString("Groups"),
                        style: 'tableHeader'
                    }
                ]
            ];
            userTableBody = userTableBody.concat((userJsonList));

            var userTableJsonString = {
                table: {
                    widths: ['auto', '*', 'auto', 'auto'],
                    headerRows: 1,
                    body: userTableBody
                },
                layout: 'headerLineOnly'
            };

            return userTableJsonString;
        };

        var getContent = function() {
            return [
                title,
                createUserList()
            ];
        };

        return {
            getContent: getContent
        };
    };

    return {
        createInstance: createInstance
    };

}])

.factory('UserAccessDataListContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    function(gettextCatalog, PDFLayout) {

        var createInstance = function(userList, groups, Config) {

            var creadeUserHeadLine = function(user) {
                var titleLine = [];
                titleLine.push({
                    text: user.get_short_name(),
                    style: 'userDataTitle'
                });
                if (user.structure_level) {
                    titleLine.push({
                        text: user.structure_level,
                        style: 'userDataHeading'
                    });
                }
                return titleLine;
            };

            var createAccessDataContent = function(user) {
                // wlan access data
                var columnWifi = [
                    {
                        text: gettextCatalog.getString("WLAN access data"),
                        style: 'userDataHeading'
                    },
                    {
                        text: gettextCatalog.getString("WLAN name (SSID)") + ":",
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_ssid').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString("WLAN password") + ":",
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_password').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString("WLAN encryption") + ":",
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_encryption').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: "\n"
                    }
                ];
                // wifi qr code
                if (Config.get('users_pdf_wlan_ssid').value && Config.get('users_pdf_wlan_encryption').value) {
                    var wifiQrCode = "WIFI:S:" + Config.get('users_pdf_wlan_ssid').value +
                        ";T:" + Config.get('users_pdf_wlan_encryption').value +
                        ";P:" + Config.get('users_pdf_wlan_password').value + ";;";
                    columnWifi.push(
                        {
                            qr: wifiQrCode,
                            fit: 120,
                            margin: [0, 0, 0, 8]
                        },
                        {
                            text: gettextCatalog.getString("Scan this QR code to connect to WLAN."),
                            style: 'small'
                        }
                    );
                }

                // openslides access data
                var columnOpenSlides = [
                    {
                        text: gettextCatalog.getString("OpenSlides access data"),
                        style: 'userDataHeading'
                    },
                    {
                        text: gettextCatalog.getString("Username") + ":",
                        style: 'userDataTopic'
                    },
                    {
                        text: user.username,
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString("Initial password") + ":",
                        style: 'userDataTopic'
                    },
                    {
                        text: user.default_password,
                        style: 'userDataValue'
                    },
                    {
                        text: "URL:",
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_url').value  || '-',
                        link: Config.get('users_pdf_url').value,
                        style: 'userDataValue'
                    },
                    {
                        text: "\n"
                    }
                ];
                // url qr code
                if (Config.get('users_pdf_url').value) {
                    columnOpenSlides.push(
                        {
                            qr: Config.get('users_pdf_url').value,
                            fit: 120,
                            margin: [0, 0, 0, 8]
                        },
                        {
                            text: gettextCatalog.getString("Scan this QR code to open URL."),
                            style: 'small'
                        }
                    );
                }

                var accessDataColumns = {
                    columns: [
                        columnWifi,
                        columnOpenSlides,
                    ],
                    margin: [0, 20]
                };

                return accessDataColumns;
            };

            var createWelcomeText = function() {
                return [
                    {
                        text:   Config.translate(Config.get('users_pdf_welcometitle').value),
                        style: 'userDataHeading'
                    },
                    {
                        text:   Config.translate(Config.get('users_pdf_welcometext').value),
                        style: 'userDataTopic'
                    }
                ];
            };

            var getContent = function() {
                var content = [];
                angular.forEach(userList, function (user) {
                    content.push(creadeUserHeadLine(user));
                    content.push(createAccessDataContent(user));
                    content.push(createWelcomeText());
                    content.push({
                        text: '',
                        pageBreak: 'after'
                    });
                });

                return [
                    content
                ];
            };

            return {
                getContent: getContent
            };
        };

        return {
            createInstance: createInstance
        };
    }
]);
}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.projector', ['OpenSlidesApp.users'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('users/user', {
            template: 'static/templates/users/slide_user.html',
        });
    }
])

.controller('SlideUserCtrl', [
    '$scope',
    'User',
    function($scope, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        User.bindOne(id, $scope, 'user');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.site', [
    'OpenSlidesApp.users',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.users.pdf',
    'OpenSlidesApp.users.csv',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'users.user.list',
            'img_class': 'user',
            'title': gettext('Participants'),
            'weight': 500,
            'perm': 'users.can_see_name',
        });
    }
])
.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Participants'),
            'collectionName': 'users/user',
            'urlDetailState': 'users.user.detail',
            'weight': 500,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
        .state('users', {
            url: '/users',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Participants'),
                basePerm: 'users.can_see_name',
            },
        })
        .state('users.user', {
            abstract: true,
            template: "<ui-view/>",
        })
        .state('users.user.list', {})
        .state('users.user.create', {})
        .state('users.user.detail', {
            resolve: {
                userId: ['$stateParams', function($stateParams) {
                    return $stateParams.id;
                }]
            }
        })
        .state('users.user.change-password', {
            url: '/change-password/{id}',
            controller: 'UserChangePasswordCtrl',
            templateUrl: 'static/templates/users/user-change-password.html',
            resolve: {
                userId: ['$stateParams', function($stateParams) {
                    return $stateParams.id;
                }]
            }
        })
        .state('users.user.import', {
            url: '/import',
            controller: 'UserImportCtrl',
        })
        // groups
        .state('users.group', {
            url: '/groups',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Groups'),
            },
        })
        .state('users.group.list', {
            resolve: {
                permissions: function(Group) {
                    return Group.getPermissions();
                }
            }
        })
        .state('login', {
            template: null,
            url: '/login',
            params: {
                guest_enabled: false,
                msg: null,
            },
            onEnter: ['$state', '$stateParams', 'ngDialog', function($state, $stateParams, ngDialog) {
                ngDialog.open({
                    template: 'static/templates/core/login-form.html',
                    controller: 'LoginFormCtrl',
                    showClose: $stateParams.guest_enabled,
                    closeByEscape: $stateParams.guest_enabled,
                    closeByDocument: $stateParams.guest_enabled,
                });
            }],
            data: {
                title: 'Login',
            },
        });
    }
])

/*
 * Directive to check for permissions
 *
 * This is the Code from angular.js ngIf.
 *
 * TODO: find a way not to copy the code.
*/
.directive('osPerms', [
    '$animate',
    function($animate) {
        return {
            multiElement: true,
            transclude: 'element',
            priority: 600,
            terminal: true,
            restrict: 'A',
            $$tlb: true,
            link: function($scope, $element, $attr, ctrl, $transclude) {
                var block, childScope, previousElements, perms;
                if ($attr.osPerms[0] === '!') {
                    perms = _.trimStart($attr.osPerms, '!');
                } else {
                    perms = $attr.osPerms;
                }
                $scope.$watch(
                    function (scope) {
                        return scope.operator && scope.operator.hasPerms(perms);
                    },
                    function (value) {
                        if ($attr.osPerms[0] === '!') {
                            value = !value;
                        }
                        if (value) {
                            if (!childScope) {
                                $transclude(function(clone, newScope) {
                                    childScope = newScope;
                                    clone[clone.length++] = document.createComment(' end ngIf: ' + $attr.ngIf + ' ');
                                    // Note: We only need the first/last node of the cloned nodes.
                                    // However, we need to keep the reference to the jqlite wrapper as it might be changed later
                                    // by a directive with templateUrl when its template arrives.
                                    block = {
                                        clone: clone
                                    };
                                    $animate.enter(clone, $element.parent(), $element);
                                });
                            }
                        } else {
                            if (previousElements) {
                                previousElements.remove();
                                previousElements = null;
                            }
                            if (childScope) {
                                childScope.$destroy();
                                childScope = null;
                            }
                            if (block) {
                                previousElements = getBlockNodes(block.clone);
                                $animate.leave(previousElements).then(function() {
                                    previousElements = null;
                                });
                                block = null;
                            }
                        }
                    }
                );
            }
        };
    }
])

.factory('PasswordGenerator', [
    function () {
        return {
            generate: function (length) {
                if (!length) {
                    length = 8;
                }
                var chars = 'abcdefghijkmnpqrstuvwxyzABCDEFGHJKLMNPQRSTUVWXYZ23456789',
                pw = '';
                for (var i = 0; i < length; ++i) {
                    pw += chars.charAt(Math.floor(Math.random() * chars.length));
                }
                return pw;
            }
        };
    }
])

// Service for generic assignment form (create and update)
.factory('UserForm', [
    '$http',
    'gettextCatalog',
    'Editor',
    'Group',
    'Mediafile',
    'PasswordGenerator',
    function ($http, gettextCatalog, Editor, Group, Mediafile, PasswordGenerator) {
        return {
            // ngDialog for user form
            getDialog: function (user) {
                return {
                    template: 'static/templates/users/user-form.html',
                    controller: (user) ? 'UserUpdateCtrl' : 'UserCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        userId: function () {return user ? user.id : void 0;},
                    }
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                var images = Mediafile.getAllImages();
                return [
                {
                    className: "row",
                    fieldGroup: [
                        {
                            key: 'title',
                            type: 'input',
                            className: "col-xs-2 no-padding-left",
                            templateOptions: {
                                label: gettextCatalog.getString('Title')
                            }
                        },
                        {
                            key: 'first_name',
                            type: 'input',
                            className: "col-xs-5 no-padding",
                            templateOptions: {
                                label: gettextCatalog.getString('Given name')
                            }
                        },
                        {
                            key: 'last_name',
                            type: 'input',
                            className: "col-xs-5 no-padding-right",
                            templateOptions: {
                                label: gettextCatalog.getString('Surname')
                            }
                        }
                    ]
                },
                {
                    className: "row",
                    fieldGroup: [
                        {
                            key: 'structure_level',
                            type: 'input',
                            className: "col-xs-9 no-padding-left",
                            templateOptions: {
                                label: gettextCatalog.getString('Structure level'),
                            }
                        },
                        {   key: 'number',
                            type: 'input',
                            className: "col-xs-3 no-padding-left no-padding-right",
                            templateOptions: {
                                label:gettextCatalog.getString('Participant number')
                            }
                        }
                    ]
                },
                {
                    key: 'username',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Username')
                    },
                    hide: hideOnCreateForm
                },
                {
                    key: 'groups_id',
                    type: 'select-multiple',
                    templateOptions: {
                        label: gettextCatalog.getString('Groups'),
                        options: Group.getAll(),
                        ngOptions: 'option.id as option.name | translate for option in to.options | ' +
                                   'filter: {id: "!1"}',
                        placeholder: gettextCatalog.getString('Select or search a group ...')
                    }
                },
                {
                    key: 'default_password',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Initial password'),
                        description: gettextCatalog.getString('Initial password can not be changed.'),
                        addonRight: {
                            text: gettextCatalog.getString('Generate'),
                            class: 'fa fa-magic',
                            onClick:function (options, scope) {
                                scope.$parent.model.default_password = PasswordGenerator.generate();
                            }
                        }
                    },
                    hide: !hideOnCreateForm
                },
                {
                    key: 'comment',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Comment'),
                        description: gettextCatalog.getString('Only for internal notes.')
                    }
                },
                {
                    key: 'more',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Show extended fields')
                    }
                },
                {
                    template: '<hr class="smallhr">',
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_present',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is present'),
                        description: gettextCatalog.getString('Designates whether this user is in the room or not.')
                    },
                    defaultValue: true,
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_active',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is active'),
                        description: gettextCatalog.getString(
                            'Designates whether this user should be treated as ' +
                            'active. Unselect this instead of deleting the account.')
                    },
                    defaultValue: true,
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_committee',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is a committee'),
                        description: gettextCatalog.getString(
                            'Designates whether this user should be treated as a committee.')
                    },
                    defaultValue: false,
                    hideExpression: '!model.more'
                },
                {
                    key: 'about_me',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('About me'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    },
                    hideExpression: '!model.more'
                }
                ];
            }
        };
    }
])

.factory('UserProfileForm', [
    'gettextCatalog',
    'Editor',
    'Mediafile',
    function (gettextCatalog, Editor, Mediafile) {
        return {
            // ngDialog for user form
            getDialog: function (user) {
                return {
                    template: 'static/templates/users/profile-password-form.html',
                    controller: 'UserProfileCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                var images = Mediafile.getAllImages();
                return [
                {
                    key: 'username',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Username'),
                        required: true
                    },
                },
                {
                    key: 'about_me',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('About me'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    },
                }
                ];
            }
        };
    }
])

.factory('UserPasswordForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            // ngDialog for user form
            getDialog: function (user) {
                return {
                    template: 'static/templates/users/profile-password-form.html',
                    controller: 'UserPasswordCtrl',
                    className: 'ngdialog-theme-default',
                    closeByEscape: false,
                    closeByDocument: false,
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                return [
                {
                    key: 'oldPassword',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('Old password'),
                        required: true
                    },
                },
                {
                    key: 'newPassword',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('New password'),
                        required: true
                    },
                },
                {
                    key: 'newPassword2',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('Confirm new password'),
                        required: true
                    },
                },
                ];
            }
        };
    }
])

.controller('UserListCtrl', [
    '$scope',
    '$state',
    '$http',
    'ngDialog',
    'UserForm',
    'User',
    'Group',
    'PasswordGenerator',
    'Projector',
    'ProjectionDefault',
    'UserListContentProvider',
    'Config',
    'UserAccessDataListContentProvider',
    'PdfMakeDocumentProvider',
    'gettextCatalog',
    'UserCsvExport',
    'osTableFilter',
    'osTableSort',
    'gettext',
    'PdfCreate',
    function($scope, $state, $http, ngDialog, UserForm, User, Group, PasswordGenerator, Projector, ProjectionDefault,
        UserListContentProvider, Config, UserAccessDataListContentProvider, PdfMakeDocumentProvider, gettextCatalog,
        UserCsvExport, osTableFilter, osTableSort, gettext, PdfCreate) {
        User.bindAll({}, $scope, 'users');
        Group.bindAll({where: {id: {'>': 1}}}, $scope, 'groups');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'users'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.alert = {};

        // Filtering
        $scope.filter = osTableFilter.createInstance('UserTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                group: [],
            };
            $scope.filter.booleanFilters = {
                isPresent: {
                    value: undefined,
                    displayName: gettext('Present'),
                    choiceYes: gettext('Is present'),
                    choiceNo: gettext('Is not present'),
                    needExtraPermission: true,
                },
                isActive: {
                    value: undefined,
                    displayName: gettext('Active'),
                    choiceYes: gettext('Is active'),
                    choiceNo: gettext('Is not active'),
                    needExtraPermission: true,
                },
                isCommittee: {
                    value: undefined,
                    displayName: gettext('Committee'),
                    choiceYes: gettext('Is a committee'),
                    choiceNo: gettext('Is not a committee'),
                },

            };
        }
        $scope.filter.propertyList = ['first_name', 'last_name', 'title', 'number', 'comment', 'structure_level'];
        $scope.filter.propertyDict = {
            'groups_id' : function (group_id) {
                return Group.get(group_id).name;
            },
        };
        $scope.getItemId = {
            group: function (user) {return user.groups_id;},
        };
        // Sorting
        $scope.sort = osTableSort.createInstance();
        $scope.sort.column = $scope.config('users_sort_by');
        $scope.sortOptions = [
            {name: 'first_name',
             display_name: gettext('Given name')},
            {name: 'last_name',
             display_name: gettext('Surname')},
            {name: 'is_present',
             display_name: gettext('Present')},
            {name: 'is_active',
             display_name: gettext('Active')},
            {name: 'is_committee',
             display_name: gettext('Committee')},
            {name: 'number',
             display_name: gettext('Number')},
            {name: 'structure_level',
             display_name: gettext('Structure level')},
            {name: 'comment',
             display_name: gettext('Comment')},
        ];

        // pagination
        $scope.currentPage = 1;
        $scope.itemsPerPage = 25;
        $scope.limitBegin = 0;
        $scope.pageChanged = function() {
            $scope.limitBegin = ($scope.currentPage - 1) * $scope.itemsPerPage;
        };

        // Toggle group from user
        $scope.toggleGroup = function (user, group) {
            if (_.indexOf(user.groups_id, group.id) > -1) {
                user.groups_id = _.filter(user.groups_id, function (group_id) {
                    return group_id != group.id;
                });
            } else {
                user.groups_id.push(group.id);
            }
            $scope.save(user);
        };
        // open new/edit dialog
        $scope.openDialog = function (user) {
            ngDialog.open(UserForm.getDialog(user));
        };
        // save changed user
        $scope.save = function (user) {
            User.save(user).then(
                function(success) {
                    //user.quickEdit = false;
                    $scope.alert.show = false;
                },
                function(error){
                    $scope.alert = ErrorMessage.forAlert(error);
                });
        };
        // delete single user
        $scope.delete = function (user) {
            User.destroy(user.id);
        };
        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            _.forEach($scope.usersFiltered, function (user) {
                user.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.users, function (user) {
                    user.selected = false;
                });
            }
        };
        var selectModeAction = function (predicate) {
            angular.forEach($scope.usersFiltered, function (user) {
                if (user.selected) {
                    predicate(user);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete all selected users
        $scope.deleteMultiple = function () {
            selectModeAction(function (user) {
                $scope.delete(user);
            });
        };
        // add group for selected users
        $scope.addGroupMultiple = function (group) {
            if (group) {
                selectModeAction(function (user) {
                    user.groups_id.push(group);
                    User.save(user);
                });
            }
        };
        // remove group for selected users
        $scope.removeGroupMultiple = function (group) {
            if (group) {
                selectModeAction(function (user) {
                    var groupIndex = _.indexOf(user.groups_id, parseInt(group));
                    if (groupIndex > -1) {
                        user.groups_id.splice(groupIndex, 1);
                        User.save(user);
                    }
                });
            }
        };
        // generate new passwords
        $scope.generateNewPasswordsMultiple = function () {
            selectModeAction(function (user) {
                var newPassword = PasswordGenerator.generate();
                user.default_password = newPassword;
                User.save(user);
                $http.post(
                    '/rest/users/user/' + user.id + '/reset_password/',
                    {'password': newPassword}
                );
            });
        };
        // set boolean properties (is_active, is_present, is_committee)
        $scope.setBoolPropertyMultiple = function (property, value) {
            selectModeAction(function (user) {
                user[property] = value;
                User.save(user);
            });
        };

        // Export as PDF
        $scope.pdfExportUserList = function () {
            var filename = gettextCatalog.getString("List of participants")+".pdf";
            var userListContentProvider = UserListContentProvider.createInstance($scope.usersFiltered, $scope.groups);
            var documentProvider = PdfMakeDocumentProvider.createInstance(userListContentProvider);
            PdfCreate.download(documentProvider.getDocument(), filename);
        };
        $scope.pdfExportUserAccessDataList = function () {
            var filename = gettextCatalog.getString("List of access data")+".pdf";
            var userAccessDataListContentProvider = UserAccessDataListContentProvider.createInstance(
                $scope.usersFiltered, $scope.groups, Config);
            var documentProvider = PdfMakeDocumentProvider.createInstance(userAccessDataListContentProvider);
            var noFooter = true;
            PdfCreate.download(documentProvider.getDocument(noFooter), filename);
        };
        // Export as a csv file
        $scope.csvExport = function () {
            UserCsvExport.export($scope.usersFiltered);
        };
    }
])

.controller('UserDetailCtrl', [
    '$scope',
    'ngDialog',
    'UserForm',
    'User',
    'userId',
    'Group',
    'Projector',
    'ProjectionDefault',
    function($scope, ngDialog, UserForm, User, userId, Group, Projector, ProjectionDefault) {
        User.bindOne(userId, $scope, 'user');
        Group.bindAll({where: {id: {'>': 1}}}, $scope, 'groups');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'users'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });

        // open edit dialog
        $scope.openDialog = function (user) {
            ngDialog.open(UserForm.getDialog(user));
        };
    }
])

.controller('UserCreateCtrl', [
    '$scope',
    '$state',
    'User',
    'UserForm',
    'Group',
    'ErrorMessage',
    function($scope, $state, User, UserForm, Group, ErrorMessage) {
        Group.bindAll({where: {id: {'>': 2}}}, $scope, 'groups');
        $scope.alert = {};
        // get all form fields
        $scope.formFields = UserForm.getFormFields(true);

        // save user
        $scope.save = function (user) {
            if (!user.groups_id) {
                user.groups_id = [];
            }
            User.create(user).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserUpdateCtrl', [
    '$scope',
    '$state',
    '$http',
    'User',
    'UserForm',
    'Group',
    'userId',
    'ErrorMessage',
    function($scope, $state, $http, User, UserForm, Group, userId, ErrorMessage) {
        Group.bindAll({where: {id: {'>': 2}}}, $scope, 'groups');
        $scope.alert = {};
        // set initial values for form model by create deep copy of user object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(User.get(userId));

        // get all form fields
        $scope.formFields = UserForm.getFormFields();

        // save user
        $scope.save = function (user) {
            if (!user.groups_id) {
                user.groups_id = [];
            }
            // inject the changed user (copy) object back into DS store
            User.inject(user);
            // save change user object on server
            User.save(user).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original user object from server
                    User.refresh(user);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserProfileCtrl', [
    '$scope',
    'Editor',
    'User',
    'operator',
    'UserProfileForm',
    'gettext',
    'ErrorMessage',
    function($scope, Editor, User, operator, UserProfileForm, gettext, ErrorMessage) {
        $scope.model = angular.copy(operator.user);
        $scope.title = gettext('Edit profile');
        $scope.formFields = UserProfileForm.getFormFields();
        $scope.save = function (user) {
            User.inject(user);
            User.save(user).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function(error) {
                    // save error: revert all changes by restore
                    // (refresh) original user object from server
                    User.refresh(user);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserChangePasswordCtrl', [
    '$scope',
    '$state',
    '$http',
    'User',
    'userId',
    'gettextCatalog',
    'PasswordGenerator',
    'ErrorMessage',
    function($scope, $state, $http, User, userId, gettextCatalog, PasswordGenerator, ErrorMessage) {
        User.bindOne(userId, $scope, 'user');
        $scope.alert = {};
        $scope.generatePassword = function () {
            $scope.new_password = PasswordGenerator.generate();
        };
        $scope.save = function (user) {
            if ($scope.new_password !== '') {
                $http.post(
                    '/rest/users/user/' + user.id + '/reset_password/',
                    {'password': $scope.new_password}
                ).then(
                    function (success) {
                        $scope.alert = {type: 'success', msg: success.data.detail, show: true};
                        $scope.new_password = '';
                    },
                    function (error) {
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.controller('UserPasswordCtrl', [
    '$scope',
    '$state',
    '$http',
    'gettext',
    'UserPasswordForm',
    'ErrorMessage',
    function($scope, $state, $http, gettext, UserPasswordForm, ErrorMessage) {
        $scope.title = 'Change password';
        $scope.alert = {};
        $scope.model = {};
        $scope.formFields = UserPasswordForm.getFormFields();
        $scope.save = function (data) {
            if (data.newPassword != data.newPassword2) {
                data.newPassword = data.newPassword2 = '';
                $scope.alert = {
                    type: 'danger',
                    msg: gettext('Password confirmation does not match.'),
                    show: true,
                };
            } else {
                $http.post(
                    '/users/setpassword/',
                    {'old_password': data.oldPassword, 'new_password': data.newPassword}
                ).then(
                    function (success) {
                        $scope.closeThisDialog();
                    },
                    function (error) {
                        // Error, e. g. wrong old password.
                        $scope.model = {};
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.controller('UserImportCtrl', [
    '$scope',
    '$q',
    'gettext',
    'gettextCatalog',
    'User',
    'Group',
    'UserCsvExport',
    function($scope, $q, gettext, gettextCatalog, User, Group, UserCsvExport) {
        // import from textarea
        $scope.importByLine = function () {
            $scope.usernames = $scope.userlist[0].split("\n");
            $scope.importcounter = 0;
            $scope.usernames.forEach(function(name) {
                // Split each full name in first and last name.
                // The last word is set as last name, rest is the first name(s).
                // (e.g.: "Max Martin Mustermann" -> last_name = "Mustermann")
                var names = name.split(" ");
                var last_name = names.slice(-1)[0];
                var first_name = names.slice(0, -1).join(" ");
                var user = {
                    first_name: first_name,
                    last_name: last_name,
                    groups_id: []
                };
                User.create(user).then(
                    function(success) {
                        $scope.importcounter++;
                    }
                );
            });
        };

        // pagination
        $scope.currentPage = 1;
        $scope.itemsPerPage = 100;
        $scope.limitBegin = 0;
        $scope.pageChanged = function() {
            $scope.limitBegin = ($scope.currentPage - 1) * $scope.itemsPerPage;
        };
        $scope.duplicateActions = [
            gettext('keep original'),
            gettext('override new'),
            gettext('create duplicate')
        ];

        // *** csv import ***
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };

        var FIELDS = ['title', 'first_name', 'last_name', 'structure_level', 'number',
        'groups', 'comment', 'is_active', 'is_present', 'is_committee', 'default_password'];
        $scope.users = [];
        $scope.onCsvChange = function (csv) {
            // All user objects are already loaded via the resolve statement from ui-router.
            var users = User.getAll();
            $scope.users = [];

            var csvUsers = [];
            _.forEach(csv.data, function (row) {
                if (row.length >= 2) {
                    var filledRow = _.zipObject(FIELDS, row);
                    csvUsers.push(filledRow);
                }
            });
            $scope.duplicates = 0;
            _.forEach(csvUsers, function (user) {
                user.selected = true;
                if (!user.first_name && !user.last_name) {
                    user.importerror = true;
                    user.name_error = gettext('Error: Given name or surname is required.');
                }
                // number
                if (!user.number) {
                    user.number = "";
                }
                // groups
                user.groups_id = []; // will be overwritten if there are groups
                if (user.groups) {
                    user.groups = user.groups.split(',');
                    user.groups = _.map(user.groups, function (group) {
                        return _.trim(group); // remove whitespaces on start or end
                    });

                    // All group objects are already loaded via the resolve statement from ui-router.
                    var allGroups = Group.getAll();
                    // in allGroupsNames ar all original group names and translated names if a
                    // translation exists (e.g. for default group Delegates)
                    var allGroupsNames = [];
                    _.forEach(allGroups, function (group) {
                        var groupTranslation = gettextCatalog.getString(group.name);
                        if (group.name !== groupTranslation) {
                            allGroupsNames.push(groupTranslation);
                        }
                        allGroupsNames.push(group.name);
                    });
                    user.groupsToCreate = _.difference(user.groups, allGroupsNames);

                    // for template:
                    user.groupsNotToCreate = _.difference(user.groups, user.groupsToCreate);
                }
                user.is_active = (user.is_active !== undefined && user.is_active === '1');
                user.is_present = (user.is_present !== undefined && user.is_present === '1');
                user.is_committee = (user.is_committee !== undefined && user.is_committee === '1');

                // Check for duplicates
                user.duplicate = false;
                users.forEach(function(user_) {
                    if (user_.first_name == user.first_name &&
                        user_.last_name == user.last_name &&
                        user_.structure_level == user.structure_level) {
                        if (user.duplicate) {
                            // there are multiple duplicates!
                            user.duplicate_info += '\n' + gettextCatalog.getString('There are more than one duplicates of this user!');
                        } else {
                            user.duplicate = true;
                            user.duplicateAction = $scope.duplicateActions[1];
                            user.duplicate_info = '';
                            if (user_.title)
                                user.duplicate_info += user_.title + ' ';
                            if (user_.first_name)
                                user.duplicate_info += user_.first_name;
                            if (user_.first_name && user_.last_name)
                                user.duplicate_info += ' ';
                            if (user_.last_name)
                                user.duplicate_info += user_.last_name;
                            user.duplicate_info += ' (';
                            if (user_.number)
                                user.duplicate_info += gettextCatalog.getString('Number') + ': ' + user_.number + ', ';
                            if (user_.structure_level)
                                user.duplicate_info += gettextCatalog.getString('Structure level') + ': ' + user_.structure_level + ', ';
                            user.duplicate_info += gettextCatalog.getString('Username') + ': ' + user_.username + ') '+
                                gettextCatalog.getString('already exists.');

                            $scope.duplicates++;
                        }
                    }
                });
                $scope.users.push(user);
            });
            $scope.calcStats();
        };

        // Stats
        $scope.calcStats = function() {
            // not imported: if importerror or duplicate->keep original
            $scope.usersWillNotBeImported = 0;
            // imported: all others
            $scope.usersWillBeImported = 0;

            $scope.users.forEach(function(user) {
                if (!user.selected || user.importerror || (user.duplicate && user.duplicateAction == $scope.duplicateActions[0])) {
                    $scope.usersWillNotBeImported++;
                } else {
                    $scope.usersWillBeImported++;
                }
            });
        };

        $scope.setGlobalAction = function (action) {
            $scope.users.forEach(function (user) {
                if (user.duplicate)
                    user.duplicateAction = action;
            });
            $scope.calcStats();
        };

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;

            // collect all needed groups and create non existing groups
            var groupsToCreate = [];
            _.forEach($scope.users, function (user) {
                if (user.selected && !user.importerror && user.groups.length) {
                    _.forEach(user.groupsToCreate, function (group) { // Just append groups, that are not listed yet.
                        if (_.indexOf(groupsToCreate, group) == -1) {
                            groupsToCreate.push(group);
                        }
                    });
                }
            });
            var createPromises = [];
            $scope.groupsCreated = 0;
            _.forEach(groupsToCreate, function (groupname) {
                var group = {
                    name: groupname,
                    permissions: []
                };
                createPromises.push(Group.create(group).then( function (success) {
                    $scope.groupsCreated++;
                }));
            });

            $q.all(createPromises).then(function () {
                // reload allGroups, now all new groups are created
                var allGroups = Group.getAll();
                var existingUsers = User.getAll();

                _.forEach($scope.users, function (user) {
                    if (user.selected && !user.importerror) {
                        // Assign all groups
                        _.forEach(user.groups, function(csvGroup) {
                            allGroups.forEach(function (allGroup) {
                                // check with and without translation
                                if (csvGroup === allGroup.name ||
                                    csvGroup === gettextCatalog.getString(allGroup.name)) {
                                    user.groups_id.push(allGroup.id);
                                }
                            });
                        });

                        // Do nothing on duplicateAction==duplicateActions[0] (keep original)
                        if (user.duplicate && (user.duplicateAction == $scope.duplicateActions[1])) {
                            // delete existing user
                            var deletePromises = [];
                            existingUsers.forEach(function(user_) {
                                if (user_.first_name == user.first_name &&
                                    user_.last_name == user.last_name &&
                                    user_.structure_level == user.structure_level) {
                                    deletePromises.push(User.destroy(user_.id));
                                }
                            });
                            $q.all(deletePromises).then(function() {
                                User.create(user).then(
                                    function(success) {
                                        user.imported = true;
                                    }
                                );
                            });
                        } else if (!user.duplicate ||
                                   (user.duplicateAction == $scope.duplicateActions[2])) {
                            // create user
                            User.create(user).then(
                                function(success) {
                                    user.imported = true;
                                }
                            );
                        }
                    }
                });
                $scope.csvimported = true;
            });
        };
        $scope.clear = function () {
            $scope.users = null;
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            UserCsvExport.downloadExample();
        };
    }
])

.controller('GroupListCtrl', [
    '$scope',
    '$http',
    '$filter',
    'operator',
    'Group',
    'permissions',
    'gettext',
    'Agenda',
    'Assignment',
    'Mediafile',
    'Motion',
    'User',
    'ngDialog',
    'OpenSlidesPlugins',
    function($scope, $http, $filter, operator, Group, permissions, gettext, Agenda,
        Assignment, Mediafile, Motion, User, ngDialog, OpenSlidesPlugins) {
        $scope.permissions = permissions;

        $scope.$watch(function() {
            return Group.lastModified();
        }, function() {
            $scope.groups = $filter('orderBy')(Group.getAll(), 'id');

            // find all groups with the 2 dangerous permissions
            var groups_danger = [];
            $scope.groups.forEach(function (group) {
                if ((_.indexOf(group.permissions, 'users.can_see_name') > -1) &&
                    (_.indexOf(group.permissions, 'users.can_manage') > -1)){
                    if (operator.isInGroup(group)){
                        groups_danger.push(group);
                    }
                }
            });
            // if there is only one dangerous group, block it.
            $scope.group_danger = groups_danger.length == 1 ? groups_danger[0] : null;
        });

        // Dict to map plugin name -> display_name
        var pluginTranslation = {};
        _.forEach(OpenSlidesPlugins.getAll(), function (plugin) {
            pluginTranslation[plugin.name] = plugin.display_name;
        });
        $scope.apps = [];
        // Create the main clustering with appname->permissions
        angular.forEach(permissions, function(perm) {
            var permissionApp = perm.value.split('.')[0]; // get appname

            // To insert perm in the right spot in $scope.apps
            var insert = function (id, perm, verboseName) {
                if (!$scope.apps[id]) {
                    $scope.apps[id] = {
                        app_name: verboseName,
                        app_visible: true,
                        permissions: []
                    };
                }
                $scope.apps[id].permissions.push(perm);
            };

            switch(permissionApp) {
                case 'core': // id 0 (projector) and id 6 (general)
                    if (perm.value.indexOf('projector') > -1) {
                        insert(0, perm, gettext('Projector'));
                    } else {
                        insert(6, perm, gettext('General'));
                    }
                    break;
                case 'agenda': // id 1
                    insert(1, perm, Agenda.verboseName);
                    break;
                case 'motions': // id 2
                    insert(2, perm, Motion.verboseNamePlural);
                    break;
                case 'assignments': // id 3
                    insert(3, perm, Assignment.verboseNamePlural);
                    break;
                case 'mediafiles': // id 4
                    insert(4, perm, Mediafile.verboseNamePlural);
                    break;
                case 'users': // id 5
                    insert(5, perm, User.verboseNamePlural);
                    break;
                default: // plugins: id>5
                    var display_name = pluginTranslation[permissionApp] || permissionApp.charAt(0).toUpperCase() +
                        permissionApp.slice(1);
                    // does the app exists?
                    var result = -1;
                    angular.forEach($scope.apps, function (app, index) {
                        if (app.app_name === display_name)
                            result = index;
                    });
                    var id = result == -1 ? $scope.apps.length : result;
                    insert(id, perm, display_name);
                    break;
            }
        });

        // sort each app: first all permission with 'see', then 'manage', then the rest
        // save the permissions in different lists an concat them in the right order together
        // Special Users: the two "see"-permissions are normally swapped. To create the right
        // order, we could simply reverse the whole permissions.
        angular.forEach($scope.apps, function (app, index) {
            if(index == 5) { // users
                app.permissions.reverse();
            } else { // rest
                var see = [];
                var manage = [];
                var others = [];
                angular.forEach(app.permissions, function (perm) {
                    if (perm.value.indexOf('see') > -1) {
                        see.push(perm);
                    } else if (perm.value.indexOf('manage') > -1) {
                        manage.push(perm);
                    } else {
                        others.push(perm);
                    }
                });
                app.permissions = see.concat(manage.concat(others));
            }
        });

        // check if the given group has the given permission
        $scope.hasPerm = function (group, permission) {
            return _.indexOf(group.permissions, permission.value) > -1;
        };

        // The current user is not allowed to lock himself out of the configuration:
        // - if the permission is 'users.can_manage' or 'users.can_see'
        // - if the user is in only one group with these permissions (group_danger is set)
        $scope.danger = function (group, permission){
            if ($scope.group_danger){
                if (permission.value == 'users.can_see_name' ||
                    permission.value == 'users.can_manage'){
                    return $scope.group_danger == group;
                }
            }
            return false;
        };

        // delete selected group
        $scope.delete = function (group) {
            Group.destroy(group.id);
        };

        // save changed permission
        $scope.changePermission = function (group, perm) {
            if (!$scope.danger(group, perm)) {
                if (!$scope.hasPerm(group, perm)) { // activate perm
                    group.permissions.push(perm.value);
                } else {
                    // delete perm in group.permissions
                    group.permissions = _.filter(group.permissions, function(value) {
                        return value != perm.value; // remove perm
                    });
                }
                Group.save(group);
            }
        };

        $scope.openDialog = function (group) {
            ngDialog.open({
                template: 'static/templates/users/group-edit.html',
                controller: group ? 'GroupRenameCtrl' : 'GroupCreateCtrl',
                className: 'ngdialog-theme-default wide-form',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    group: function () {return group;},
                }
            });
        };
    }
])

.controller('GroupRenameCtrl', [
    '$scope',
    'Group',
    'group',
    'ErrorMessage',
    function($scope, Group, group, ErrorMessage) {
        $scope.group = group;
        $scope.new_name = group.name;

        $scope.alert = {};
        $scope.save = function() {
            var old_name = $scope.group.name;
            $scope.group.name = $scope.new_name;
            Group.save($scope.group).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                   $scope.alert = ErrorMessage.forAlert(error);
                   $scope.group.name = old_name;
                }
            );
        };
    }
])

.controller('GroupCreateCtrl', [
    '$scope',
    'Group',
    'ErrorMessage',
    function($scope, Group, ErrorMessage) {
        $scope.new_name = '';
        $scope.alert = {};

        $scope.save = function() {
            var group = {
                name: $scope.new_name,
                permissions: []
            };

            Group.create(group).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                   $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('userMenu', [
    '$scope',
    '$http',
    'DS',
    'User',
    'operator',
    'ngDialog',
    'UserProfileForm',
    'UserPasswordForm',
    function($scope, $http, DS, User, operator, ngDialog, UserProfileForm, UserPasswordForm) {
        $scope.logout = function () {
            $http.post('/users/logout/').then(function (response) {
                operator.setUser(null);
                window.location.reload();
            });
        };
        $scope.editProfile = function () {
            ngDialog.open(UserProfileForm.getDialog());
        };
        $scope.changePassword = function () {
            ngDialog.open(UserPasswordForm.getDialog());
        };
    }
])

.controller('LoginFormCtrl', [
    '$rootScope',
    '$scope',
    '$http',
    '$state',
    '$stateParams',
    '$q',
    'operator',
    'gettext',
    'autoupdate',
    'mainMenu',
    'DS',
    'ngDialog',
    function ($rootScope, $scope, $http, $state, $stateParams, $q, operator, gettext,
        autoupdate, mainMenu, DS, ngDialog) {
        $scope.alerts = [];

        if ($stateParams.msg) {
            $scope.alerts.push({
                type: 'danger',
                msg: $stateParams.msg,
            });
        }

        // check if guest login is allowed
        $scope.guestAllowed = $rootScope.guest_enabled;

        // get login info-text from server
        $http.get('/users/login/').then(function (success) {
            if(success.data.info_text) {
                $scope.alerts.push({
                    type: 'success',
                    msg: success.data.info_text
                });
            }
        });
        // check if cookies are enabled
        if (!navigator.cookieEnabled) {
            $scope.alerts.push({
                type: 'danger',
                msg: gettext('You have to enable cookies to use OpenSlides.'),
            });
        }

        // close alert function
        $scope.closeAlert = function(index) {
            $scope.alerts.splice(index, 1);
        };
        // login
        $scope.login = function () {
            $scope.closeThisDialog();
            $scope.alerts = [];
            var data = { 'username': $scope.username, 'password': $scope.password };
            if (!navigator.cookieEnabled) {
                data.cookies = false;
            }
            $http.post('/users/login/', data).then(
                function (response) {
                    // Success: User logged in.
                    // Clear store and reset deferred first message, if guests was enabled before.
                    DS.clear();
                    autoupdate.firstMessageDeferred = $q.defer();
                    // The next lines are partly the same lines as in core/start.js
                    autoupdate.newConnect();
                    autoupdate.firstMessageDeferred.promise.then(function () {
                        operator.setUser(response.data.user_id, response.data.user);
                        $rootScope.operator = operator;
                        mainMenu.updateMainMenu();
                        $state.go('home');
                        $rootScope.openslidesBootstrapDone = true;
                    });
                },
                function (error) {
                    // Error: Username or password is not correct.
                    $state.transitionTo($state.current, {msg: error.data.detail}, { 
                          reload: true, inherit: false, notify: true
                    });
                }
            );
        };
        // guest login
        $scope.guestLogin = function () {
            $scope.closeThisDialog();
            $state.go('home');
        };
    }
])

// Mark all users strings for translation in JavaScript.
.config([
    'gettext',
    function (gettext) {
        // permission strings (see models.py of each Django app)
        // agenda
        gettext('Can see agenda');
        gettext('Can manage agenda');
        gettext('Can see hidden items and time scheduling of agenda');
        gettext('Can put oneself on the list of speakers');
        // assignments
        gettext('Can see elections');
        gettext('Can nominate another participant');
        gettext('Can nominate oneself');
        gettext('Can manage elections');
        // core
        gettext('Can see the projector');
        gettext('Can manage the projector');
        gettext('Can see the front page');
        gettext('Can manage tags');
        gettext('Can manage configuration');
        gettext('Can use the chat');
        gettext('Can manage the chat');
        // mediafiles
        gettext('Can see the list of files');
        gettext('Can upload files');
        gettext('Can manage files');
        gettext('Can see hidden files');
        // motions
        gettext('Can see motions');
        gettext('Can create motions');
        gettext('Can support motions');
        gettext('Can manage motions');
        gettext('Can see and manage comments');
        // users
        gettext('Can see names of users');
        gettext('Can see extra data of users (e.g. present and comment)');
        gettext('Can manage users');

        // config strings in users/config_variables.py
        gettext('General');
        gettext('Sort name of participants by');
        gettext('Participants');
        gettext('Given name');
        gettext('Surname');
        gettext('PDF');
        gettext('Welcome to OpenSlides');
        gettext('Title for access data and welcome PDF');
        gettext('[Place for your welcome and help text.]');
        gettext('Help text for access data and welcome PDF');
        gettext('System URL');
        gettext('Used for QRCode in PDF of access data.');
        gettext('WLAN name (SSID)');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WLAN password');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WLAN encryption');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WEP');
        gettext('WPA/WPA2');
        gettext('No encryption');
    }
]);


// this is code from angular.js. Find a way to call this function from this file
function getBlockNodes(nodes) {
  // TODO(perf): just check if all items in `nodes` are siblings and if they are return the original
  //             collection, otherwise update the original collection.
  var node = nodes[0];
  var endNode = nodes[nodes.length - 1];
  var blockNodes = [node];

  do {
    node = node.nextSibling;
    if (!node) break;
    blockNodes.push(node);
  } while (node !== endNode);

  return $(blockNodes);
}

}());

//# sourceMappingURL=data:application/json;base64,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